/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
import FavoriteButton from './favorites/FavoriteButton'
import PropTypes from 'prop-types'
import React from 'react'
import SidebarMenu from './SidebarMenu'
import styles from './style.css'
import {
  addHeading,
  convertLinks,
  determineMenuWidthClass
} from './utils.js'
import { get, isEqual, map } from 'lodash'

export default class LinkGroupMenu extends React.Component {
  static propTypes = {
    id: PropTypes.string,
    group: PropTypes.object.isRequired,
    backdoorId: PropTypes.string,
    checkedLinkFilters: PropTypes.array,
    removeFavorite: PropTypes.func,
    addFavorite: PropTypes.func,
    expanded: PropTypes.bool,
    header: PropTypes.node
  }

  shouldComponentUpdate (nextProps) {
    const buildCompare = props => ({
      label: props.group.label,
      activities: {
        labels: map(props.group.links.activities, 'label'),
        favorites: map(props.group.links.activities, 'favorite')
      },
      reference: {
        labels: map(props.group.links.reference, 'label'),
        favorites: map(props.group.links.reference, 'favorite')
      },
      administration: {
        labels: map(props.group.links.administration, 'label'),
        favorites: map(props.group.links.administration, 'favorite')
      }
    })
    return !(
      isEqual(buildCompare(this.props), buildCompare(nextProps)) &&
      isEqual(this.props.expanded, nextProps.expanded) &&
      isEqual(this.props.checkedLinkFilters, nextProps.checkedLinkFilters)
    )
  }

  buildLinksForType (type) {
    const {
      group,
      backdoorId,
      checkedLinkFilters = [],
      addFavorite,
      removeFavorite
    } = this.props

    const FavoriteButtonWrapper = (link, linkComponent, key, groupName) => (
      <FavoriteButton
        key={key}
        link={link}
        linkComponent={linkComponent}
        group={groupName}
        addFavorite={addFavorite}
        removeFavorite={removeFavorite}
      />
    )
    const links = get(group, ['links', type])
    const hasCheckedLinkFilters = checkedLinkFilters.indexOf(type) !== -1
    if (links && hasCheckedLinkFilters) {
      return convertLinks(links, type, backdoorId, FavoriteButtonWrapper, group)
    }
    return []
  }

  render () {
    const {
      id,
      group,
      emptyMessage,
      expanded,
      handleClick,
      header
    } = this.props
    const { label } = group

    const activitiesLinks = this.buildLinksForType('activities')
    const referenceLinks = this.buildLinksForType('reference')
    const administrationLinks = this.buildLinksForType('administration')

    let links = addHeading(activitiesLinks, 'Activities')
    links = links.concat(addHeading(referenceLinks, 'Reference'))
    links = links.concat(addHeading(administrationLinks, 'Administration'))

    let headingCount =
      links.length -
      activitiesLinks.length +
      referenceLinks.length +
      administrationLinks.length
    if (headingCount > 0) {
      headingCount--
    }

    const widthClass = determineMenuWidthClass(links.length, headingCount)

    return (
      <SidebarMenu
        id={id}
        header={header}
        className={widthClass}
        label={label}
        onCloseClick={handleClick}
        expanded={expanded}
      >
        {links.length > 0 ? (
          <div className={styles.linksContainer}>{links}</div>
        ) : (
          emptyMessage || 'No links found.'
        )}
      </SidebarMenu>
    )
  }
}
