/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/* eslint-disable import/first */
jest.mock('../../../utils/utils')
import KfsUtils from '../../../utils/utils'

import React from 'react'
import { setupContainer } from '../../../../test-utils/test-utils'
import SidebarError from '../SidebarError'
import SidebarWaiting from '../SidebarWaiting'
import styles from '../style.css'
import { mount, shallow } from 'enzyme'

describe('sidebar', () => {
  let Sidebar
  let wrapper

  beforeAll(() => {
    setupContainer('sidebar')
  })

  beforeEach(async () => {
    global.stayOnPage = jest.fn()
    global.matchMedia = jest.fn()
    KfsUtils.apiCall.mockResolvedValue({ data: {} })
    Sidebar = require('../sidebar')
    Sidebar = Sidebar.default
    wrapper = await shallow(<Sidebar />)
  })

  afterAll(() => {
    jest.unmock('../../../utils/utils')
  })

  it('should match the snapshot', () => {
    expect(wrapper).toMatchSnapshot()
  })

  it('should collapse', () => {
    wrapper.setState({ sidebarOut: true, loadingData: false, loadError: false })
    const menuToggleSelector = '#menu-toggle'
    wrapper.find(menuToggleSelector).simulate('click')

    expect(wrapper.find(menuToggleSelector).hasClass('rotated')).toBe(true)
    expect(wrapper.state('sidebarOut')).toBe(false)
  })

  describe('SidebarWaiting', () => {
    it('should render when out and loading', () => {
      wrapper.setState({ sidebarOut: true, loadingData: true })
      expect(wrapper.find(SidebarWaiting)).toHaveLength(1)
    })

    it('should not render if sidebar is not out', () => {
      wrapper.setState({ sidebarOut: false, loadingData: true })
      expect(wrapper.find(SidebarWaiting)).toHaveLength(0)
    })

    it('should not render if not loading data', () => {
      wrapper.setState({ sidebarOut: true, loadingData: false })
      expect(wrapper.find(SidebarWaiting)).toHaveLength(0)
    })
  })

  describe('SidebarError', () => {
    it('should render when out and error is present', () => {
      wrapper.setState({ sidebarOut: true, loadError: true })
      expect(wrapper.find(SidebarError)).toHaveLength(1)
    })

    it('should not render if sidebar is not out', () => {
      wrapper.setState({ sidebarOut: false, loadError: true })
      expect(wrapper.find(SidebarError)).toHaveLength(0)
    })

    it('should not render if no error', () => {
      wrapper.setState({ sidebarOut: true, loadError: false })
      expect(wrapper.find(SidebarError)).toHaveLength(0)
    })
  })

  describe('Search', () => {
    const searchSelector = `li.${styles.search} input`
    beforeEach(async () => {
      wrapper = await mount(<Sidebar />)
      const institutionPreferences = {
        linkGroups: [
          {
            label: 'Accounts Receivables',
            links: {
              activities: [{ label: 'Cash Control', navLinkId: '1' }],
              administration: [{ label: 'Customer Address', navLinkId: '2' }],
              reference: [{ label: 'Balances', navLinkId: '3' }]
            }
          }
        ],
        sidebarOut: true,
        loadingData: false
      }
      wrapper.setState({ institutionPreferences })
    })

    it('refreshes results when query has more than 2 characters', () => {
      const searchInput = wrapper.find(searchSelector)

      searchInput.simulate('change', { target: { value: 'ca' } })
      expect(wrapper.state('searchResults')).toBeUndefined()

      searchInput.simulate('change', { target: { value: 'cas' } })
      expect(
        wrapper.find(`.${styles.search} .${styles.favoriteContainer} a`).text()
      ).toEqual('Cash Control')
    })

    describe('Accessibility', () => {
      it('exposes search count for screen readers', () => {
        const searchInput = wrapper.find(searchSelector)
        searchInput.simulate('change', { target: { value: 'cas' } })
        expect(wrapper.find('#aria-search-results-text').text()).toEqual(
          'There are 1 search results for cas.'
        )
      })
    })
  })
})
