/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/* eslint-disable import/first */
jest.mock('../../search/utils')

import FavoritesLinkConfig from '../FavoritesLinkConfig'
import React from 'react'
import { searchLinkGroups } from '../../search/utils'
import { mount, shallow } from 'enzyme'

describe('Favorites Link Config', () => {
  let props, state
  beforeEach(() => {
    searchLinkGroups.mockReturnValue({
      'Accounts Receivables': [{ label: 'Cash Control' }]
    })

    props = {
      id: 'id',
      handleClick: jest.fn(),
      addFavorite: jest.fn(),
      removeFavorite: jest.fn(),
      header: 'header',
      label: 'label',
      expanded: false,
      checkedLinkFilters: ['activities', 'administration', 'reference'],
      linkGroups: [
        {
          label: 'Accounts Receivables',
          links: {
            activities: [{ label: 'Cash Control', navLinkId: 0 }],
            administration: [{ label: 'Customer Address', navLinkId: 1 }],
            reference: [
              { label: 'Balances', navLinkId: 2 },
              { label: 'Customer', navLinkId: 3, favorite: true }
            ]
          }
        }
      ],
      layoutActivePanel: jest.fn()
    }

    state = {
      expandedLabel: 'Accounts Receivables'
    }
  })

  afterAll(() => {
    jest.unmock('../../search/utils')
  })

  it('should render shallow without issue', () => {
    const wrapper = shallow(<FavoritesLinkConfig {...props} />)
    wrapper.setState(state)
    expect(wrapper).toMatchSnapshot()
  })

  it('toggling favorite adds favorite for non-favorite', () => {
    const wrapper = mount(<FavoritesLinkConfig {...props} />)
    wrapper.setState(state)
    wrapper.find('#selectable-link-Balances').simulate('change')
    expect(props.addFavorite).toHaveBeenCalledWith(2)
  })

  it('toggling favorite removes favorite for current favorite', () => {
    const wrapper = mount(<FavoritesLinkConfig {...props} />)
    wrapper.setState(state)
    wrapper.find('#selectable-link-Customer').simulate('change')
    expect(props.removeFavorite).toHaveBeenCalledWith(3)
  })

  it('filters links based on the checked link filters', () => {
    props.checkedLinkFilters = ['activities']
    const wrapper = mount(<FavoritesLinkConfig {...props} />)
    wrapper.setState(state)
    expect(wrapper.find('#selectable-link-Customer').length).toBe(0)
    expect(wrapper.find('#selectable-link-Balances').length).toBe(0)
    expect(wrapper.find('#selectable-link-CustomerAddress').length).toBe(0)
    expect(wrapper.find('#selectable-link-CashControl').length).toBe(1)
  })

  it('updating component with new linkGroups should call updateSearchResults and clear searchResults', () => {
    props.linkGroups.push({
      label: 'Test',
      links: {}
    })
    const wrapper = mount(<FavoritesLinkConfig {...props} />)
    wrapper.setState({
      ...state,
      searchResults: { 'Accounts Receivables': [{ label: 'Cash Control' }] }
    })

    let searchResults = wrapper.find('SearchResults')
    expect(searchResults).toHaveLength(1)
    expect(searchResults.find('div > h4').text()).toEqual(
      'Accounts Receivables'
    )
    const selectableLink = searchResults.find('SelectableLink')
    expect(selectableLink).toHaveLength(1)
    expect(selectableLink.find('label').text()).toEqual('Cash Control')

    props.linkGroups = [
      {
        label: 'Accounts Receivables',
        links: {
          activities: [{ label: 'Cash Control', navLinkId: 0 }],
          administration: [{ label: 'Customer Address', navLinkId: 1 }],
          reference: [
            { label: 'Balances', navLinkId: 2 },
            { label: 'Customer', navLinkId: 3, favorite: true }
          ]
        }
      }
    ]
    wrapper.setProps(props)
    searchResults = wrapper.find('SearchResults')
    expect(searchResults).toHaveLength(0)
  })

  it('clicking expand label toggles expanded property of SelectableLinkGroup', () => {
    const wrapper = mount(<FavoritesLinkConfig {...props} />)
    wrapper.setState(state)
    expect(wrapper.find('SelectableLinkGroup').prop('expanded')).toBe(true)
    wrapper.find('.link-group-toggle-button').simulate('click')
    expect(wrapper.find('SelectableLinkGroup').prop('expanded')).toBe(false)
    wrapper.find('.link-group-toggle-button').simulate('click')
    expect(wrapper.find('SelectableLinkGroup').prop('expanded')).toBe(true)
  })

  describe('search', () => {
    const preSearchState = {
      searchTerm: 'searching',
      searchResults: {}
    }

    it('clicking the remove span should clear the search', () => {
      const wrapper = mount(<FavoritesLinkConfig {...props} />)
      wrapper.setState({ ...state, ...preSearchState })
      expect(wrapper.find('input[type="search"]').prop('value')).toEqual(
        'searching'
      )
      wrapper.find('span.remove').simulate('click')
      expect(wrapper.find('input[type="search"]').prop('value')).toEqual('')
    })

    it('changing the search input to empty string should call handleSearch but not search', () => {
      const wrapper = mount(<FavoritesLinkConfig {...props} />)
      wrapper.setState({ ...state, ...preSearchState })
      expect(wrapper.find('SearchResults')).toHaveLength(1)
      wrapper
        .find('input[type="search"]')
        .simulate('change', { target: { value: '' } })
      expect(wrapper.find('SearchResults')).toHaveLength(0)
    })

    it('changing the search input to account should call handleSearch and search and return nothing', () => {
      searchLinkGroups.mockReturnValue({})
      const wrapper = mount(<FavoritesLinkConfig {...props} />)
      wrapper.setState({ ...state, ...preSearchState })
      wrapper
        .find('input[type="search"]')
        .simulate('change', { target: { value: 'account' } })
      expect(wrapper.find('SelectableLink')).toHaveLength(0)
      const searchResults = wrapper.find('SearchResults')
      expect(searchResults).toHaveLength(1)
      expect(searchResults.find('div.center').text()).toEqual(
        'No results found'
      )
    })

    it('changing the search input to cash should call handleSearch and search and return something', () => {
      const wrapper = mount(<FavoritesLinkConfig {...props} />)
      wrapper.setState({ ...state, ...preSearchState })
      wrapper
        .find('input[type="search"]')
        .simulate('change', { target: { value: 'cash' } })
      const searchResults = wrapper.find('SearchResults')
      expect(searchResults).toHaveLength(1)
      expect(searchResults.find('div > h4').text()).toEqual(
        'Accounts Receivables'
      )
      const selectableLink = searchResults.find('SelectableLink')
      expect(selectableLink).toHaveLength(1)
      expect(selectableLink.find('label').text()).toEqual('Cash Control')
    })
  })
})
