/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import Immutable from 'immutable'
import SubLinkType from './SubLinkType'
import { withDragDropContext } from '../../utils/sortable'
import {
  animateScrollTo,
  getFirstElementBySelector
} from '../../utils/dom_utils.js'
import { isScrolledIntoView, validateForm } from '../institutionConfigUtils.js'
import React, { Component } from 'react'

export class SubLinkGroup extends Component {
  constructor (props) {
    super(props)
    this.state = {
      customLinkFormOpen: false,
      errors: [],
      errorMessages: [],
      newLink: Immutable.Map({ label: '', link: '', linkType: 'custom' }),
      newLinkType: 'activities'
    }
  }

  openAddCustomLink = () => {
    this.setState({
      customLinkFormOpen: !this.state.customLinkFormOpen,
      errors: [],
      errorMessages: [],
      newLink: Immutable.Map({ label: '', link: '', linkType: 'custom' }),
      newLinkType: 'activities',
      moveToGroupIndex: this.props.groupIndex,
      update: false
    })
  }

  addCustomLink = () => {
    let errorObj = validateForm(
      this.state.newLink.get('label'),
      this.state.newLink.get('link')
    )

    if (errorObj.errors.length < 1) {
      this.props.stateMaintenance.stateAddNewCustomLink(
        this.props.groupIndex,
        this.state.newLink,
        this.state.newLinkType
      )
      this.setState({
        customLinkFormOpen: false,
        errors: [],
        errorMessages: [],
        newLink: Immutable.Map({ label: '', link: '', linkType: 'custom' }),
        newLinkType: 'activities'
      })
    } else {
      this.setState(errorObj)
    }
  }

  stateOpenUpdateCustomLink = (link, type) => {
    this.setState({
      customLinkFormOpen: true,
      errors: [],
      errorMessages: [],
      newLink: link,
      newLinkType: type,
      oldLink: link,
      oldLinkType: type,
      moveToGroupIndex: this.props.groupIndex,
      update: true
    })
    if (
      !isScrolledIntoView(getFirstElementBySelector('.active .add-custom-link'))
    ) {
      animateScrollTo(100)
    }
  }

  stateUpdateCustomLink = () => {
    let errorObj
    if (this.state.newLink.get('linkType') === 'custom') {
      errorObj = validateForm(
        this.state.newLink.get('label'),
        this.state.newLink.get('link')
      )
    }

    if (!errorObj || errorObj.errors.length < 1) {
      this.props.stateMaintenance.stateUpdateExistingCustomLink(
        this.props.groupIndex,
        this.state.oldLink,
        this.state.oldLinkType,
        this.state.moveToGroupIndex,
        this.state.newLink,
        this.state.newLinkType
      )
      this.state = {
        customLinkFormOpen: false,
        errors: [],
        errorMessages: [],
        newLink: Immutable.Map({ label: '', link: '', linkType: 'custom' }),
        newLinkType: 'activities'
      }
    } else {
      this.setState(errorObj)
    }
  }

  stateDeleteCustomLink = (groupIndex, link, type) => {
    this.props.stateMaintenance.stateDeleteExistingCustomLink(
      groupIndex,
      type,
      link
    )
    this.state = {
      customLinkFormOpen: false,
      errors: [],
      errorMessages: [],
      newLink: Immutable.Map({
        label: '',
        link: '',
        linkType: 'custom',
        newTarget: false
      }),
      newLinkType: 'activities'
    }
  }

  updateNewLinkValue = (key, event) => {
    let value = event.target.value
    let updatedNewLink = this.state.newLink.set(key, value)
    this.setState({ newLink: updatedNewLink })
  }

  updateNewLinkType = event => {
    let type = event.target.value
    this.setState({ newLinkType: type })
  }

  updateNewLinkNewTarget = event => {
    let newTarget = event.target.checked
    let updatedNewLink = this.state.newLink.set('newTarget', newTarget)
    this.setState({ newLink: updatedNewLink })
  }

  updateLinkGroup = event => {
    let group = event.target.value
    this.setState({ moveToGroupIndex: group })
  }

  render () {
    let stateMaintenance = this.props.stateMaintenance
    stateMaintenance.stateOpenUpdateCustomLink = this.stateOpenUpdateCustomLink
    stateMaintenance['stateUpdateCustomLink'] = this.stateUpdateCustomLink
    stateMaintenance['stateDeleteCustomLink'] = this.stateDeleteCustomLink

    let divClassName = 'admin-sublinks'
    if (this.props.groupLabel !== this.props.expandedLinkGroup) {
      divClassName += ' collapse'
    } else {
      divClassName += ' active'
    }
    let formClass = this.state.customLinkFormOpen
      ? 'customLinkForm form'
      : 'customLinkForm form hidden'
    let labelClass = this.state.errors.indexOf('label') > -1 ? 'error' : ''
    let linkClass = this.state.errors.indexOf('link') > -1 ? 'error' : ''

    let groupSelectItems = this.props.linkGroupLabels.map(
      (linkGroupLabel, index) => {
        return (
          <option key={index} value={index}>
            {linkGroupLabel.toUpperCase()}
          </option>
        )
      }
    )

    let errorMessage
    if (this.state.errorMessages && this.state.errorMessages.length > 0) {
      let messages = this.state.errorMessages.map(function (message, index) {
        return <li key={index}>{message}</li>
      })
      errorMessage = <ul className='errorMessages'>{messages}</ul>
    }

    let editLinkName
    let editLinkURL
    let newTargetToggle
    let formActionFunction = this.stateUpdateCustomLink
    let formActionText = 'Update'
    if (this.state.newLink.get('linkType') === 'custom') {
      editLinkName = (
        <div>
          <div>
            <label>LINK NAME:</label>
          </div>
          <div>
            <input
              className={labelClass}
              type='text'
              value={this.state.newLink.get('label')}
              onChange={this.updateNewLinkValue.bind(null, 'label')}
            />
          </div>
        </div>
      )
      editLinkURL = (
        <div>
          <div>
            <label>URL:</label>
          </div>
          <div>
            <input
              className={linkClass}
              type='text'
              value={this.state.newLink.get('link')}
              onChange={this.updateNewLinkValue.bind(null, 'link')}
            />
          </div>
        </div>
      )

      if (!this.state.update) {
        formActionFunction = this.addCustomLink
        formActionText = 'Add'
      }
    } else if (this.state.newLink.get('linkType') === 'kfs') {
      newTargetToggle = (
        <div>
          <div>
            <input
              checked={this.state.newLink.get('newTarget')}
              type='checkbox'
              value='newTarget'
              onChange={this.updateNewLinkNewTarget}
            />
            <label>Open In New Window</label>
          </div>
        </div>
      )
    }

    return (
      <div id={this.props.id + '-menu'} className={divClassName}>
        <div className='add-custom-link'>
          <button className='btn btn-default' onClick={this.openAddCustomLink}>
            <span className='glyphicon glyphicon-plus' />Add Custom Link
          </button>
          <div className={formClass}>
            {errorMessage}
            {editLinkName}
            {editLinkURL}
            <div>
              <input
                checked={this.state.newLinkType === 'activities'}
                type='radio'
                value='activities'
                id={this.props.id + '-activities-radio'}
                onChange={this.updateNewLinkType}
              />
              <label htmlFor={this.props.id + '-activities-radio'}>
                Activities
              </label>
              <input
                checked={this.state.newLinkType === 'reference'}
                type='radio'
                value='reference'
                id={this.props.id + '-reference-radio'}
                onChange={this.updateNewLinkType}
              />
              <label htmlFor={this.props.id + '-reference-radio'}>
                Reference
              </label>
              <input
                checked={this.state.newLinkType === 'administration'}
                type='radio'
                value='administration'
                id={this.props.id + '-administration-radio'}
                onChange={this.updateNewLinkType}
              />
              <label htmlFor={this.props.id + '-administration-radio'}>
                Administration
              </label>
            </div>
            {newTargetToggle}
            <div>
              <label>GROUP:</label>
            </div>
            <div>
              <select
                value={this.state.moveToGroupIndex}
                onChange={this.updateLinkGroup}
              >
                {groupSelectItems}
              </select>
            </div>
            <div>
              <button className='btn btn-green' onClick={formActionFunction}>
                {formActionText}
              </button>
              <button
                className='btn btn-default'
                onClick={this.openAddCustomLink}
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
        {['activities', 'reference', 'administration'].map((type, index) => (
          <SubLinkType
            key={type}
            groupLabel={this.props.groupLabel}
            links={this.props.links}
            type={type}
            typeIndex={index}
            linkGroups={this.props.linkGroups}
            stateMaintenance={stateMaintenance}
          />
        ))}
      </div>
    )
  }
}

export default withDragDropContext(SubLinkGroup)
