/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import axios from 'axios'
import KfsUtils from '../utils'
import MockAdapter from 'axios-mock-adapter'

describe('KfsUtils', () => {
  describe('getKualiSessionId', () => {
    afterEach(() => {
      if (document.cookie) {
        delete document.cookie
      }
    })

    it('gets the Kuali session ID from cookies', () => {
      Object.defineProperty(document, 'cookie', {
        value: 'kualiSessionId=foo',
        configurable: true
      })
      expect(KfsUtils.getKualiSessionId()).toBe('foo')
    })

    it('returns null if no session ID', () => {
      expect(KfsUtils.getKualiSessionId()).toBe(null)
    })
  })

  describe('getUrlPathPrefix', () => {
    const originalUrl = global.location.href

    afterEach(() => {
      global.jsdom.reconfigure({
        url: originalUrl
      })
    })

    it('gets the URL path prefix', () => {
      global.jsdom.reconfigure({
        url: 'http://website.com/foo/path/to/somewhere'
      })
      expect(KfsUtils.getUrlPathPrefix()).toBe('/foo/')
    })

    it('returns empty string if no pathname', () => {
      global.jsdom.reconfigure({
        url: 'http://website.com'
      })
      expect(KfsUtils.getUrlPathPrefix()).toBe('')
    })

    it('returns empty string if no path prefix match found', () => {
      global.jsdom.reconfigure({
        url: 'http://website.com#stuff'
      })
      expect(KfsUtils.getUrlPathPrefix()).toBe('')
    })
  })

  describe('buildBackdoorIdAppender', () => {
    describe('Valid Backdoor Ids', () => {
      const backdoorId = '1337'
      const appender = KfsUtils.buildBackdoorIdAppender(backdoorId)

      it('returns a function that appends backdoor ID to a given URL', () => {
        const link = 'http://foo.com/stuff'
        const backdoorLink = appender(link)
        expect(backdoorLink).toBe('http://foo.com/stuff?backdoorId=1337')
      })

      it('returns a function that appends backdoor ID and preserves query params', () => {
        const link = 'http://foo.com/stuff?a=1&b=2'
        const backdoorLink = appender(link)
        expect(backdoorLink).toBe(
          'http://foo.com/stuff?a=1&b=2&backdoorId=1337'
        )
      })

      it('returns a function that appends backdoor ID and keeps URL query params at the end', () => {
        const link = 'http://foo.com/stuff?a=1&b=2&c=http://google.com'
        const backdoorLink = appender(link)
        expect(backdoorLink).toBe(
          'http://foo.com/stuff?a=1&b=2&backdoorId=1337&c=http://google.com'
        )
      })

      it('returns a function that appends backdoor ID and omits empty query values', () => {
        const link = 'http://foo.com/stuff?a=1&b=2&c='
        const backdoorLink = appender(link)
        expect(backdoorLink).toBe(
          'http://foo.com/stuff?a=1&b=2&c&backdoorId=1337'
        )
      })

      it('returns a function that appends backdoor ID without any query params prefixed', () => {
        const link = 'http://foo.com/stuff?a=http://google.com'
        const backdoorLink = appender(link)
        expect(backdoorLink).toBe(
          'http://foo.com/stuff?backdoorId=1337&a=http://google.com'
        )
      })

      it('should return null if no link is provided', () => {
        const backdoorLink = appender(null)
        expect(backdoorLink).toBeNull()
      })

      it('should return an empty string if the link is empty', () => {
        const backdoorLink = appender('')
        expect(backdoorLink).toEqual('')
      })
    })

    describe('Invalid backdoor ids', () => {
      it('returns an identity function if no backdoor id is provided', () => {
        const appender = KfsUtils.buildBackdoorIdAppender()
        const link = 'http://foo.com/stuff?a=1&b=2&c=http://google.com'
        const backdoorLink = appender(link)
        expect(backdoorLink).toEqual(link)
      })

      it('returns an identity function is an empty backdoor id is provided', () => {
        const appender = KfsUtils.buildBackdoorIdAppender('')
        const link = 'http://foo.com/stuff?a=1&b=2&c=http://google.com'
        const backdoorLink = appender(link)
        expect(backdoorLink).toEqual(link)
      })
    })
  })

  describe('buildKeyFromLabel', () => {
    it('creates a key from a label', () => {
      const key = KfsUtils.buildKeyFromLabel('Tom & Jerry')
      expect(key).toBe('tom-and-jerry')
    })
  })

  describe('getAuthHeader', () => {
    afterEach(() => {
      if (document.cookie) {
        delete document.cookie
      }
    })

    it('gets a value for a bearer header with the auth token in cookies', () => {
      const authToken = 'authTokensAreCool123'
      Object.defineProperty(document, 'cookie', {
        value: `financialsAuthToken=${authToken}`,
        configurable: true
      })
      const header = KfsUtils.getAuthHeader()
      expect(header).toBe(`Bearer ${authToken}`)
    })

    it('returns undefined if no auth token in cookies', () => {
      const header = KfsUtils.getAuthHeader()
      expect(header).toBe(undefined)
    })
  })

  describe('apiCall', () => {
    let axiosMock

    beforeEach(() => {
      axiosMock = new MockAdapter(axios)
    })

    it('makes an axios request', () => {
      axiosMock.onGet('foo').reply(200, 'congratulations!')
      KfsUtils.apiCall('foo', { stuff: 'hi' }).then(response => {
        expect(response.data).toBe('congratulations!')
      })
    })
  })
})
