/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import React from 'react'
import { render } from 'react-dom'

function getFirstElementBySelector (selector, parentNode = document) {
  const matchingEls = parentNode.querySelectorAll(selector)
  if (matchingEls.length > 0) {
    return matchingEls[0]
  }
  return null
}

function closest (el, selector) {
  if (!document.documentElement.contains(el)) {
    return null
  }
  while (el !== null && el.nodeType === 1) {
    if (el.matches(selector)) {
      return el
    }
    el = el.parentElement || el.parentNode
  }
  return null
}

function indexInParent (el) {
  const siblings = el.parentNode.childNodes
  for (let i = 0; i < siblings.length; i++) {
    if (siblings[i] === el) return i
  }
}

function addClassToElement (element, className) {
  if (element && className && !elementHasClass(element, className)) {
    element.className = element.className + ' ' + className
  }
}

function removeClassFromElement (element, className) {
  if (element && className) {
    element.className = element.className.replace(className, '')
  }
}

function setStyleRule (element, property, value) {
  if (element) {
    element.style[property] = value
  }
}

function getElementOffset (element) {
  const rect = element.getBoundingClientRect()
  const body = document.body

  return {
    top: rect.top + body.scrollTop,
    left: rect.left + body.scrollLeft
  }
}

function getElementOuterHeight (element, withMargins) {
  let height = element ? element.offsetHeight : 0
  if (element && withMargins) {
    const style = getComputedStyle(element)

    height += parseInt(style.marginTop) + parseInt(style.marginBottom)
    return height
  }

  return height
}

function animateScrollTo (y) {
  const direction = window.scrollY > y ? 'up' : 'down'
  const duration = 1000
  const distance = Math.abs(window.scrollY - y)
  const increment = distance / (duration / 15)
  const scrollInterval = setInterval(function () {
    if (
      (direction === 'down' && window.scrollY >= y) ||
      (direction === 'up' && window.scrollY <= y)
    ) {
      clearInterval(scrollInterval)
    } else {
      const scrollAmt = direction === 'down' ? increment : -increment
      window.scrollBy(0, scrollAmt)
    }
  }, 15)
}

function getElementHeight (element) {
  return element && getComputedStyle(element).height
}

function elementHasClass (element, className) {
  return element && element.classList.contains(className)
}

function toggleClass (element, className) {
  return element && element.classList.toggle(className)
}

function replaceWindowLocation (url) {
  window.location.replace(url)
}

function renderComponentIntoElement (Component, selector) {
  render(<Component />, document.getElementById(selector))
}

module.exports = {
  addClassToElement,
  animateScrollTo,
  closest,
  elementHasClass,
  getFirstElementBySelector,
  getElementHeight,
  getElementOffset,
  getElementOuterHeight,
  indexInParent,
  removeClassFromElement,
  renderComponentIntoElement,
  replaceWindowLocation,
  setStyleRule,
  toggleClass
}
