/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import { apiCall } from '../utils'
import { merge } from 'lodash'
import shortid from 'shortid'
import styles from './style.css'
import { URLS } from '../constants'

export const RANGE_LOWER_BOUND_KEY_PREFIX = 'rangeLowerBoundKeyPrefix'

export async function formTemplate (entityName) {
  const formDefinition = await fetchFormDefinition(entityName)
  if (formDefinition) {
    return buildFormTemplate(entityName, formDefinition)
  }
}

export function lookupURL (entityName) {
  return `${URLS.API.SYS.LOOKUP}/${entityName}`
}

async function fetchFormDefinition (entityName) {
  try {
    const response = await apiCall(lookupURL(entityName))
    return response.data
  } catch (e) {
    return null
  }
}

async function fetchFormValues (entityName) {
  try {
    const response = await apiCall(`${lookupURL(entityName)}/values`)
    return response.data
  } catch (e) {
    return {}
  }
}

async function buildFormTemplate (entityName, attributes) {
  let gadgetValues
  let gadgetDefinitions = []
  for (let i = 0; i < attributes.length; i++) {
    const attribute = attributes[i]
    const gadgetBuilder = getGadgetBuilder(attribute)
    let { gadgets, buildDetailsFromValues } = gadgetBuilder

    if (buildDetailsFromValues) {
      if (!gadgetValues) {
        gadgetValues = await fetchFormValues(entityName)
      }

      const details = buildDetailsFromValues(gadgetValues[attribute.name])
      gadgets = gadgets.map(gadget => merge(gadget, { details }))
    }
    gadgetDefinitions = gadgetDefinitions.concat(gadgets)
  }

  return {
    type: 'Section',
    id: 'ROOT',
    children: gadgetDefinitions
  }
}

function baseGadgetDefinition (type, attribute, overrides) {
  return merge(
    {
      type,
      formKey: attribute.name,
      label: attribute.label,
      id: shortid.generate(),
      details: {}
    },
    overrides
  )
}

// some of this is based on createAndPopulateFieldsForLookup
function getGadgetBuilder (attribute) {
  const { control } = attribute
  if (control.datePicker) {
    if (control.ranged) {
      return {
        gadgets: [
          baseGadgetDefinition('Date', attribute, {
            formKey: `${RANGE_LOWER_BOUND_KEY_PREFIX}_${attribute.name}`,
            label: `${attribute.label} From`,
            fieldSize: 'medium'
          }),
          baseGadgetDefinition('Date', attribute, {
            label: `${attribute.label} To`,
            fieldSize: 'medium'
          })
        ]
      }
    } else {
      return {
        gadgets: [baseGadgetDefinition('Date', attribute)]
      }
    }
  } else if (control.multiselect && control.hierarchical) {
    return {
      gadgets: [baseGadgetDefinition('Tree', attribute)],
      buildDetailsFromValues: values => ({
        valueKey: 'value',
        nodes: values,
        gadgetClassName: styles.tree
      })
    }
  } else {
    return {
      gadgets: [baseGadgetDefinition('Text', attribute)]
    }
  }
}
