/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import React, { Component } from 'react'
import { findDOMNode } from 'react-dom'
import { map, throttle, get, isFunction, isEmpty } from 'lodash'
import shortid from 'shortid'
import { DragSource, DropTarget } from 'react-dnd'
import styles from './builder.css'

import _RenderSection from '../components/section-descriptor'
import _RenderGadget from '../components/gadget-descriptor'
import _AddItem from '../components/add-item'

const boxSource = {
  beginDrag (props) {
    return {
      gadget: props.section || props.gadget,
      onMove: props.onMove
    }
  },

  isDragging (props, monitor) {
    if (props.isDragging) return true
    const gadget = props.section || props.gadget
    if (gadget === monitor.getItem().gadget) return true
    return false
  },

  endDrag (props, monitor) {
    if (!monitor.didDrop()) return
    const item = monitor.getItem()
    const dropResult = monitor.getDropResult()
    if (dropResult) {
      if (item.gadget.id === get(dropResult, 'target.id')) return
      item.onMove(
        item.gadget,
        dropResult.target,
        dropResult.direction,
        dropResult.targetParent
      )
    }
  }
}

const boxTarget = {
  canDrop (props, monitor, component) {
    if (props.isDragging) return false
    return true
  },

  drop (props, monitor, component) {
    if (monitor.getDropResult()) return
    const { y } = monitor.getClientOffset()
    const { top, height } = findDOMNode(component).getBoundingClientRect()

    const direction = y < top + height / 2 ? 0 : 1
    return {
      target: props.section || props.gadget,
      targetParent: props.parentId,
      direction
    }
  },

  hover (props, monitor, component) {
    if (!monitor.canDrop()) {
      return
    }

    const rawComponent = undecorate(component)
    if (!rawComponent.setHovering) return
    const { y } = monitor.getClientOffset()
    const { top, height } = findDOMNode(component).getBoundingClientRect()

    if (y < top + height / 2) {
      rawComponent.setHovering('above')
    } else {
      rawComponent.setHovering('below')
    }
  }
}

function undecorate (component) {
  let curr = component
  while (typeof curr.getDecoratedComponentInstance === 'function') {
    curr = curr.getDecoratedComponentInstance()
  }
  return curr
}

const DropZone = props => {
  return <div className={styles.dropzone} />
}

const getDragNDropInterface = Component => {
  return class _DragNDropInterface extends React.Component {
    setHovering = throttle(position => {
      this.setState({ hovering: position })
    }, 100)

    state = {
      hovering: null
    }

    render () {
      const props = this.props
      const { hovering } = this.state

      return props.connectDragSource(
        props.connectDropTarget(
          <div>
            {!props.isDragging &&
              props.isOver &&
              hovering === 'above' &&
              props.connectDropTarget(
                <div>
                  <DropZone />
                </div>
              )}

            <Component {...props} />
            {!props.isDragging &&
              props.isOver &&
              hovering === 'below' &&
              props.connectDropTarget(
                <div>
                  <DropZone />
                </div>
              )}
          </div>
        )
      )
    }
  }
}

const RenderSection = DropTarget('Gadget', boxTarget, (connect, monitor) => ({
  connectDropTarget: connect.dropTarget(),
  isOver: monitor.isOver({ shallow: true })
}))(
  DragSource('Gadget', boxSource, (connect, monitor) => {
    return {
      connectDragSource: connect.dragSource(),
      isDragging: monitor.isDragging()
    }
  })(getDragNDropInterface(_RenderSection))
)

const RenderGadget = DropTarget('Gadget', boxTarget, (connect, monitor) => ({
  connectDropTarget: connect.dropTarget(),
  isOver: monitor.isOver({ shallow: true })
}))(
  DragSource('Gadget', boxSource, (connect, monitor) => {
    return {
      connectDragSource: connect.dragSource(),
      isDragging: monitor.isDragging()
    }
  })(getDragNDropInterface(_RenderGadget))
)

const SectionDropzone = DropTarget('Gadget', boxTarget, (connect, monitor) => ({
  connectDropTarget: connect.dropTarget(),
  isOver: monitor.isOver({ shallow: true })
}))(
  class SectionDropzone extends React.Component {
    render () {
      const _DropZone = !this.props.isDragging &&
        this.props.isOver && <DropZone />
      return this.props.connectDropTarget(
        <div style={{ width: '100%' }}>{this.props.children(_DropZone)}</div>
      )
    }
  }
)

export default class Builder extends Component {
  onAddGadgetToEmptyTemplate = gadget => {
    const initialTemplate = isFunction(get(gadget, 'meta.initialTemplate'))
      ? gadget.meta.initialTemplate()
      : {}

    const newGadget = {
      ...initialTemplate,
      type: gadget.type,
      id: shortid.generate()
    }
    const template = this.props.template

    if (isEmpty(template)) {
      this.props.onChange({
        id: shortid.generate(),
        children: [newGadget]
      })
    } else {
      const newChildren = [...template.children, newGadget]
      this.props.onChange(template.id, {
        ...template,
        children: newChildren
      })
    }

    this.props.onClickSettings(newGadget.id)
  }

  render () {
    const {
      template,
      gadgets,
      onChange,
      onClickSettings,
      onMove,
      onDelete
    } = this.props
    return (
      <div className={styles.form}>
        {map(template.children, child => {
          if (child.children) {
            return (
              <RenderSection
                key={child.id}
                section={child}
                onChange={onChange}
                onDelete={onDelete}
                gadgets={gadgets}
                onMove={onMove}
                SectionRenderer={RenderSection}
                GadgetRenderer={RenderGadget}
                onClickSettings={onClickSettings}
                parentId={template.id}
                SectionDropzone={SectionDropzone}
              />
            )
          }
          return (
            <RenderGadget
              key={child.id}
              gadget={child}
              gadgets={gadgets}
              onClickSettings={onClickSettings}
              onDelete={onDelete}
              onMove={onMove}
              parentId={template.id}
            />
          )
        })}
        <div style={{ display: 'flex' }}>
          <_AddItem
            gadgets={gadgets}
            onGadgetSelection={this.onAddGadgetToEmptyTemplate}
          />
        </div>
      </div>
    )
  }
}
