/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import React, { Component } from 'react'

import { get, map, find, reduce, size } from 'lodash'

import { SelectField, Subheader } from '@kuali/kuali-ui'
import Button from 'react-md/lib/Buttons'

import styles from './general-settings.css'

export default class ProgressiveDisclosureConfigure extends Component {
  handleRemovePdPart = i => {
    const { value, onChange } = this.props
    const type = get(value, 'progressiveDisclosure.type') || 'any'
    let parts = get(value, 'progressiveDisclosure.parts') || []
    parts = [...parts]
    parts.splice(i, 1)
    let newVal = { type, parts }
    if (!parts.length) newVal = null
    onChange('progressiveDisclosure', newVal)
  }

  handleAddPdPart = () => {
    const { value, onChange } = this.props
    const type = get(value, 'progressiveDisclosure.type') || 'any'
    const parts = get(value, 'progressiveDisclosure.parts') || []
    const newVal = {
      type,
      parts: [...parts, { formKey: '', data: {} }]
    }
    onChange('progressiveDisclosure', newVal)
  }

  render () {
    const { progressiveDisclosures, onChange, value, validating } = this.props
    const pdParts = get(value, 'progressiveDisclosure.parts') || []
    const progressiveDisclosuresByParent = reduce(
      progressiveDisclosures,
      (memo, pd) => {
        if (!pd.parent) {
          memo.__no_parent.children.push(pd)
        } else {
          memo[pd.parent.id] = memo[pd.parent.id] || {
            id: pd.parent.id,
            label: pd.parent.label,
            children: []
          }
          memo[pd.parent.id].children.push(pd)
        }
        return memo
      },
      { __no_parent: { label: '', children: [] } }
    )
    return (
      <div>
        <div style={{ paddingTop: 15 }}>
          {pdParts.length >= 2 && (
            <div className={styles.anyOrAll}>
              <SelectField
                className={styles.select}
                onChange={val => onChange('progressiveDisclosure.type', val)}
                value={get(value, 'progressiveDisclosure.type')}
                menuItems={[
                  { value: 'any', label: 'Any' },
                  { value: 'all', label: 'All' }
                ]}
              />
              <div className={styles.subtitle}>
                {' of the following are true'}
              </div>
            </div>
          )}
          {map(pdParts, (part, i) => {
            const { formKey } = part
            const Chosen = find(progressiveDisclosures, { formKey })
            const prefix = `progressiveDisclosure.parts.${i}`

            let configIsValid = get(
              Chosen,
              'validator',
              () => size(part.data) > 0
            )

            let className = styles.pdPart
            if (validating && (!part.data || !configIsValid(part.data))) {
              className += ` ${styles.incompletePDPart}`
            }
            return (
              <div key={i} className={className}>
                <Button
                  aria-label='Remove Progressive Disclosure Row'
                  icon
                  secondary
                  onClick={() => this.handleRemovePdPart(i)}
                >
                  delete
                </Button>
                <SelectField
                  aria-label='Gadget Picker'
                  className={styles.select}
                  onChange={val => onChange(`${prefix}.formKey`, val)}
                  value={get(value, `${prefix}.formKey`)}
                  menuItems={[
                    { value: '', label: '- - -' },
                    ...reduce(
                      progressiveDisclosuresByParent,
                      (items, pdParent) => {
                        if (pdParent.label) {
                          items.push(
                            <Subheader
                              className={styles.listSubheader}
                              key={`${pdParent.id}`}
                              primaryText={pdParent.label}
                            />
                          )
                        }
                        return items.concat(
                          pdParent.children.map(pd => ({
                            value: pd.formKey,
                            label: `${pd.label} (${pd.formKey})`
                          }))
                        )
                      },
                      []
                    )
                  ]}
                />
                {Chosen && (
                  <Chosen.component
                    context={this.props.context}
                    value={get(value, `${prefix}.data`, {})}
                    onChange={val => onChange(`${prefix}.data`, val)}
                    details={Chosen.details}
                  />
                )}
              </div>
            )
          })}
        </div>
        <Button
          aria-label='Add Progressive Disclosure Condition'
          floating
          secondary
          mini
          onClick={this.handleAddPdPart}
        >
          add
        </Button>
      </div>
    )
  }
}
