/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import cx from 'classnames'
import {
  cloneDeep,
  compact,
  keys,
  map,
  reduce,
  noop,
  set,
  size,
  some,
  uniq,
  values,
  includes,
  camelCase,
  isEmpty,
  get,
  findIndex,
  find
} from 'lodash'
import React, { Component } from 'react'
import PropTypes from 'prop-types'
import { Dialog, RaisedButton, Icon } from '@kuali/kuali-ui'
import { utils } from '@kuali/formbot'

import styles from './config-settings.css'
import GeneralSettings from './general-settings'
import ProgressiveDisclosure from './config-progressive-disclosure'
const { gatherGadgetsComplex } = utils

export default class FormbotConfigSettings extends Component {
  static displayName = 'FormbotConfigSettings'

  static propTypes = {
    dismiss: PropTypes.func.isRequired,
    Formbot: PropTypes.object.isRequired,
    id: PropTypes.string.isRequired,
    onChange: PropTypes.func.isRequired,
    template: PropTypes.object.isRequired,
    trashed: PropTypes.array.isRequired,
    value: PropTypes.object.isRequired
  }

  constructor (props) {
    super(props)
    this.state = { selected: 0, pendingValues: cloneDeep(props.value || {}), validating: false }
    this.handlePropChange = this.handlePropChange.bind(this)
    this.renderFieldPreview = this.renderFieldPreview.bind(this)
  }

  componentWillReceiveProps (newProps) {
    this.setState({
      pendingValues: cloneDeep(newProps.value),
      validating: false
    })
  }

  select (e, selected) {
    e.stopPropagation()
    this.setState({ selected })
  }

  handlePropChange (name, val) {
    const values = this.state.pendingValues
    set(values, name, val)
    this.setState({
      pendingValues: values
    })
  }

  handleDismiss = _ => {
    const { template, trashed: trash, onChange } = this.props
    if (
      !get(this, 'props.value.label') &&
      !get(this, 'props.value.formKey') &&
      !get(this, 'props.value.children.length')
    ) {
      const gadgetIdToDelete = this.props.id
      const gadgetInstances = gatherGadgetsComplex(template, null, trash)
      const gadgetInstance = find(gadgetInstances, { id: gadgetIdToDelete })
      const trashed = null
      if (!gadgetInstance.parent) {
        onChange({}, null, trashed)
      }

      const newParent = cloneDeep(gadgetInstance.parent)
      const childIndex = findIndex(
        gadgetInstance.parent.children,
        child => child.id === gadgetIdToDelete
      )
      if (childIndex !== -1) newParent.children.splice(childIndex, 1)
      onChange(gadgetInstance.parent.id, newParent, trashed)
    }
    this.props.dismiss()
  }

  handleSaveClick = _ => {
    if (isEmpty(this.state.pendingValues.label)) {
      if (!this.state.validating) {
        this.setState({ validating: true })
      }
      return
    }
    const { trashed, Formbot, template } = this.props
    const { onChange } = this.props
    const gadgetDefinition = Formbot.getGadget(this.state.pendingValues.type)

    const { progressiveDisclosure } = this.state.pendingValues
    if (get(progressiveDisclosure, 'parts.length', 0) > 0) {
      const invalidPartsExist = some(progressiveDisclosure.parts, part => {
        if (!part.data) {
          return true
        }
        const gadgetReferenced = utils.recursiveFind(template.children, {
          formKey: part.formKey
        })
        if (!gadgetReferenced) {
          return true
        }

        let validationFunction = get(
          Formbot.getGadget(gadgetReferenced.type),
          'progressiveDisclosure.configIsValid',
          () => size(part.data) > 0
        )
        return !validationFunction(part.data)
      })

      if (invalidPartsExist) {
        this.setState({ validating: true })
        return
      }
    }
    if (!this.state.pendingValues.formKey && !gadgetDefinition.layout) {
      const label = get(this, 'state.pendingValues.label', '')
      let formKey = camelCase(label)
      const allGadgets = utils.gatherGadgetsById(template, trashed)
      const takenKeys = this.takenKeys(allGadgets, trashed, Formbot)
      for (let i = 1; includes(takenKeys, formKey); ++i) {
        formKey = camelCase(label) + i
      }

      this.state.pendingValues.formKey = formKey
    }
    onChange(this.props.id, this.state.pendingValues)
    this.props.dismiss()
  }

  renderFieldPreview (Formbot, template, id, context) {
    const gTemplate = utils.findGadgetById(template, id)
    const aTemplate = Object.assign({}, gTemplate, this.state.pendingValues)
    const gDefinition = Formbot.getGadget(gTemplate.type)
    aTemplate.formKey = 'temp'
    return Formbot.renderGadget(Formbot, 'edit', aTemplate, {
      context,
      shouldShow: true,
      onChange: noop,
      value: gDefinition.defaultValue
    })
  }

  takenKeys = (allGadgets, trashed, Formbot) => {
    return uniq(
      compact([
        ...map(values(allGadgets), 'formKey'),
        ...map(trashed, 'formKey'),
        ...keys(Formbot.context.presets)
      ])
    )
  }

  render () {
    const { context, Formbot, value, template, trashed, id } = this.props
    const { pendingValues, validating } = this.state
    const allGadgets = utils.gatherGadgetsById(template, trashed)
    const GadgetConfig = utils.getGadgetConfig(Formbot, template, id)
    const meta = utils.getGadgetMeta(Formbot, template, id)
    const takenKeys = this.takenKeys(allGadgets, trashed, Formbot)

    const progressiveDisclosures = reduce(
      allGadgets,
      (memo, gadget) => {
        const definition = Formbot.getGadget(gadget.type)
        if (!definition) {
          return memo
        }
        const { progressiveDisclosure } = definition
        if (progressiveDisclosure &&
          value.formKey !== gadget.formKey) {
          // This is not very performant. It's essentially traversing the entire
          // formbot template on every item. The lookup to the parents should be
          // stored somehow on the gadgets. This would be a breaking change on
          // progressive disclosure and is perhaps an indication of a need to
          // remodel the data handling in formbot due to evolving requirements.
          const path = utils.findGadgetPathById(template, gadget.id)
          const parentPath = path.split('.').slice(0, -2)
          const parent = parentPath.reduce(
            (parent, path) => parent[path],
            template
          )
          const { component, configIsValid: validator } = progressiveDisclosure
          memo.push({
            parent: {
              id: parent.id,
              label: parent.label
            },
            formKey: gadget.formKey,
            label: gadget.label,
            component,
            details: gadget.details || {},
            validator
          })
        }
        return memo
      },
      Formbot.getProgressiveDisclosures()
    )

    const actions = [
      <RaisedButton label='Cancel' onClick={this.handleDismiss} />,
      <RaisedButton
        label='Save'
        onClick={this.handleSaveClick}
        disabled={validating && isEmpty(this.state.pendingValues.label)}
      >
        <Icon name='check' />
      </RaisedButton>
    ]

    return (
      <Dialog
        id={`config-dialog-${value.id}`}
        modal
        title={meta.lbl}
        actions={actions}
        visible
        onHide={this.handleDismiss}
        focusOnMount
        containFocus
        className={styles.dialog}
        lastChild
        portal
      >
        <div
          className={cx(
            styles['gadget-config--section'],
            styles['field-preview-container']
          )}
        >
          <h3>Field Preview</h3>
          {this.renderFieldPreview(Formbot, template, id, context)}
        </div>
        <div className={styles['gadget-config--section']}>
          <h3>General</h3>
          <GeneralSettings
            value={pendingValues}
            onChange={this.handlePropChange}
            progressiveDisclosures={progressiveDisclosures}
            takenKeys={takenKeys}
            gadgetMeta={meta}
            validating={validating}
          />
        </div>
        <div className={styles['gadget-config--section']}>
          <h3>Progressive Disclosure</h3>
          <ProgressiveDisclosure
            context={context}
            value={pendingValues}
            onChange={this.handlePropChange}
            progressiveDisclosures={progressiveDisclosures}
            validating={validating}
          />
        </div>
        {GadgetConfig && (
          <div className={styles['gadget-config--section']}>
            <GadgetConfig
              context={context}
              id={id}
              value={pendingValues.details || {}}
              gadgets={allGadgets}
              onChange={newVal => {
                this.handlePropChange('details', newVal)
              }}
            />
          </div>
        )}
      </Dialog>
    )
  }
}
