/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import cx from 'classnames'
import {
  chain,
  cloneDeep,
  isFunction,
  keyBy,
  map,
  reduce,
  values
} from 'lodash'
import React, { Component } from 'react'
import PropTypes from 'prop-types'
import { DragSource } from 'react-dnd'

import styles from './config-side-bar.css'
import { utils as fb } from '@kuali/formbot'

export default class FormbotConfigSideBar extends Component {
  static displayName = 'FormbotConfigSideBar'

  static propTypes = {
    classNames: PropTypes.object,
    Formbot: PropTypes.object.isRequired,
    template: PropTypes.object.isRequired,
    trashed: PropTypes.array.isRequired
  }

  constructor () {
    super()
    this.state = { chosenGroup: 0, chosenSection: 0 }
  }

  render () {
    const { Formbot, classNames = {}, trashed, template } = this.props
    const { chosenGroup, chosenSection } = this.state
    const sections = chain(Formbot.context.gadgets)
      .map(({ meta }, type) => ({
        ...meta,
        getDragData: () => {
          const initialTemplate = isFunction(meta.initialTemplate)
            ? meta.initialTemplate()
            : {}
          return { ...initialTemplate, type }
        }
      }))
      .filter(({ hidden }) => !hidden)
      .groupBy('category')
      .map((items, lbl) => ({ lbl, items }))
      .value()
    const templateByKey = keyBy(
      values(fb.gatherGadgetsById(template)),
      'formKey'
    )
    const trashItems = reduce(
      trashed,
      (memo, t) => {
        if (!templateByKey[t.formKey]) {
          memo.push({
            icon: Formbot.getGadget(t.type).meta.icon,
            lbl: t.label,
            key: t.formKey,
            getDragData: () => cloneDeep(t)
          })
        }
        return memo
      },
      []
    )
    const trashedByKey = keyBy(trashed, 'formKey')
    const presetItems = reduce(
      Formbot.context.presets,
      (memo, preset) => {
        const { data, meta } = preset
        if (!templateByKey[data.formKey] && !trashedByKey[data.formKey]) {
          memo.push({
            ...meta,
            key: data.formKey,
            getDragData: () => cloneDeep(data)
          })
        }
        return memo
      },
      []
    )
    const data = [
      {
        lbl: 'Gadgets',
        sections
      },
      {
        lbl: 'Presets',
        sections: [{ items: presetItems }]
      },
      {
        lbl: 'Trash',
        sections: [{ items: trashItems }]
      }
    ]
    return (
      <div>
        {map(data, (group, i) => (
          <div
            key={group.lbl || i}
            className={cx(styles.group, classNames.group)}
          >
            <div
              className={cx(styles.groupLbl, classNames.groupLbl)}
              onClick={() =>
                this.setState({ chosenGroup: i, chosenSection: 0 })
              }
              children={group.lbl}
            />
            {chosenGroup === i && (
              <div>
                <div className={cx(styles.sections, classNames.sections)}>
                  {map(group.sections, (section, j) => {
                    if (!section.lbl) {
                      return null
                    }
                    const className = cx(
                      styles.sectionLbl,
                      classNames.sectionLbl,
                      {
                        [styles.active]: chosenSection === j
                      }
                    )
                    return (
                      <div
                        key={section.lbl || j}
                        onClick={() => this.setState({ chosenSection: j })}
                        className={className}
                        children={section.lbl}
                      />
                    )
                  })}
                </div>
                <div className={cx(styles.section, classNames.section)}>
                  {map(group.sections[chosenSection].items, item => (
                    <ActualItem key={item.key || item.lbl} item={item} />
                  ))}
                </div>
              </div>
            )}
          </div>
        ))}
      </div>
    )
  }
}

class Item extends Component {
  static displayName = 'Item'

  static propTypes = {
    connectDragSource: PropTypes.func.isRequired,
    isDragging: PropTypes.bool.isRequired,
    item: PropTypes.object.isRequired
  }

  render () {
    const { item, isDragging, connectDragSource } = this.props
    const className = cx(styles.item, { [styles.dragging]: isDragging })
    return connectDragSource(
      <div className={className}>
        <img className={styles.icon} role='presentation' src={item.icon} />
        <div>{item.lbl}</div>
      </div>
    )
  }
}

const itemSource = {
  beginDrag (props) {
    return { data: props.item.getDragData() }
  }
}

function collect (connect, monitor) {
  return {
    connectDragSource: connect.dragSource(),
    isDragging: monitor.isDragging()
  }
}

const ActualItem = DragSource('Gadget', itemSource, collect)(Item)
