/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import { get, map, isEmpty } from 'lodash'
import React, { Component } from 'react'
import PropTypes from 'prop-types'
import {
  Form,
  FieldGroup,
  Label,
  TextField,
  SelectField
} from '@kuali/kuali-ui'
import { FieldCell } from '@kuali/kuali-ui/lib/forms'
import { sanitize } from './utils'
import cx from 'classnames'
import Checkbox from 'react-md/lib/SelectionControls/Checkbox'

import styles from './general-settings.css'

let Quill
if (typeof document !== 'undefined') {
  Quill = require('react-quill')
}

const toolbarItems = [
  [{ font: [] }],
  [{ size: [] }],
  [{ align: [] }],
  ['bold', 'italic', 'strike', 'underline'],
  [{ color: [] }, { background: [] }],
  [{ list: 'ordered' }, { list: 'bullet' }],
  ['link']
]

const gadgetSizeMenuItems = {
  small: 'Small (A few words)',
  medium: 'Medium (A small sentence)',
  large: 'Large (A large sentence)',
  xlarge: 'Extra Large (A paragraph)'
}

const keyboard = {
  bindings: {
    tab: {
      key: 9,
      handler: () => true // allow normal tab key event propagation
    }
  }
}

export default class GeneralSettings extends Component {
  static displayName = 'GeneralSettings'

  static propTypes = {
    onChange: PropTypes.func.isRequired,
    onUpdateFormKey: PropTypes.func,
    takenKeys: PropTypes.array.isRequired,
    value: PropTypes.object.isRequired,
    gadgetMeta: PropTypes.object,
    validating: PropTypes.bool.isRequired
  }

  render () {
    const {
      onChange,
      value,
      gadgetMeta,
      validating
    } = this.props

    const sizeMenu =
      gadgetMeta && gadgetMeta.sizes
        ? map(gadgetMeta.sizes, size => {
            return { name: get(gadgetSizeMenuItems, size), value: size }
          })
        : null
    return (
      <div>
        <Form showGrid>
          <div className={styles.textContainer}>
            <FieldGroup>
              <FieldCell>
                <Label htmlFor='fieldLabel'>Field Label</Label>
                <TextField
                  id='fieldLabel'
                  error={validating && isEmpty(value.label)}
                  defaultValue={value.label}
                  onChange={value => {
                    onChange('label', value)
                  }}
                />
              </FieldCell>
            </FieldGroup>
            <FieldGroup>
              <FieldCell>
                <Checkbox
                  id='displayLabel'
                  checked={get(
                    value,
                    'displayLabel',
                    !get(value, 'displayQuestionOnly', false)
                  )}
                  onChange={value => {
                    onChange('displayLabel', value)
                  }}
                  label='Display Field Label on form'
                />
              </FieldCell>
            </FieldGroup>
          </div>
          {(<div className={styles.textContainer}>
              <FieldGroup>
                <FieldCell>
                  <Label className={styles.quillLabel} htmlFor='question'>
                    Question (optional)
                  </Label>
                  <Quill
                    className={styles.quill}
                    modules={{ toolbar: toolbarItems, keyboard }}
                    theme='snow'
                    defaultValue={sanitize(value.question || '')}
                    onChange={val => onChange('question', val)}
                  />
                </FieldCell>
              </FieldGroup>
              <FieldGroup>
                <FieldCell>
                  <Checkbox
                    id='displayQuestion'
                    checked={get(value, 'displayQuestion', true)}
                    onChange={value => {
                      onChange('displayQuestion', value)
                    }}
                    label='Display Question on form'
                  />
                </FieldCell>
              </FieldGroup>
            </div>
          )}
          <div className={styles.textContainer}>
            <FieldGroup>
              <FieldCell>
                <Label
                  htmlFor='fieldDescription'
                  className={styles.quillLabel}
                >
                  Description (optional)
                </Label>
                {(<Quill
                    className={styles.quill}
                    modules={{ toolbar: toolbarItems, keyboard }}
                    theme='snow'
                    defaultValue={sanitize(value.desc || '')}
                    onChange={val => onChange('desc', val)}
                  />
                )}
              </FieldCell>
            </FieldGroup>
            <FieldGroup>
              <FieldCell>
                <Checkbox
                  id='displayDescInline'
                  checked={get(value, 'displayDescInline', true)}
                  onChange={value => {
                    onChange('displayDescInline', value)
                  }}
                  label='Display Description inline in form'
                />
              </FieldCell>
            </FieldGroup>
          </div>
          {sizeMenu && (
            <FieldGroup>
              <FieldCell>
                <Label htmlFor='fieldSize'>Field Width</Label>
                <SelectField
                  id='fieldSize'
                  placeholder='Choose a field width'
                  menuItems={sizeMenu}
                  itemLabel='name'
                  itemValue='value'
                  onChange={value => {
                    onChange('fieldSize', value)
                  }}
                  value={value.fieldSize}
                />
              </FieldCell>
            </FieldGroup>
          )}
        </Form>
      </div>
    )
  }
}
