/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import MockAdapter from 'axios-mock-adapter'
import axios from '../axios'

describe('lib/axios', () => {
  let mock
  beforeAll(() => {
    mock = new MockAdapter(axios)
    mock.onAny().reply(config => [200, config])
  })

  afterAll(() => {
    mock.reset()
  })

  afterEach(() => {
    document.cookie = 'authToken='
  })

  it('adds Content-Type headers if not present', async () => {
    const response = await axios.get('/')
    expect(response.data.headers).toHaveProperty(
      'Content-Type',
      'application/json'
    )
  })

  it('does not change Content-Type headers if previously set', async () => {
    const response = await axios.get('/', {
      headers: {
        'Content-Type': 'text/css'
      }
    })
    expect(response.data.headers).toHaveProperty('Content-Type', 'text/css')
  })

  it("sets Authorization token if it's in the cookie", async () => {
    document.cookie = 'authToken=foobar'
    let response = await axios.get('/')
    expect(response.data.headers).toHaveProperty(
      'Authorization',
      'Bearer foobar'
    )
    document.cookie = 'authToken=barbaz;anotherkey=foo'
    response = await axios.get('/')
    expect(response.data.headers).toHaveProperty(
      'Authorization',
      'Bearer barbaz'
    )
  })

  it('does not set Authorization header if token does not exist', async () => {
    document.cookie = 'authToken='
    const response = await axios.get('/')
    expect(response.data.headers).not.toHaveProperty('Authorization')
  })

  it('does not set Authorization header if Authorization header is set', async () => {
    document.cookie = 'authToken=foobar'
    const response = await axios.get('/', {
      headers: {
        Authorization: 'Bearer anotherKey'
      }
    })
    expect(response.data.headers).toHaveProperty(
      'Authorization',
      'Bearer anotherKey'
    )
  })
})
