/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import KeyedList from '../keyed-list'
import { cloneDeep } from 'lodash'
import React from 'react'
import { mount } from 'enzyme'

describe('Keyed List', () => {
  let props, wrapper
  beforeEach(() => {
    props = {
      onChange: jest.fn(),
      value: [
        { lbl: 'First Label', key: 'firstKey' },
        { lbl: 'Another Label', key: 'secondKey' },
        { lbl: 'Temp Label', tempKey: 'tempKey' }
      ]
    }
    wrapper = mount(<KeyedList {...props} />)
  })

  it('should render', () => {
    expect(wrapper).toMatchSnapshot()
  })

  describe('removeValue', () => {
    it('should remove one of options and call onChange with the new list', () => {
      const index = 1
      const newValues = cloneDeep(wrapper.props().value)
      newValues.splice(index, 1)
      wrapper.getNode().removeValue(index)
      expect(props.onChange).toHaveBeenLastCalledWith(newValues)
    })
  })

  describe('addItem', () => {
    it('should create a new generic item update options to onChange and focusIndex', () => {
      wrapper.getNode().addItem()
      expect(props.onChange).toHaveBeenCalledWith([
        ...props.value,
        { key: expect.any(String), lbl: expect.stringMatching(/^(?![\s\S])/) }
      ])
    })
  })

  describe('Field Focus', () => {
    it('should set the focusIndex if setFocus is called', () => {
      expect(wrapper.state().focusIndex).toEqual(-1)
      wrapper.getNode().setFocus(1)
      expect(wrapper.state().focusIndex).toEqual(1)
    })

    it('should focus a field if the focusIndex corresponds to a field being rendered', () => {
      const fields = wrapper.find('TextField').getNodes()
      const index = 1
      fields.forEach(field => (field.focus = jest.fn()))
      wrapper.setState({ focusIndex: index })
      fields.forEach((field, i) => {
        if (i === index) {
          expect(field.focus).toHaveBeenCalled()
        } else {
          expect(field.focus).not.toHaveBeenCalled()
        }
      })
    })
  })

  describe('addValue', () => {
    it('should update the label for a value and notifiy onChange', () => {
      const newLabel = 'New Label'
      const newValues = cloneDeep(wrapper.props().value)
      wrapper.getNode().addValue('firstKey', newLabel)
      newValues[0].lbl = newLabel
      expect(props.onChange).toHaveBeenCalledWith(newValues)
    })

    it('should not call onChange if the key cannot be found', () => {
      wrapper.getNode().addValue('notAKey', 'New Label')
      expect(props.onChange).not.toHaveBeenCalled()
    })
  })

  describe('Component Tests', () => {
    let spy
    beforeEach(() => {
      spy = jest.fn()
    })

    describe('Add Button', () => {
      it('should call addItem when the add button is clicked', () => {
        const btn = wrapper.find('FlatButton').first()
        wrapper.getNode().addItem = spy
        btn.props().onClick()
        expect(spy).toHaveBeenCalled()
      })
    })

    describe('Remove Button', () => {
      it('should remove the item with index of that item', () => {
        const btn = wrapper.find('IconButton').first()
        wrapper.getNode().removeValue = spy
        btn.props().onClick()
        expect(spy).toHaveBeenCalledWith(0)
      })
    })

    describe('Fields', () => {
      let field
      beforeEach(() => {
        field = wrapper.find('TextField').first()
      })

      it('should set the focus index to -1 when a text field is blurred', () => {
        wrapper.getNode().setFocus = spy
        field.props().onBlur()
        expect(spy).toHaveBeenCalledWith(-1)
      })

      it('should add a value to the list when a text field changes', () => {
        wrapper.getNode().addValue = spy
        field.props().onChange('new value')
        expect(spy).toHaveBeenLastCalledWith('firstKey', 'new value')
      })

      it('should add an item if the enter key is pressed in a text field', () => {
        const preventDefault = jest.fn()
        wrapper.getNode().addItem = spy
        field.props().onKeyDown({ keyCode: 13, preventDefault })
        expect(preventDefault).toHaveBeenCalled()
        expect(spy).toHaveBeenCalled()
      })

      it('should not add an item if a non-enter key was pressed', () => {
        wrapper.getNode().addItem = spy
        field.props().onKeyDown({ keyCode: 10 })
        expect(spy).not.toHaveBeenCalled()
      })
    })
  })
})
