/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import cx from 'classnames'
import { get } from 'lodash'
import { sanitize } from './utils'
import React from 'react'
import { Icon } from '@kuali/kuali-ui'
import { Tooltipped } from 'react-md'
import styles from './gadget-renderer.css'

const ERROR_STYLE = {
  backgroundColor: '#D9534F',
  color: 'white',
  padding: 15,
  fontSize: 20,
  textAlign: 'center'
}

export function buildHeadingParts (Formbot, mode, template) {
  const label = template.label &&
    get(
      template,
      'displayLabel',
      !get(template, 'displayQuestionOnly', false)
    ) && <label className={styles.label}>{template.label}</label>
  const question = template.question &&
    get(template, 'displayQuestion', true) && (
      <p
        className={`${styles.question} ${styles.richText} ql-editor`}
        dangerouslySetInnerHTML={{ __html: sanitize(template.question) }}
      />
    )
  const displayDescriptionInline = get(template, 'displayDescInline', true)
  const descInner = template.desc && (
    <p
      className={cx(
        {
          [styles.desc]: displayDescriptionInline,
          [styles.descTooltip]: !displayDescriptionInline
        },
        styles.richText,
        'ql-editor'
      )}
      dangerouslySetInnerHTML={{ __html: sanitize(template.desc) }}
    />
  )
  const desc =
    mode === 'edit' &&
    descInner &&
    (displayDescriptionInline ? (
      descInner
    ) : (
      <Tooltipped label={descInner} position='right'>
        <span className={styles.tooltipWrapper}>
          <Icon name='help' variant='info' />
        </span>
      </Tooltipped>
    ))

  return { label, desc, question }
}

export default function renderGadget (Formbot, mode, template, options) {
  const {
    context,
    children,
    dependent,
    fbRender,
    shouldShow,
    onChange,
    value
  } = options

  const gadgetDef = Formbot.getGadget(template.type)
  const Gadget = gadgetDef[mode]

  if (!shouldShow || (!template.formKey && !gadgetDef.layout)) return null

  createErrorBoundary(Gadget)
  const wrapperStyles = cx(
    styles.gadget,
    styles[`size-${template.fieldSize || 'large'}`],
    {
      [styles.noPad]: options.noPad
    }
  )

  const a11yDesc = template.label
  const props = {
    context,
    formKey: template.formKey,
    id: template.id,
    value,
    onChange,
    dependent: dependent || {},
    fbRender,
    details: template.details || {},
    children,
    NOTFOUND_TYPE: template.type,
    a11yDesc
  }
  const decorators = Formbot.getDecorators({ type: template.type, mode })
  const headerParts = buildHeadingParts(Formbot, mode, template)
  const createComponent = _props => {
    const gadget = <Gadget {..._props} />
    const parts = decorators.length
      ? Formbot.decorate(
          decorators,
          { ...headerParts, gadget },
          { ..._props, template },
          gadgetDef
        )
      : { ...headerParts, gadget }

    return (
      <div className={wrapperStyles} key={template.id}>
        {parts.label}
        {parts.question}
        {parts.desc}
        {parts.gadget}
      </div>
    )
  }
  const component = createComponent(props)
  const decProps = { ...props, template }
  return decorators.length
    ? Formbot.decorate(
        decorators,
        { component },
        decProps,
        gadgetDef,
        createComponent
      ).component
    : component
}

// This is the best we can do until we upgrade to React 15, in which version
// they released Error Boundaries
// https://github.com/facebook/react/pull/5602
// https://github.com/facebook/react/pull/6020
// Also to note: this won't work in dev mode. We use a library that catches our
// errors for us (before we can catch it here) and throws up a big red screen.
function createErrorBoundary (Component) {
  if (Component.prototype.originalRender) {
    return
  }
  Component.prototype.originalRender = Component.prototype.render
  Component.prototype.render = function render () {
    try {
      return Component.prototype.originalRender.call(this)
    } catch (e) {
      console.error(e)
      return (
        <div style={ERROR_STYLE}>
          {'Sorry! This piece of the form is temporarily unavailable.'}
        </div>
      )
    }
  }
}
