/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import {
  each,
  reduce,
  isEmpty,
  get,
  find,
  filter,
  map,
  matches,
  concat,
  flatten,
  partialRight,
  merge
} from 'lodash'

import _sanitize from 'sanitize-html'

export function gatherGadgetsById (tmpl, trashed) {
  let gadgets = {}
  if (tmpl.id) {
    gadgets[tmpl.id] = { ...tmpl }
    if (tmpl.children) {
      each(tmpl.children, child => {
        const moar = gatherGadgetsById(child)
        gadgets = { ...gadgets, ...moar }
      })
    }
  }
  if (trashed) {
    each(trashed, t => {
      gadgets[t.id] = t
    })
  }
  return gadgets
}

export function gatherGadgetsComplex (tmpl, parent, trashed) {
  let gadgets = []
  if (tmpl.id) gadgets.push({ ...tmpl, parent })
  if (tmpl.children) {
    each(tmpl.children, child => {
      const moar = gatherGadgetsComplex(child, tmpl)
      gadgets = [...gadgets, ...moar]
    })
  }
  if (trashed) gadgets = [...gadgets, ...trashed]
  return gadgets
}

export function gatherGadgetsByIdComplex (tmpl, parent) {
  let gadgets = {}
  if (tmpl.id) gadgets[tmpl.id] = { ...tmpl, parent }
  if (tmpl.children) {
    each(tmpl.children, child => {
      const moar = gatherGadgetsByIdComplex(child, tmpl)
      gadgets = { ...gadgets, ...moar }
    })
  }
  return gadgets
}

export function getGadgetConfig (Formbot, tmpl, id) {
  const gadget = findGadgetComplex(tmpl, id)
  const gadgetDefinition = Formbot.getGadget(gadget.type)
  return gadgetDefinition.config
}

export function getGadgetPD (Formbot, tmpl, id) {
  const gadget = findGadgetComplex(tmpl, id)
  const gadgetDefinition = Formbot.getGadget(gadget.type)
  return gadgetDefinition.progressiveDisclosure
}

export function getGadgetMeta (Formbot, tmpl, id) {
  const gadget = findGadgetComplex(tmpl, id)
  const gadgetDefinition = Formbot.getGadget(gadget.type)
  return gadgetDefinition.meta
}

export function getSettingsPanels (Formbot, GeneralSettings, tmpl, id) {
  const gadget = findGadgetComplex(tmpl, id)
  const gadgetDefinition = Formbot.getGadget(gadget.type)
  const panels = [
    {
      label: 'General',
      component: GeneralSettings
    }
  ]
  if (gadgetDefinition.config) {
    panels.push({
      key: 'details',
      label: 'Specific',
      component: gadgetDefinition.config
    })
  }
  // panels.push(VALIDATIONS)
  if (gadget.parent && gadget.parent.type) {
    const parentGadgetDefinition = Formbot.getGadget(gadget.parent.type)
    if (parentGadgetDefinition && parentGadgetDefinition.childConfig) {
      panels.push({
        key: 'parentDetails',
        label: 'Layout',
        component: parentGadgetDefinition.childConfig
      })
    }
  }
  // panels.push(ROLES)
  return panels
}

function findGadgetComplex (tmpl, id) {
  const gadgets = gatherGadgetsByIdComplex(tmpl)
  return gadgets[id]
}

export function findGadgetById (tmpl, id) {
  const gadgets = gatherGadgetsById(tmpl)
  return gadgets[id]
}

export function findGadgetPathById (tmpl, id) {
  const res = findGadgetPath(tmpl, id, []) || []
  return res.join('.')
}

function findGadgetPath (tmpl, id, chunks) {
  if (tmpl.id === id) return chunks
  if (tmpl.children) {
    const path = reduce(
      tmpl.children,
      (memo, child, i) => {
        const newChunks = [...chunks, 'children', i]
        const res = findGadgetPath(child, id, newChunks)
        return memo || res
      },
      null
    )
    if (path) return path
  }
  return null
}

export function findGadgetsInProgDiscFromList (gadget, checkGadgetList) {
  const selfCheck = findDependentProgDiscGadgets(gadget, checkGadgetList)

  const children = gadget.children || []

  const childrenChecks = flatten(
    concat(
      map(children, child =>
        findGadgetsInProgDiscFromList(child, checkGadgetList)
      )
    )
  )

  return concat(childrenChecks, selfCheck)
}

function findDependentProgDiscGadgets (targetGadget, gadgetList) {
  if (!targetGadget.formKey) return []
  const items = filter(gadgetList, instance =>
    gadgetUsesFormKeyInProgDisc(instance, targetGadget.formKey)
  )
  return map(items, item => ({
    progDiscGadget: item,
    targetGadget
  }))
}

function gadgetUsesFormKeyInProgDisc (gadget, formKey) {
  if (isEmpty(gadget.progressiveDisclosure)) return false
  const parts = get(gadget, 'progressiveDisclosure.parts', [])
  return !!find(parts, { formKey })
}

const extendedAllowedAttributes = merge(
  {},
  _sanitize.defaults.allowedAttributes,
  {
    '*': ['style', 'class']
  }
)

const extendedAllowedTags = _sanitize.defaults.allowedTags.concat([
  'span',
  's',
  'u'
])

export const sanitize = partialRight(_sanitize, {
  allowedAttributes: extendedAllowedAttributes,
  allowedTags: extendedAllowedTags
})

export function recursiveFind (gadgets, matchPattern) {
  const isMatch = matches(matchPattern)
  for (const gadget of gadgets) {
    if (isMatch(gadget)) {
      return gadget
    } else if (gadget.children) {
      const childResult = recursiveFind(gadget.children, matchPattern)
      if (childResult) {
        return childResult
      }
    }
  }
}
