'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

var getStartOfDay = exports.getStartOfDay = function getStartOfDay() {
  var sod = new Date();
  sod.setHours(0);
  sod.setMinutes(0);
  sod.setSeconds(0);
  sod.setMilliseconds(0);
  return sod;
};

var getSecondsFromString = exports.getSecondsFromString = function getSecondsFromString(str) {
  var split = str.split(':');
  return split[0] * Math.pow(60, 2) + split[1] * 60;
};

var OPERATION_TYPES = exports.OPERATION_TYPES = {
  AND: 'And',
  OR: 'Or'
};

var FIELD_TYPES = exports.FIELD_TYPES = {
  DATE: 'DATE',
  ENUM: 'ENUM',
  TEXT: 'TEXT',
  NUMBER: 'NUMBER',
  TIME: 'TIME'
};

var OPERATOR_TYPES = exports.OPERATOR_TYPES = {
  EQUAL_TO: 'is equal to',
  NOT_EQUAL_TO: 'is not equal to',
  GREATER_THAN: 'is greater than',
  GREATER_THAN_OR_EQUAL_TO: 'is greater than or equal to',
  LESS_THAN: 'is less than',
  LESS_THAN_OR_EQUAL_TO: 'is less than or equal to',
  BEFORE: 'is before',
  AFTER: 'is after',
  CONTAINS: 'contains',
  DOES_NOT_CONTAIN: 'does not contain',
  STARTS_WITH: 'starts with',
  ENDS_WITH: 'ends with',
  BLANK: 'is blank',
  NOT_BLANK: 'is not blank'
};

var OPERATOR_KEYS = exports.OPERATOR_KEYS = Object.keys(OPERATOR_TYPES).reduce(function (acc, key) {
  acc[OPERATOR_TYPES[key]] = key;
  return acc;
}, {});

// exported for testing
var getFieldFromRow = exports.getFieldFromRow = function getFieldFromRow(field, row) {
  return typeof field === 'function' ? field(row) : row[field];
};
var EQUAL_TO = function EQUAL_TO(field, value) {
  return function (row) {
    return getFieldFromRow(field, row) === value;
  };
};
var NOT_EQUAL_TO = function NOT_EQUAL_TO(field, value) {
  return function (row) {
    return getFieldFromRow(field, row) !== value;
  };
};
var LESS_THAN = function LESS_THAN(field, value) {
  return function (row) {
    return getFieldFromRow(field, row) < value;
  };
};
var LESS_THAN_OR_EQUAL_TO = function LESS_THAN_OR_EQUAL_TO(field, value) {
  return function (row) {
    return getFieldFromRow(field, row) <= value;
  };
};
var GREATER_THAN = function GREATER_THAN(field, value) {
  return function (row) {
    return getFieldFromRow(field, row) > value;
  };
};
var GREATER_THAN_OR_EQUAL_TO = function GREATER_THAN_OR_EQUAL_TO(field, value) {
  return function (row) {
    return getFieldFromRow(field, row) >= value;
  };
};
var BLANK = function BLANK(field) {
  return function (row) {
    return !getFieldFromRow(field, row);
  };
};
var NOT_BLANK = function NOT_BLANK(field) {
  return function (row) {
    return !!getFieldFromRow(field, row);
  };
};

var OPERATOR_FUNCTIONS = exports.OPERATOR_FUNCTIONS = {
  DATE: {
    AFTER: function AFTER(field, value) {
      if (!value) return function () {};
      var vDate = new Date(value);
      if (isNaN(vDate.getTime())) {
        throw Error('Rule value for ' + field + ', \'' + value + '\' is not a valid Date. Unable to return filter');
      }
      return function (row) {
        var rowValue = getFieldFromRow(field, row);
        var rowDate = new Date(rowValue);
        if (isNaN(rowDate.getTime())) {
          throw Error('Row value for ' + field + ', \'' + rowValue + '\' is not a valid Date. Unable to run filter');
        }
        return rowDate > vDate;
      };
    },
    BEFORE: function BEFORE(field, value) {
      if (!value) return function () {};
      var vDate = new Date(value);
      if (isNaN(vDate.getTime())) {
        throw Error('Rule value for ' + field + ', \'' + value + '\' is not a valid Date. Unable to return filter');
      }
      return function (row) {
        var rowValue = getFieldFromRow(field, row);
        var rowDate = new Date(rowValue);
        if (isNaN(rowDate.getTime())) {
          throw Error('Row value for ' + field + ', \'' + rowValue + '\' is not a valid Date. Unable to run filter');
        }
        return rowDate < vDate;
      };
    },
    BLANK: BLANK,
    EQUAL_TO: function EQUAL_TO(field, value) {
      if (!value) return function () {};
      var time = new Date(value).getTime();
      if (isNaN(time)) {
        throw Error('Rule value for ' + field + ', \'' + value + '\' is not a valid Date. Unable to return filter');
      }
      return function (row) {
        var rowValue = getFieldFromRow(field, row);
        var rowDate = new Date(rowValue);
        if (isNaN(rowDate.getTime())) {
          throw Error('Row value for ' + field + ', \'' + rowValue + '\' is not a valid Date. Unable to run filter');
        }
        return rowDate.getTime() === time;
      };
    },
    NOT_BLANK: NOT_BLANK,
    NOT_EQUAL_TO: function NOT_EQUAL_TO(field, value) {
      if (!value) return function () {};
      var time = new Date(value).getTime();
      if (isNaN(time)) {
        throw Error('Rule value for ' + field + ', \'' + value + '\' is not a valid Date. Unable to return filter');
      }
      return function (row) {
        var rowValue = getFieldFromRow(field, row);
        var rowDate = new Date(rowValue);
        if (isNaN(rowDate.getTime())) {
          throw Error('Row value for ' + field + ', \'' + rowValue + '\' is not a valid Date. Unable to run filter');
        }
        return rowDate.getTime() !== time;
      };
    }
  },
  TIME: {
    AFTER: function AFTER(field, value) {
      return function (row) {
        return getSecondsFromString(getFieldFromRow(field, row)) > getSecondsFromString(value);
      };
    },
    BEFORE: function BEFORE(field, value) {
      return function (row) {
        return getSecondsFromString(getFieldFromRow(field, row)) < getSecondsFromString(value);
      };
    },
    BLANK: BLANK,
    EQUAL_TO: EQUAL_TO,
    NOT_BLANK: NOT_BLANK,
    NOT_EQUAL_TO: NOT_EQUAL_TO
  },
  ENUM: {
    BLANK: BLANK,
    EQUAL_TO: EQUAL_TO,
    NOT_BLANK: NOT_BLANK,
    NOT_EQUAL_TO: NOT_EQUAL_TO
  },
  TEXT: {
    BLANK: BLANK,
    CONTAINS: function CONTAINS(field, value) {
      return function (row) {
        return ~getFieldFromRow(field, row).toUpperCase().search(value.toUpperCase());
      };
    },
    DOES_NOT_CONTAIN: function DOES_NOT_CONTAIN(field, value) {
      return function (row) {
        return !~getFieldFromRow(field, row).toUpperCase().search(value.toUpperCase());
      };
    },
    ENDS_WITH: function ENDS_WITH(field, value) {
      return function (row) {
        return getFieldFromRow(field, row).toUpperCase().substr(value.length * -1) === value.toUpperCase();
      };
    },
    EQUAL_TO: EQUAL_TO,
    NOT_BLANK: NOT_BLANK,
    NOT_EQUAL_TO: NOT_EQUAL_TO,
    STARTS_WITH: function STARTS_WITH(field, value) {
      return function (row) {
        return getFieldFromRow(field, row).toUpperCase().search(value.toUpperCase()) === 0;
      };
    }
  },
  NUMBER: {
    EQUAL_TO: EQUAL_TO,
    NOT_EQUAL_TO: NOT_EQUAL_TO,
    GREATER_THAN: GREATER_THAN,
    GREATER_THAN_OR_EQUAL_TO: GREATER_THAN_OR_EQUAL_TO,
    LESS_THAN: LESS_THAN,
    LESS_THAN_OR_EQUAL_TO: LESS_THAN_OR_EQUAL_TO,
    BLANK: BLANK,
    NOT_BLANK: NOT_BLANK
  }
};

var AVAILABLE_OPERATORS = exports.AVAILABLE_OPERATORS = {
  DATE: [OPERATOR_TYPES.AFTER, OPERATOR_TYPES.BEFORE, OPERATOR_TYPES.BLANK, OPERATOR_TYPES.EQUAL_TO, OPERATOR_TYPES.NOT_BLANK, OPERATOR_TYPES.NOT_EQUAL_TO],
  TIME: [OPERATOR_TYPES.AFTER, OPERATOR_TYPES.BEFORE, OPERATOR_TYPES.BLANK, OPERATOR_TYPES.EQUAL_TO, OPERATOR_TYPES.NOT_BLANK, OPERATOR_TYPES.NOT_EQUAL_TO],
  ENUM: [OPERATOR_TYPES.BLANK, OPERATOR_TYPES.EQUAL_TO, OPERATOR_TYPES.NOT_BLANK, OPERATOR_TYPES.NOT_EQUAL_TO],
  TEXT: [OPERATOR_TYPES.BLANK, OPERATOR_TYPES.CONTAINS, OPERATOR_TYPES.DOES_NOT_CONTAIN, OPERATOR_TYPES.ENDS_WITH, OPERATOR_TYPES.EQUAL_TO, OPERATOR_TYPES.NOT_BLANK, OPERATOR_TYPES.NOT_EQUAL_TO, OPERATOR_TYPES.STARTS_WITH],
  NUMBER: [OPERATOR_TYPES.BLANK, OPERATOR_TYPES.EQUAL_TO, OPERATOR_TYPES.GREATER_THAN, OPERATOR_TYPES.GREATER_THAN_OR_EQUAL_TO, OPERATOR_TYPES.LESS_THAN, OPERATOR_TYPES.LESS_THAN_OR_EQUAL_TO, OPERATOR_TYPES.NOT_BLANK, OPERATOR_TYPES.NOT_EQUAL_TO]
};