/* global describe, it, beforeEach */

const { exec } = require('child_process')
const expect = require('expect')
const path = require('path')
const snapshot = require('snap-shot')

const execAsync = (cmd, opts) => {
  return new Promise((resolve, reject) => {
    exec(cmd, opts, (err, stdout, stderr) => {
      if (!err) return resolve({ stdout, stderr })
      err.stdout = stdout
      err.stderr = stderr
      reject(err)
    })
  })
}

const failure = Symbol('EXPECT_THROW FAIL')
async function expectThrow (asyncFn) {
  try {
    await asyncFn()
    throw failure
  } catch (e) {
    if (e === failure) {
      throw new Error('Expected function to throw but it did not')
    }
    return e
  }
}

describe('Kuali-Utils CLI', () => {
  const cwd = path.resolve(__dirname, 'mock')

  async function cli (args) {
    const output = await execAsync(`node ../../src/cli.js ${args}`, { cwd })
    expect(output.stderr).toBeFalsy()
    return output
  }

  async function cliThrow (args) {
    const output = await expectThrow(async () => {
      await execAsync(`node ../../src/cli.js ${args}`, { cwd })
    })
    expect(output.stdout).toBeFalsy()
    return output
  }

  beforeEach(async () => {
    await execAsync(`rm -r ${cwd}`)
    await execAsync(`mkdir -p ${cwd}`)
  })

  describe('help command', () => {
    it('kuali-utils', async () => {
      const output = await cli('')
      snapshot(output.stdout)
    })

    it('kuali-utils help', async () => {
      const output = await cli('help')
      snapshot(output.stdout)
    })

    it('kuali-utils h', async () => {
      const output = await cli('h')
      snapshot(output.stdout)
    })

    it('kuali-utils --help', async () => {
      const output = await cli('--help')
      snapshot(output.stdout)
    })

    it('kuali-utils -h', async () => {
      const output = await cli('-h')
      snapshot(output.stdout)
    })

    it('kuali-utils no-idea-what-i-am-doing', async () => {
      const output = await cli('no-idea-what-i-am-doing')
      snapshot(output.stdout)
    })
  })

  describe('version command', () => {
    it('kuali-utils version', async () => {
      const output = await cli('version')
      snapshot(output.stdout)
    })

    it('kuali-utils v', async () => {
      const output = await cli('v')
      snapshot(output.stdout)
    })

    it('kuali-utils --version', async () => {
      const output = await cli('--version')
      snapshot(output.stdout)
    })

    it('kuali-utils -v', async () => {
      const output = await cli('-v')
      snapshot(output.stdout)
    })
  })

  describe('license command', () => {
    it('kuali-utils license - adds the license if a js file has none', async () => {
      await execAsync(`touch foo.js`, { cwd })
      const output = await cli('license')
      snapshot(output.stdout)
    })

    it('kuali-utils license - adds the license if a css file has none', async () => {
      await execAsync(`touch foo.css`, { cwd })
      const output = await cli('license')
      snapshot(output.stdout)
    })

    it('kuali-utils license - works if no files are present', async () => {
      const output = await cli('license')
      snapshot(output.stdout)
    })

    it('kuali-utils license - does not add the license to a non js/css file', async () => {
      await execAsync(`touch foo.md`, { cwd })
      const output = await cli('license')
      snapshot(output.stdout)
    })

    it('kuali-utils license - does not add the license if it already exists', async () => {
      await execAsync(`cat ../../src/copyright-notice.txt > foo.js`, { cwd })
      const output = await cli('license')
      snapshot(output.stdout)
    })
  })

  describe('check command', () => {
    it('kuali-utils check - works if no files are present', async () => {
      const output = await cli('check')
      expect(output.stdout).toBeFalsy()
    })

    it('kuali-utils check - works if a non js/css file is present', async () => {
      await execAsync(`touch foo.md`, { cwd })
      const output = await cli('check')
      expect(output.stdout).toBeFalsy()
    })

    it('kuali-utils check - works if the js/css file has a license', async () => {
      await execAsync(`cat ../../src/copyright-notice.txt > foo.js`, { cwd })
      const output = await cli('check')
      expect(output.stdout).toBeFalsy()
    })

    it('kuali-utils check - throws an error if a js file has no license', async () => {
      await execAsync(`touch foo.js`, { cwd })
      const output = await cliThrow('check')
      snapshot(output.stderr)
    })

    it('kuali-utils check - throws an error if a css file has no license', async () => {
      await execAsync(`touch foo.css`, { cwd })
      const output = await cliThrow('check')
      snapshot(output.stderr)
    })
  })
})
