/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// This file is used to shim the redux store between struts and react

const RTK = window.RTK;

const loadStateFromStorage = () => {
  try {
    const serialState = sessionStorage.getItem('appState');
    if (serialState === null) {
      return undefined;
    }
    const stateWrapper = JSON.parse(serialState);
    const anHourAgo = Date.now() - (1000 * 60 * 60);
    if (stateWrapper.lastUpdateDate < anHourAgo) {
      return undefined;
    }
    return stateWrapper.state;
  } catch (err) {
    return undefined;
  }
};

const saveStateToStorage = (state) => {
  try {
    const serialState = JSON.stringify({ lastUpdateDate: Date.now(), state });
    sessionStorage.setItem('appState', serialState);
  } catch (err) {
    console.log(err);
  }
};

const browserStorageListener = RTK.createListenerMiddleware();

browserStorageListener.startListening({
  predicate: () => true,
  effect: (_action, listenerApi) => {
    saveStateToStorage(listenerApi.getState());
  }
});

// dummy slice to use for slices only used by the react app piece
const dummySlice = RTK.createSlice({
  name: 'dummy',
  initialState: null,
  reducers: {
    noop: () => {}
  }
});

// Leaving this here to show what the shape of the history object should be until
// we convert all to typescript.
// eslint-disable-next-line no-unused-vars
const pageHistoryEntryShape = {
  title: 'Page Title', // Title to display in breadcrumbs
  legacy: true, // Indicates if source page is struts or react,
  formData: {}, // Data from all form fields, visible and hidden
  fieldMap: {}, // Mapping from parent to child data, used when returning data
  id: 'xxx', // Unique id to identify page easily
  parentId: 'xx' // Id to parent page
};

const pageHistorySlice = RTK.createSlice({
  name: 'pageHistory',
  initialState: { history: [] },
  reducers: {
    pushHistory: (state, action) => {
      state.history.push({
        title: action.payload.title,
        legacy: true,
        formData: action.payload.formData,
        fieldMap: action.payload.fieldMap,
        id: action.payload.id,
        parentId: action.payload.parentId
      });
    },
    popHistory: (state) => {
      state.history.pop();
    }
  }
});

const rootReducer = RTK.combineReducers({
  userPreferences: dummySlice.reducer,
  [dummySlice.name]: dummySlice.reducer,
  [pageHistorySlice.name]: pageHistorySlice.reducer
});

document.addEventListener('DOMContentLoaded', () => {
  const store = RTK.configureStore({
    reducer: rootReducer,
    middleware: (getDefaultMiddleware) =>
      getDefaultMiddleware().concat(browserStorageListener.middleware),
    preloadedState: loadStateFromStorage() || {}
  });
  window.ReduxShim = { store, pageHistory: { actions: pageHistorySlice.actions } };
});
