/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import axios from 'axios';
import { config } from 'dotenv';
import { stringify } from 'qs';

// When running locally, it expects a .env file for environment variables
config({ path: './node-scripts/.env' });

const URLS = {
  SEARCH: 'rest/api/3/search',
  EDIT: 'rest/api/3/issue',
  BULK: 'rest/api/3/issue/bulk'
};

const JIRA_IDS = {
  ISSUE_TYPE: '4', // Improvement
  PARENT: '71485', // Security
  PROJECT: '11602', // FINP
  PRIORITY: '10003', // Normal,
  IMPACTED_MODULES: '10614' // System
};

const CUSTOM_FIELD_MAP = {
  ACCEPTANCE_CRITERIA: 'customfield_11400',
  TESTING_STEPS: 'customfield_14800',
  IMPACTED_MODULES: 'customfield_11401'
};

const SEVERITIES = ['low', 'medium', 'high', 'critical'];

const SUMMARY = '[Dependabot ([[ecosystem]])] [[[packageName]]] [[severity]]';

const BASE_SEARCH_JQL =
  'project = "FINP" AND status != "Closed" AND summary ~ "\\\\[Dependabot ([[ecosystem]])\\\\]" AND summary ~ "\\\\[[[packageName]]\\\\]"';

const FIELD_LIMITER =
  '&fields=issuetype,project,customfield_11400,priority,summary,parent,fixVersions,description';

// This will contain a list of packages that we are unable to upgrade at this time
const skipCheck = [];

// This is a list of packages that need to be split when searching, as for whatever
// reason Jira doesn't return search results when sent intact.
const tokenize = ['regenerator-runtime'];

const generateMarkupBlock = (plainText, list) => {
  if (list) {
    return {
      version: 1,
      type: 'doc',
      content: [
        {
          type: 'paragraph',
          content: [
            {
              type: 'text',
              text: plainText
            }
          ]
        },
        {
          type: 'bulletList',
          content: list.map((item) => {
            return {
              type: 'listItem',
              content: [
                {
                  type: 'paragraph',
                  content: [
                    {
                      type: 'text',
                      text: item,
                      marks: [
                        {
                          type: 'link',
                          attrs: {
                            href: item
                          }
                        }
                      ]
                    }
                  ]
                }
              ]
            };
          })
        }
      ]
    };
  }
  return {
    version: 1,
    type: 'doc',
    content: [
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: plainText
          }
        ]
      }
    ]
  };
};

const generateAcceptanceCriteria = (packageName) => {
  return generateMarkupBlock(
    `Package ${packageName} has been updated to resolve the security alerts.`
  );
};

const generateDescription = (packageName, issueUrls) => {
  return generateMarkupBlock(
    `The following security alerts are related to ${packageName}:`,
    issueUrls
  );
};

const generateTestingSteps = () => {
  return {
    version: 1,
    type: 'doc',
    content: [
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to the following lookups and ensure they work as expected:'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Organization Options'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Asset'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Object Code'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Labor Benefits Calculation '
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Dunning Letter Template'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to Institution Configuration and do a few edits to verify the form works as expected.'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to Customer Invoice and enter a description along with the following details:'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Customer: ABB2'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Chart: BL'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Account: 1031400'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Object: 5000'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Item Quantity: 1'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Invoice Item UOM Code: EA'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Invoice Item Unit Price: 300'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click Add icon'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click Submit'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to Cash Control and enter a description along with the following details:'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Medium Code: Check'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Customer: ABB2'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Date: today’s date'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Amount: 150'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click Add icon'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click Submit'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click on the APPLICATION DOC # link (will appear blue on Cash Control doc) to open the Payment Application.'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Enter 150 in the Apply Amount field in the APPLY TO INVOICE DETAIL section and click the Apply button'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Click Blanket Approve'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Login as wwren and approve the Cash Control document in the Action List.'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to the Doc Search and search on today’s date. '
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Open the Payment Application document and click the Adjust button. '
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Update the Item Amount Applied to 100.  In the Unapplied section, add Customer Number ABB2 and enter 50 as the Unapplied Amount. Click Submit.'
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to Batch Schedule and run nightly jobs '
          }
        ]
      },
      {
        type: 'paragraph',
        content: [
          {
            type: 'text',
            text: 'Jump to Batch File and open the scrubberJob.txt file and verify the document opens successfully.'
          }
        ]
      }
    ]
  };
};

const getGithubHeaders = () => {
  return {
    Authorization: `Basic ${Buffer.from(process.env.GITHUB_API_TOKEN).toString(
      'base64'
    )}`,
    Accept: 'application/vnd.github+json'
  };
};

const getJiraHeaders = () => {
  return {
    Authorization: `Basic ${Buffer.from(process.env.JIRA_TOKEN).toString(
      'base64'
    )}`,
    Accept: 'application/json'
  };
};

const search = async (packageName, ecosystem) => {
  const url = `${process.env.JIRA_URL}${URLS.SEARCH}${stringify(
    {
      jql: BASE_SEARCH_JQL.replace(
        '[[packageName]]',
        tokenize.includes(packageName)
          ? packageName.split('-').join('" AND summary ~ ')
          : packageName
      ).replace('[[ecosystem]]', ecosystem)
    },
    { addQueryPrefix: true }
  )}${FIELD_LIMITER}`;
  try {
    const { data } = await axios.get(url, { headers: getJiraHeaders() });
    if (data.total > 0) {
      return data.issues[0];
    }
  } catch (error) {
    if (error.response.data) {
      console.dir(error.response.data, { depth: null });
    }
    process.exitCode = 1;
  }
};

const generateCreateBody = (packageName, alertInfo) => {
  const body = {
    fields: {
      summary: SUMMARY.replace('[[packageName]]', packageName)
        .replace('[[severity]]', alertInfo.severity.toUpperCase())
        .replace('[[ecosystem]]', alertInfo.ecosystem),
      issuetype: {
        id: JIRA_IDS.ISSUE_TYPE
      },
      project: {
        id: JIRA_IDS.PROJECT
      },
      description: generateDescription(packageName, alertInfo.issueUrls),
      priority: {
        id: JIRA_IDS.PRIORITY
      },
      parent: {
        id: JIRA_IDS.PARENT
      },
      [CUSTOM_FIELD_MAP.ACCEPTANCE_CRITERIA]:
        generateAcceptanceCriteria(packageName),
      [CUSTOM_FIELD_MAP.IMPACTED_MODULES]: [{ id: JIRA_IDS.IMPACTED_MODULES }]
    }
  };
  if (alertInfo.ecosystem === 'frontend') {
    body.fields[CUSTOM_FIELD_MAP.TESTING_STEPS] = generateTestingSteps();
  }
  return body;
};

const generateUpdateBody = (
  issue,
  packageName,
  issueUrls,
  severity,
  ecosystem
) => {
  let updateSummary = false;
  let updateDescription = false;

  let body = null;

  if (issue.fields.summary.indexOf(severity.toUpperCase()) === -1) {
    updateSummary = true;
  }

  if (!issue.fields.description) {
    updateDescription = true;
  } else {
    issue.fields.description.content.forEach((content) => {
      if (updateDescription || content.type !== 'bulletList') {
        return;
      }
      content.content.forEach((listContent) => {
        if (!issueUrls.includes(listContent.content[0].content[0].text)) {
          updateDescription = true;
        }
      });
    });

    if (!updateDescription) {
      issue.fields.description.content.forEach((content) => {
        if (updateDescription || content.type !== 'bulletList') {
          return;
        }
        content.content.forEach((listContent) => {
          if (!issueUrls.includes(listContent.content[0].content[0].text)) {
            updateDescription = true;
          }
        });
      });
    }
  }

  if (updateSummary) {
    body = {
      ...(body ?? {}),
      summary: SUMMARY.replace('[[packageName]]', packageName)
        .replace('[[severity]]', severity.toUpperCase())
        .replace('[[ecosystem]]', ecosystem)
    };
  }

  if (updateDescription) {
    body = {
      ...(body ?? {}),
      description: generateDescription(packageName, issueUrls)
    };
  }

  return body;
};

const bulkCreate = async (alertMap) => {
  const url = `${process.env.JIRA_URL}${URLS.BULK}`;
  const headers = getJiraHeaders();
  const bulk = [];
  for (const [key, value] of alertMap) {
    if (value.issue) {
      continue;
    }
    bulk.push(generateCreateBody(key, value));
  }
  let iteration = 0;
  const take = 50; // Maximum number of issues per bulk command
  while (iteration * take < bulk.length) {
    const body = {
      issueUpdates: bulk.slice(iteration * take, take)
    };
    iteration++;
    try {
      await axios.post(url, body, { headers });
    } catch (error) {
      console.dir(error.response.data, { depth: null });
      process.exitCode = 1;
    }
  }
};

const updateExisting = async (alertMap) => {
  const headers = getJiraHeaders();
  for (const [key, value] of alertMap) {
    if (!value.issue) {
      continue;
    }
    const body = generateUpdateBody(
      value.issue,
      key,
      value.issueUrls,
      value.severity
    );

    if (body === null) {
      continue;
    }

    const url = `${process.env.JIRA_URL}${URLS.EDIT}/${value.issue.id}`;
    try {
      await axios.put(url, { fields: body }, { headers });
    } catch (error) {
      console.dir(error.response.data.errors, { depth: null });
      process.exitCode = 1;
    }
  }
};

const getSeverity = (currentSeverity, incomingSeverity) => {
  if (!currentSeverity || currentSeverity === incomingSeverity) {
    return incomingSeverity;
  }
  const currentSeverityIndex = SEVERITIES.indexOf(currentSeverity);
  const incomingSeverityIndex = SEVERITIES.indexOf(incomingSeverity);
  if (currentSeverityIndex < incomingSeverityIndex) {
    return incomingSeverity;
  }
  return currentSeverity;
};

const processAlerts = async (alerts) => {
  const alertMap = new Map();
  for (const alert of alerts) {
    const name = alert.security_vulnerability.package.name;
    const severity = alert.security_vulnerability.severity;
    if (skipCheck.includes(name)) {
      return;
    }
    const alertRecord = alertMap.get(name) ?? { issueUrls: [] };
    alertRecord.severity = getSeverity(alertRecord.severity, severity);
    alertRecord.issueUrls.push(alert.html_url);
    alertRecord.ecosystem =
      alert.security_vulnerability.package.ecosystem === 'maven'
        ? 'backend'
        : 'frontend';
    alertMap.set(name, alertRecord);
  }

  for (const [key, value] of alertMap) {
    value.issue = await search(key, value.ecosystem);
  }
  await bulkCreate(alertMap);
  await updateExisting(alertMap);
};

const getAlerts = async () => {
  try {
    const { data } = await axios.get(
      `${process.env.GITHUB_DEPENDABOT_URL}?state=open`,
      { headers: getGithubHeaders() }
    );
    processAlerts(data);
  } catch (error) {
    if (error?.response?.data) {
      console.dir(error.response.data, { depth: null });
    } else {
      console.dir(error, { depth: null });
    }
    process.exitCode = 1;
  }
};

await getAlerts();
