/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// This file is used to shim the redux store between struts and react

const RTK = window.RTK;
const Redux = window.Redux;

const loadStateFromStorage = () => {
  try {
    const serialState = sessionStorage.getItem('appState');
    if (serialState === null) {
      return undefined;
    }
    const stateWrapper = JSON.parse(serialState);
    const anHourAgo = Date.now() - (1000 * 60 * 60);
    if (stateWrapper.lastUpdateDate < anHourAgo) {
      return undefined;
    }
    return stateWrapper.state;
  } catch (err) {
    return undefined;
  }
};

const saveStateToStorage = (state) => {
  try {
    const serialState = JSON.stringify({ lastUpdateDate: Date.now(), state });
    sessionStorage.setItem('appState', serialState);
  } catch (err) {
    console.log(err);
  }
};

const browserStorageListener = RTK.createListenerMiddleware();

browserStorageListener.startListening({
  predicate: () => true,
  effect: (_action, listenerApi) => {
    saveStateToStorage(listenerApi.getState());
  }
});

// dummy slice to use for slices only used by the react app piece
const dummySlice = RTK.createSlice({
  name: 'dummy',
  initialState: null,
  reducers: {
    noop: () => {}
  }
});

const rootReducer = RTK.combineReducers({
  userPreferences: dummySlice.reducer,
  [dummySlice.name]: dummySlice.reducer
});

document.addEventListener('DOMContentLoaded', () => {
  const store = RTK.configureStore({
    reducer: rootReducer,
    middleware: (getDefaultMiddleware) =>
      getDefaultMiddleware().concat(browserStorageListener.middleware),
    preloadedState: loadStateFromStorage() || {}
  });
  // TODO: Once real reducers are created, we can add their actions to this object to have them available to the app
  window.ReduxShim = { store };
});
