package co.kuali.coeus.s3.api;

import co.kuali.coeus.s3.impl.S3FileServiceimpl;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.S3ClientBuilder;

/**
 * A factory bean that creates instances of the S3FileService using an amazon client.
 */
public class S3FileServiceFactoryBean implements FactoryBean<S3FileService>, InitializingBean {

    private String bucketName;
    private boolean replicationBucket = false;
    private String region;
    private String accessKey;
    private String secretKey;
    private boolean encryptionEnabled = false;
    private String encryptionKey;
    private String confidentialDataTagName;
    private boolean confidentialData = true;

    private boolean singleton = true;

    @Override
    public S3FileService getObject() {

        final boolean awsCredsProvided = accessKey != null && !accessKey.trim().isEmpty() && secretKey != null && !secretKey.trim().isEmpty();
        final S3ClientBuilder builder = S3Client.builder().region(Region.of(getRegion()));
        final S3Client s3 = (awsCredsProvided ? builder.credentialsProvider(StaticCredentialsProvider.create(AwsBasicCredentials.create(accessKey, secretKey))) : builder).build();

        final S3FileServiceimpl s3FileService = new S3FileServiceimpl();
        s3FileService.setAmazonS3(s3);
        s3FileService.setBucketName(bucketName);
        if (replicationBucket) {
            s3FileService.setReplicationBucketName(toReplicationBucket(bucketName));
        }
        s3FileService.setEncryptionEnabled(encryptionEnabled);
        s3FileService.setEncryptionKey(encryptionKey);
        s3FileService.setConfidentialDataTagName(confidentialDataTagName);
        s3FileService.setConfidentialData(confidentialData);

        return s3FileService;
    }

    @Override
    public Class<?> getObjectType() {
        return S3FileService.class;
    }

    @Override
    public boolean isSingleton() {
        return singleton;
    }

    public void setSingleton(boolean singleton) {
        this.singleton = singleton;
    }

    public String getBucketName() {
        return bucketName;
    }

    public void setBucketName(String bucketName) {
        this.bucketName = bucketName;
    }

    public String getAccessKey() {
        return accessKey;
    }

    public void setAccessKey(String accessKey) {
        this.accessKey = accessKey;
    }

    public String getSecretKey() {
        return secretKey;
    }

    public void setSecretKey(String secretKey) {
        this.secretKey = secretKey;
    }

    public String getRegion() {
        return region;
    }

    public void setRegion(String region) {
        this.region = region;
    }

    public boolean isReplicationBucket() {
        return replicationBucket;
    }

    public void setReplicationBucket(boolean replicationBucket) {
        this.replicationBucket = replicationBucket;
    }

    public boolean isEncryptionEnabled() {
        return encryptionEnabled;
    }

    public void setEncryptionEnabled(boolean encryptionEnabled) {
        this.encryptionEnabled = encryptionEnabled;
    }

    public String getEncryptionKey() {
        return encryptionKey;
    }

    /**
     * Sets a master KMS key to be used to encrypt auto-provisioned S3 buckets
     *
     * @param encryptionKey - A master KMS key ID or ARN
     */
    public void setEncryptionKey(String encryptionKey) {
        this.encryptionKey = encryptionKey;
    }

    public String getConfidentialDataTagName() {
        return confidentialDataTagName;
    }

    public void setConfidentialDataTagName(String confidentialDataTagName) {
        this.confidentialDataTagName = confidentialDataTagName;
    }

    public boolean isConfidentialData() {
        return confidentialData;
    }

    public void setConfidentialData(boolean confidentialData) {
        this.confidentialData = confidentialData;
    }

    private String toReplicationBucket(String bucketName) {
        return bucketName + "-r";
    }

    @Override
    public void afterPropertiesSet() {
        if (StringUtils.isBlank(bucketName)) {
            throw new IllegalStateException("bucket name is blank");
        }

        if (Region.of(region) == null) {
            throw new IllegalStateException("Invalid region " + region);
        }
    }
}
