package co.kuali.coeus.s3.conv;


import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import java.util.List;
import java.util.Properties;
import java.util.concurrent.ConcurrentMap;


public class Main {

    private static final Logger LOG = LogManager.getLogger(Main.class);
    private static final String DB_CONVERT_PREFIX = "db.convert.";

    public static void main(String[] args) {
        final ApplicationContext context = new ClassPathXmlApplicationContext("classpath:co/kuali/coeus/s3/conv/applicationContext.xml");

        final Properties properties = (Properties) context.getBean("configProperties");

        LOG.info("The conversion configuration is " + properties + ".");

        final ConcurrentMap<String, String> tableConfig = properties.entrySet().stream()
                .map(CollectionUtils::<String, String>unsafeCast)
                .filter(e -> e.getKey().startsWith(DB_CONVERT_PREFIX)).collect(CollectionUtils.entriesToConcurrentMap());

        final var concurrentProp =  properties.getProperty("process.concurrent");
        final boolean concurrent = concurrentProp != null && concurrentProp.equalsIgnoreCase("true");
        if (concurrent) {
            LOG.info("processing tables concurrently");
        } else {
            LOG.info("processing tables sequentially");
        }


        final String tablesToProcess = tableConfig.get("db.convert.tables");
        if (StringUtils.isNotBlank(tablesToProcess)) {
            final List<String> tables = CsvUtils.csvToList(tablesToProcess);
            (concurrent ? tables.parallelStream() : tables.stream()).forEach(table -> {
                final ConversionInfo info = new ConversionInfo();
                info.setTargetSchema(tableConfig.get(DB_CONVERT_PREFIX + table + ".schema"));
                info.setTable(table);
                info.setDataColumn(tableConfig.get(DB_CONVERT_PREFIX + table + ".data.column"));
                info.setPkColumns(CsvUtils.csvToList(tableConfig.get(DB_CONVERT_PREFIX + table + ".pk.columns")));
                info.setContentTypeColumn(tableConfig.get(DB_CONVERT_PREFIX + table + ".content.type.column"));
                info.setFileNameColumn(tableConfig.get(DB_CONVERT_PREFIX + table + ".file.name.column"));
                info.setAdditionalMetadataColumns(CsvUtils.csvToList(tableConfig.get(DB_CONVERT_PREFIX + table + ".addtional.metadata.columns")));
                info.setNullConverted(Boolean.parseBoolean(properties.getProperty("null.converted", "true")));
                final ConversionService conversionService = context.getBean(ConversionService.class);
                conversionService.convert(info);
            });
        } else {
            LOG.warn("There are no tables configured.  Nothing will happen.");
        }
        System.exit(0);
    }
}
