package com.rsmart.kuali.coeus.hr.service.impl;

import com.rsmart.kuali.coeus.hr.rest.model.AddressCollection;
import com.rsmart.kuali.coeus.hr.rest.model.Affiliation;
import com.rsmart.kuali.coeus.hr.rest.model.AffiliationCollection;
import com.rsmart.kuali.coeus.hr.rest.model.AppointmentCollection;
import com.rsmart.kuali.coeus.hr.rest.model.DegreeCollection;
import com.rsmart.kuali.coeus.hr.rest.model.EmailCollection;
import com.rsmart.kuali.coeus.hr.rest.model.Employment;
import com.rsmart.kuali.coeus.hr.rest.model.HRImport;
import com.rsmart.kuali.coeus.hr.rest.model.HRImportRecord;
import com.rsmart.kuali.coeus.hr.rest.model.KCExtendedAttributes;
import com.rsmart.kuali.coeus.hr.rest.model.NameCollection;
import com.rsmart.kuali.coeus.hr.rest.model.PhoneCollection;
import com.rsmart.kuali.coeus.hr.service.HRImportService;
import com.rsmart.kuali.coeus.hr.service.ImportError;
import com.rsmart.kuali.coeus.hr.service.ImportStatusService;
import com.rsmart.kuali.coeus.hr.service.adapter.PersistableBoMergeAdapter;
import com.rsmart.kuali.coeus.hr.service.adapter.impl.PersonAppointmentBoAdapter;
import com.rsmart.kuali.coeus.hr.service.adapter.impl.EntityAddressBoAdapter;
import com.rsmart.kuali.coeus.hr.service.adapter.impl.EntityAffiliationBoAdapter;
import com.rsmart.kuali.coeus.hr.service.adapter.impl.EntityEmailBoAdapter;
import com.rsmart.kuali.coeus.hr.service.adapter.impl.EntityEmploymentBoAdapter;
import com.rsmart.kuali.coeus.hr.service.adapter.impl.EntityNameBoAdapter;
import com.rsmart.kuali.coeus.hr.service.adapter.impl.EntityPhoneBoAdapter;
import com.rsmart.kuali.coeus.hr.service.adapter.impl.PersonDegreeBoAdapter;

import org.kuali.coeus.common.framework.person.attr.CitizenshipType;
import org.kuali.coeus.sys.framework.auth.CoreUsersPushStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.kuali.coeus.common.framework.person.KcPerson;
import org.kuali.coeus.common.framework.person.attr.KcPersonExtendedAttributes;
import org.kuali.coeus.common.framework.unit.UnitService;
import org.kuali.coeus.sys.framework.auth.AuthServicePushService;
import org.kuali.coeus.sys.framework.auth.AuthServiceUserLoginFilter;
import org.kuali.coeus.sys.framework.service.KcServiceLocator;
import org.kuali.rice.core.api.cache.CacheManagerRegistry;
import org.kuali.rice.core.api.mo.common.Defaultable;
import org.kuali.rice.core.impl.services.CoreImplServiceLocator;
import org.kuali.rice.kim.api.KimApiConstants;
import org.kuali.rice.kim.api.identity.IdentityService;
import org.kuali.rice.kim.api.identity.principal.Principal;
import org.kuali.rice.kim.impl.identity.address.EntityAddressBo;
import org.kuali.rice.kim.impl.identity.affiliation.EntityAffiliationBo;
import org.kuali.rice.kim.impl.identity.email.EntityEmailBo;
import org.kuali.rice.kim.impl.identity.employment.EntityEmploymentBo;
import org.kuali.rice.kim.impl.identity.entity.EntityBo;
import org.kuali.rice.kim.impl.identity.phone.EntityPhoneBo;
import org.kuali.rice.kim.impl.identity.principal.PrincipalBo;
import org.kuali.rice.kim.impl.identity.type.EntityTypeContactInfoBo;
import org.kuali.rice.krad.service.BusinessObjectService;
import org.kuali.rice.krad.service.LegacyDataAdapter;
import org.kuali.rice.krad.util.GlobalVariables;
import org.springframework.cache.CacheManager;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.*;
import java.util.stream.Collectors;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;

/**
 * This implements the core logic for performing an import of an HR file. This class is dependent
 * upon several services from KIM and KRA:
 * {@link org.kuali.rice.kim.api.identity.IdentityService IdentityService}, 
 * {@link org.kuali.rice.krad.service.LegacyDataAdapter legacyDataAdapter},
 * {@link org.kuali.coeus.common.framework.unit.UnitService}, and
 * {@link org.kuali.rice.core.api.cache.CacheManagerRegistry CacheManagerRegistry}
 * 
 * Processing starts in the {@link #startImport startImport} method.
 * 
 * @author duffy
 *
 */
public class HRImportServiceImpl implements HRImportService {
  private static final Logger LOG = LoggerFactory.getLogger(HRImportServiceImpl.class);
  
  private static final String PERSON = "PERSON";
  
  private static final HashSet<String> runningImports = new HashSet<String>();

  private Validator                   validator;
  
  //dependencies to be injected or looked up from KIM/KC
  private IdentityService             identityService;
  private LegacyDataAdapter           legacyDataAdapter;
  private CacheManagerRegistry        cacheManagerRegistry;
  private AuthServicePushService      authServicePushService;
  
  private ImportStatusService         statusService = null;
  
  private EntityAddressBoAdapter      addressAdapter = new EntityAddressBoAdapter();
  private EntityAffiliationBoAdapter  affiliationAdapter = new EntityAffiliationBoAdapter();
  private EntityEmailBoAdapter        emailAdapter = new EntityEmailBoAdapter();
  private EntityEmploymentBoAdapter   employmentAdapter = new EntityEmploymentBoAdapter();
  private EntityNameBoAdapter         nameAdapter = new EntityNameBoAdapter();
  private EntityPhoneBoAdapter        phoneAdapter = new EntityPhoneBoAdapter();
  private PersonAppointmentBoAdapter  appointmentAdapter = new PersonAppointmentBoAdapter();
  private PersonDegreeBoAdapter       degreeAdapter = new PersonDegreeBoAdapter();

  public HRImportServiceImpl() {
    ValidatorFactory factory = Validation.buildDefaultValidatorFactory();
    validator = factory.getValidator();
  }
  
  public void setImportStatusService(final ImportStatusService svc) {
    statusService = svc;
  }
  
  private static boolean isRunning(final String importId) {
    synchronized (runningImports) {
      return runningImports.contains(importId);
    }
  }
  
  private static void addRunningImport(final String importId) {
    synchronized (runningImports) {
      runningImports.add(importId);
    }
  }
  
  private static void stopRunningImport(final String importId) {
    synchronized (runningImports) {
      runningImports.remove(importId);
    }
  }

  // Duffy - not sure what purpose this might have served or should serve.
  @SuppressWarnings("unused")
  private HashSet<String> getAllIDs() {
    final Collection<KcPersonExtendedAttributes> allAttribs = legacyDataAdapter.findAll(KcPersonExtendedAttributes.class);
    final HashSet<String> ids = new HashSet<String>();
    
    for (KcPersonExtendedAttributes attribs : allAttribs) {
      final String id = attribs.getPersonId();
      
      if (ids.contains(id)) {
        // this a duplicate - shouldn't happen, but this is not the place to deal with it
        // log the issue and skip the duplicate
        LOG.warn("duplicate ID '" + id + "' found when getting all IDs");
      } else {
        ids.add(id);
      }
    }
    
    return ids;
  }
  
  protected void deactivatePeople(final List<String> ids) {
    LOG.debug("deactivatePeople(final List<String> " + ids + ")");
    for (final String id : ids) {
      if (id != null && !"".equals(id.trim())) {
         PrincipalBo principal = legacyDataAdapter.findBySinglePrimaryKey(PrincipalBo.class, id);
        if (principal != null) {
          LOG.debug("deactivating principal: " + id);
          principal.setActive(false);
            principal = legacyDataAdapter.save(principal);
        } else {
          LOG.warn("unable to find principal to deactivate with id: " + id);
        }
      } else {
        final Error e = new Error("Ignoring blank principalId: '" + id + "'");
        LOG.warn(e.getLocalizedMessage(), e);
      }
    }
  }
  
  //TODO: figure out how to evict only those EntityBO objects affected by the update
  /**
   * Addresses a cache eviction bug. OJB implementation does not evict EntityBO objects from
   * ehcache during a businessObjectService.save() operation. Thus stale EntityBO objects
   * hang around in memory and are used by the running KC system. This method explicitly
   * flushes the cache containing EntityBO objects.
   * 
   * If this issue is resolved within KIM it should be possible to remove the dependency of
   * this class upon CacheManagerRegistry.
   */
  private final void flushCache() {
    LOG.debug("flushCache()");
    
    if (cacheManagerRegistry == null) {
      cacheManagerRegistry = CoreImplServiceLocator.getCacheManagerRegistry();
    }
    
    final CacheManager manager = cacheManagerRegistry.getCacheManager(KimApiConstants.Cache.KIM_DISTRIBUTED_CACHE_MANAGER);
    for (String cacheName : manager.getCacheNames()) {
    	manager.getCache(cacheName).clear();
    }
    LOG.debug("finished flushing cache");
  }
  
  /**
   * Convenience method which gets the records from the incoming import.
   * This method will do an error check to ensure that the number of items in the import
   * matches the reported record count declared in the XML header.
   * 
   * @param toImport
   * @return
   */
  private final Iterator<HRImportRecord> getRecords(final HRImport toImport) {
    if (toImport == null) {
      throw new IllegalArgumentException("HRImport toImport == null!");
    }
    if (toImport.getRecords() == null) {
      throw new IllegalStateException("toImport.getRecords() == null!");
    }
    return toImport.getRecords().iterator();
  }
  
  /**
   * This duplicates the logic of the static method {@link org.kuali.coeus.common.framework.person.KcPerson#fromPersonId KcPerson.fromPersonId()}.
   * It has been duplicated intentionally to expose identityService and businessObjectService
   * to me mocked. Since KcPerson.fromPersonId() is static it cannot be mocked
   * with Mockito and therefore its return object cannot be wired with the mock services.
   */
  private final KcPerson getKcPerson(final String principalId) {
    final KcPerson kcPerson = new KcPerson();
    kcPerson.setIdentityService(identityService);

    kcPerson.setBusinessObjectService(KcServiceLocator.getService(BusinessObjectService.class));
    kcPerson.setPersonId(principalId);
    kcPerson.refresh();
    
    return kcPerson;
  }
  
  private final KcPersonExtendedAttributes getKcPersonExtendedAttributes(final KcPerson person) {
    KcPersonExtendedAttributes attribs = person.getExtendedAttributes();
    
    if (attribs == null) {
      attribs = new KcPersonExtendedAttributes();
      attribs.setPersonId(person.getPersonId());
    }
    
    return attribs;
  }
  
  /**
   * In the event of an Exception while processing, this method pretty prints the Exception's
   * details to the error log for later analysis.
   * 
   * @param recordIndex
   * @param e
   */
  private final void logErrorForRecord (final int recordIndex, final Exception e) {
    final StringWriter strWriter = new StringWriter();
    final PrintWriter errorWriter = new PrintWriter(strWriter);
    errorWriter.append("import failed for record ").append(Integer.toString(recordIndex))
           .append(": ").append(e.getMessage()).append('\n');
    e.printStackTrace(errorWriter);
    errorWriter.flush();
    LOG.error(strWriter.toString());
  }
  
  protected Set<ConstraintViolation<HRImportRecord>> validateRecord (final HRImportRecord record) {
    return validator.validate(record);
  }
  
  protected void handleRecord (final HRImportRecord record)
      throws Exception {

      Set<ConstraintViolation<HRImportRecord>> violations = validateRecord(record);
      if (violations != null && !violations.isEmpty()) {
          throw new IllegalArgumentException (violations.toString());
      }

    final String principalId = record.getPrincipalId();
    
    EntityBo entity = EntityBo.from(identityService.getEntityByPrincipalId(principalId));
    if (entity != null) {
      LOG.debug("updating existing entity: " + principalId);
    } else {
      // check for ID collisions
      final String suppliedEntityId = record.getEntityId();
      final HashMap<String, String> fields = new HashMap<String, String>();
      fields.put("id", suppliedEntityId);
      if(suppliedEntityId != null && legacyDataAdapter.findBySinglePrimaryKey(EntityBo.class, suppliedEntityId) != null) {
        LOG.error("entity ID collision: " + suppliedEntityId);
        throw new IllegalArgumentException ("A person already exists with entity ID: " + suppliedEntityId);
      }
      
      LOG.debug("creating new entity: " + principalId);

      entity = newEntityBo(record);
    }          

    //TODO: It would be super-great to have updateEntityBo and updateExtendedAttributes in a single transaction
    //      that rolls back if there is an error
    //TODO: updateEntityBo now returns principal because of KIM caching issues. identityService.getPrincipal...
    //      calls are not returning the new Principal even though it has been successfully inserted in the DB!!!
    final PrincipalBo principal = updateEntityBo(entity, record);

    // if this is an update Extended attributes are not required if they are not being changed.
    //
    // if this is a new record this will not be null because the validate step above would
    //  have caused a failure in that case
    if (record.getKcExtendedAttributes() != null) {
      updateExtendedAttributes (principal.getPrincipalId(), record);
    }
    
  }
  
  /**
   * This is the workhorse of HRImportServiceImpl. It takes an HRImport and walks through
   * the records it contains. For each record the contained dependent business objects are 
   * merged with the list of current business objects. Each set of dependent objects is handled
   * in the following manner:
   * 
   * 1) if the list of dependent objects does not exist in the import, no change is made.
   * 2) if the list exists
   *   2.1) any new items in the list are added
   *   2.2) any old items currently stored for the user which are not in the list are deleted
   *   2.3) any new items in the list that are equivalent to items already attached to the user
   *        are ignored.
   *        
   * The net effect is that the state of incoming collections of business objects will replace
   * the state stored for the user. If no change is desired for a particular set of business
   * objects (eg. no changes have occurred for a user's name or email records) it can be omitted entirely
   * from the import. This is *different* than including an empty element in the import.
   * An empty element will cause all of that type of business object to be deleted for the user.
   */
  public void startImport(final String importId, final HRImport toImport) {
      LOG.debug("starting import " + importId);
    // Cache coherency seems completely broken in Rice.
    // Ensure we are reading the latest information from KIM.
    // https://github.com/rSmart/issues/issues/331
    flushCache();

    try {
      Map<String, Integer> principalToRecordNumber = new HashMap<>();
      addRunningImport(importId);
      
      //records from the HRImport
      final Iterator<HRImportRecord> records = getRecords(toImport);
      final int numRecords = toImport.getRecordCount();
      final HashSet<String> processedPrincipals = new HashSet<String> (numRecords);
      
      // loop through records
      int i = 0;
      while (i < numRecords && records.hasNext()) {
        final int recordNumber = i + 1;
        if (!isRunning(importId)) {
            LOG.debug("import aborted. stopping at record " + (recordNumber));
          break;
        }
        String principalId = null;
        try {
          final HRImportRecord record = records.next();
          principalId = record.getPrincipalId();
          principalToRecordNumber.put(principalId, recordNumber);

          // error on duplicate records in import
          // this is critical since cache is flushed only once (after all records)
          if (processedPrincipals.contains(principalId)) {
            throw new IllegalArgumentException ("Duplicate records for the same principalId not allowed in a single import");
          }
          
          handleRecord(record);

          if (record.isActive()) {
            statusService.recordProcessed(importId, principalId);
          } else {
            statusService.recordInactivated(importId, principalId);
          }
        } catch (Exception e) {
          // log the spot where the exception occurred, then add it to the exception collection and move on
          statusService.recordError(importId, new ImportError(recordNumber, principalId, e));
          logErrorForRecord(recordNumber, e);
        } finally {
          // track the ID of the import so we can spot duplicates
          if (principalId != null) {
            processedPrincipals.add (principalId);
          }
          i = recordNumber;
        }
      }

      if (records.hasNext()) {
        throw new IllegalArgumentException ("recordCount " + numRecords + " is less than number of records");
      }
      if (i < numRecords) {
        throw new IllegalArgumentException ("recordCount " + numRecords + " is more than number of records");
      }

        LOG.debug("finished processing records");
      // if the import has been aborted, do not try to inactivate records
      if (isRunning(importId)) {
          LOG.debug("deactivating people who were missing from import");
        deactivatePeople(statusService.getActivePrincipalNamesMissingFromImport(importId));
          LOG.debug("finished deactivating people");
      }
      flushCache();
      final String authToken = AuthServiceUserLoginFilter.getAuthToken(GlobalVariables.getUserSession());
      if (authToken != null && !authToken.trim().equals("")) {
    	  CoreUsersPushStatus pushStatus = authServicePushService.pushAllUsers();
    	  pushStatus.getErrors().entrySet().stream()
    	  	.forEach(entry -> 
    	  		statusService.recordError(importId, 
    	  			new ImportError(principalToRecordNumber.get(entry.getKey()).intValue(),
    	  				entry.getKey(), new Exception(entry.getValue()))));
      }
    } finally {
      stopRunningImport(importId);
    }
      LOG.debug("HRImportServiceImpl.startImport(...) finished");
  }
  
  /**
   * This converts an incoming set of JAXB objects into their corresponding KIM entity objects.
   * The list is returned sorted so that it is possible to determine which incoming objects are
   * new versus updates, versus when an existing object has been omitted in order to delete it.
   * 
   * @param entityId
   * @param toImport
   * @param adapter
   * @return
   */
  protected final <T extends Object, Z> List<T> adaptAndSortList 
      (final String entityId, final List<Z> toImport, final PersistableBoMergeAdapter<T,Z> adapter) {
    final ArrayList<T> adaptedList = new ArrayList<T>();
    boolean defaultSet = false;
    
    int index = 0;
    for (final Z name : toImport) {
      final T newBo = adapter.setFields(++index, adapter.newBO(entityId),name);
      
      // take this opportunity to enforce default value flag as unique within collection
      if (adapter.isBoDefaultable() && ((Defaultable)newBo).isDefaultValue()) {
        if (defaultSet) {
          throw new IllegalArgumentException ("Multiple records of type " 
                + newBo.getClass().getSimpleName() + " set as default value");
        } else {
          defaultSet = true;
        }
      }
      
      final int insertAt = Collections.binarySearch(adaptedList, newBo, adapter);
      
      if (insertAt >= 0) {
        throw new IllegalArgumentException ("Duplicate records of type " 
              + newBo.getClass().getSimpleName() + " encountered");
      } else {
        adaptedList.add(-(insertAt + 1), newBo);
      }
    }
    
    return adaptedList;
  }
  
  /**
   * This merges an incoming list of JAXB objects into a list of existing business objects
   * for a single Entity.
   * 
   * @param objsToMerge
   * @param existingBOs
   * @param adapter
   * @param entityId
   * @return
   */
  protected <T extends Object, Z> 
    boolean mergeImportedBOs (final List<Z> objsToMerge, final List<T> existingBOs,
                              final PersistableBoMergeAdapter<T, Z> adapter, final String entityId) {
    
    // create our own copy of the current BOs to sort and modify without
    // fear of EntityBO innards
    final ArrayList<T> currBOs = existingBOs != null ? new ArrayList<T> (existingBOs) : new ArrayList<T>();

    // do we have objects to merge? if not why bother?
    if (objsToMerge != null) {

      // adapt the incoming List of objects into their BO type and sort the results
      final List<T> importBOs = this.adaptAndSortList(entityId, objsToMerge, adapter);
      
      // sort the existing collection
      Collections.sort(currBOs, adapter);

      // track whether the collection has been modified and needs to be refreshed
      boolean collectionModified = false;
      final LinkedList<T> oldToDelete = new LinkedList<T>();
      final LinkedList<T> newToAdd = new LinkedList<T>();
      
      int lowerBound = 0;
      
      for (T importBO : importBOs) {
        boolean importHandled = false;
        // use this loop to check if import is a duplicate, or if some old BOs need deleting
        for (int i = lowerBound; i < currBOs.size() && !importHandled; i++) {
          final T oldBO = currBOs.get(i);
          final int comp = adapter.compare(importBO, oldBO);
          if (comp < 0) {
            // importBO comes before any of the existing BOs, add it
            newToAdd.add(importBO);
            importHandled = true;
          } else if (comp == 0) {
            // importBO already exists, skip it and currBO
            importHandled = true;
            lowerBound++;
          } else {
            // oldBO comes before any of the imports, thus it is not in the import - delete it
            oldToDelete.add(oldBO);
            lowerBound++;
          }
        }
        // if we've exhausted the list of existing BOs without a match or a place to insert,
        // then we insert at the end.
        if (!importHandled) {
          newToAdd.add(importBO);
        }
      }
      
      // if we've exhausted the imports and there are still left over existing BOs
      // then we need to delete the remaining existing imports.
      for (int i = lowerBound; i < currBOs.size(); i++) {
        oldToDelete.add(currBOs.get(i));
      }
      
      // all the adds and deletes have been queued up - run through them here
      
      for (T oldBO : oldToDelete) {
        adapter.delete(legacyDataAdapter, oldBO);
        collectionModified = true;
      }
      
      int index = 0;
      for (T newBO : newToAdd) {
        adapter.save(++index, legacyDataAdapter, newBO);
        collectionModified = true;
      }

      return collectionModified;
    }
    return false;
  }
  
  /**
   * This handles each of the KIM dependent entity business object collections connected to a single
   * Entity.
   * 
   * @param entity
   * @param record
   */
  protected PrincipalBo updateEntityBo(EntityBo entity, final HRImportRecord record) {
    final PrincipalBo principal = updatePrincipal(entity, record);

    final NameCollection nameColl = record.getNameCollection();
    if (nameColl != null) {
    	mergeImportedBOs (nameColl.getNames(), entity.getNames(), nameAdapter, entity.getId());
    }

    final AffiliationCollection affColl = record.getAffiliationCollection();
    if (affColl != null) {
    	mergeAffiliations(entity, affColl);
    }
    
    // Handle all the different contact business object types
      final EntityTypeContactInfoBo contactInfo = getEntityTypeContactInfoBo(entity);
 
    final AddressCollection addrColl = record.getAddressCollection();
    if (addrColl != null) {
    	mergeImportedBOs (addrColl.getAddresses(), contactInfo.getAddresses(), addressAdapter, entity.getId());
    }

    final PhoneCollection phoneColl = record.getPhoneCollection();
    if (phoneColl != null) {
    	mergeImportedBOs (phoneColl.getPhones(), contactInfo.getPhoneNumbers(), phoneAdapter, entity.getId());
    }
    
    final EmailCollection emailColl = record.getEmailCollection();
    if (emailColl != null) {
    	mergeImportedBOs (emailColl.getEmails(), contactInfo.getEmailAddresses(), emailAdapter, entity.getId());
    }
    
    return principal;
  }

  private void mergeAffiliations(EntityBo entity,
		final AffiliationCollection affColl) {
	Map<String, EntityAffiliationBo> matchedAffiliations = new HashMap<>();
	for (Affiliation affil : affColl.getAffiliations()) {
      List<Employment> employments = affil.getEmployments();
      EntityAffiliationBo newAffil = affiliationAdapter.setFields(affiliationAdapter.newBO(entity.getId()), affil);
      boolean exists = false;
      for (EntityAffiliationBo dbAffil : entity.getAffiliations()) {
        if (affiliationAdapter.compare(newAffil, dbAffil) == 0) {
          dbAffil.setActive(newAffil.isActive());
          dbAffil.setDefaultValue(newAffil.getDefaultValue());
          legacyDataAdapter.save(dbAffil);
          mergeAffiliationEmployment(entity.getId(), employments, dbAffil, entity.getEmploymentInformation());
          matchedAffiliations.put(dbAffil.getId(), dbAffil);
          exists = true;
          break;
        }
      }
      if (!exists) {
        entity.getAffiliations().add(newAffil);
        newAffil = legacyDataAdapter.save(newAffil);
        matchedAffiliations.put(newAffil.getId(), newAffil);
        for (Employment employment : employments ) {
          if (employment != null) {
            EntityEmploymentBo newEmployment = createNewEmploymentInfo(
                    entity.getId(), employment, newAffil, entity.getEmploymentInformation());
            legacyDataAdapter.save(newEmployment);
          }
        }
      }
	}
	for (Iterator<EntityAffiliationBo> iter = entity.getAffiliations().iterator(); iter.hasNext(); ) {
		EntityAffiliationBo dbAffil = iter.next();
		if (!matchedAffiliations.containsKey(dbAffil.getId())) {
			for (Iterator <EntityEmploymentBo> empIter = entity.getEmploymentInformation().iterator(); empIter.hasNext(); ) {
				EntityEmploymentBo employment = empIter.next();
				if (Objects.equals(employment.getEntityAffiliationId(), dbAffil.getId())) {
					empIter.remove();
					legacyDataAdapter.delete(employment);
				}
			}
			iter.remove();
			legacyDataAdapter.delete(dbAffil);
		}
	}
}

  private void mergeAffiliationEmployment(final String entityId,
		final List<Employment> employments, final EntityAffiliationBo dbAffil,
		final List<EntityEmploymentBo> empBos) {

    employments.forEach(employment -> employment.setEntityAffiliationId(dbAffil.getId()));
    mergeImportedBOs(employments, empBos.stream().filter(empBo -> Objects.equals(empBo.getEntityAffiliationId(), dbAffil.getId())).collect(Collectors.toList()), employmentAdapter, entityId);
  }

  private EntityEmploymentBo createNewEmploymentInfo(final String entityId,
		final Employment employment, final EntityAffiliationBo newAffil,
		final List<EntityEmploymentBo> empBos) {
	EntityEmploymentBo newEmployment = employmentAdapter.setFields(getMaxEmploymentRecordId(empBos)+1, employmentAdapter.newBO(entityId), employment);
	newEmployment.setActive(newAffil.isActive());
	newEmployment.setEntityAffiliationId(newAffil.getId());
	newEmployment.setEntityAffiliation(newAffil);
	return newEmployment;
}

  private int getMaxEmploymentRecordId(final List<EntityEmploymentBo> empBos) {
	return empBos.stream().map(EntityEmploymentBo::getEmploymentRecordId).mapToInt(Integer::parseInt).max().orElse(1);
}

/**
   * Removes a single person and his/her dependent entities
   */
  @Override
  public void deletePerson(final String entityId) {
    final EntityBo entity = legacyDataAdapter.findBySinglePrimaryKey(EntityBo.class, entityId);
    delete(entity);
  }
  
  /**
   * Deletes and entity and all its dependent entities.
   * 
   * @param entity
   */
  protected void delete(final EntityBo entity) {
      LOG.debug("Deleting Entity: " + entity);
    legacyDataAdapter.delete(entity.getPrincipals());
      legacyDataAdapter.delete(entity.getNames());
      legacyDataAdapter.delete(entity.getEmploymentInformation());
      legacyDataAdapter.delete(entity.getAffiliations());

    for (final EntityTypeContactInfoBo contactInfo : entity.getEntityTypeContactInfos()) {
        legacyDataAdapter.delete(contactInfo.getAddresses());
        legacyDataAdapter.delete(contactInfo.getPhoneNumbers());
        legacyDataAdapter.delete(contactInfo.getEmailAddresses());
      // contactInfo.refresh();
      // contactInfo.refreshNonUpdateableReferences();
    }
      legacyDataAdapter.delete(entity.getEntityTypeContactInfos());
    // IIUC this should cascade and delete degrees, appointments, etc.
    final KcPerson person = KcPerson.fromPersonId(entity.getId());
    final KcPersonExtendedAttributes attributes = person.getExtendedAttributes();
    if (attributes.getPersonId() != null) {
        legacyDataAdapter.delete(person.getExtendedAttributes());
    }
    // entity.refresh();
    // entity.refreshNonUpdateableReferences();
      legacyDataAdapter.delete(entity);
  }

  /**
   * Set the principal name and Id.
   * 
   * @param entity
   * @param record
   */
  protected PrincipalBo updatePrincipal(final EntityBo entity, final HRImportRecord record) {
    final String principalId = record.getPrincipalId();
    final String principalName = record.getPrincipalName();
    PrincipalBo principal = legacyDataAdapter.findBySinglePrimaryKey(PrincipalBo.class, principalId);
        
    boolean modified = false;
    if (principal != null) {
      if (!principalId.equals(principal.getPrincipalId())) {
        // This seems silly, but an error was encountered where IDs '2' and '0002' were being treated as equivalent.
        // See: https://jira.kuali.org/browse/KULRICE-12298
        throw new IllegalStateException ("selected for principal with ID " + principalId 
            + " but retrieved a principal with ID " + principal.getPrincipalId());        
      }
      if (entity.getId() != null  && principal.getEntityId() != null && !entity.getId().equals(principal.getEntityId())) {
        throw new IllegalArgumentException ("principal with ID " + principalId + " is already assigned to another person");
      }
      if (!principal.getPrincipalName().equals(record.getPrincipalName())) {
        principal.setPrincipalName(record.getPrincipalName());
        modified = true;
      }
      if (principal.isActive() != record.isActive()) {
        principal.setActive(record.isActive());
        modified = true;
      }
    } else {
      modified = true;
      principal = new PrincipalBo();
      principal.setPrincipalName(principalName);
      principal.setActive(true);
      principal.setPrincipalId(principalId);
      principal.setEntityId(entity.getId());
    }

    final Principal existingPrincipal = identityService
        .getPrincipalByPrincipalName(principalName);
    if (existingPrincipal != null
        && !existingPrincipal.getPrincipalId().equals(principalId)) {
      throw new IllegalArgumentException("Cannot set principal_name for principal_id '"
          + principalId + "' to '" + principalName
          + "' because it is already in use by principal_id: '"
          + existingPrincipal.getPrincipalId() + "'.");
    }

    if (modified) {
        return legacyDataAdapter.save(principal);
    }
    
    return principal;
  }

  /**
   * Finds or prepares the contact info object for a given entity.
   * 
   * @param entity
   * @return
   */
  private final EntityTypeContactInfoBo getEntityTypeContactInfoBo(final EntityBo entity) {
    EntityTypeContactInfoBo contactInfo = entity
        .getEntityTypeContactInfoByTypeCode(PERSON);

    if (contactInfo == null) {
      contactInfo = new EntityTypeContactInfoBo();
      contactInfo.setEntityId(entity.getId());
      contactInfo.setAddresses(new LinkedList<EntityAddressBo>());
      contactInfo.setPhoneNumbers(new LinkedList<EntityPhoneBo>());
      contactInfo.setEmailAddresses(new LinkedList<EntityEmailBo>());
      contactInfo.setEntityTypeCode(PERSON);
      contactInfo.setActive(true);

        return legacyDataAdapter.save(contactInfo);
      // Can we safely avoid this in JPA?
      // entity.refreshReferenceObject("entityTypeContactInfos");
    }
    return contactInfo;
  }

  protected EntityBo newEntityBo (final HRImportRecord record) {
    final EntityBo entity = new EntityBo();
    final String entityId = record.getEntityId();
    
    if (entityId != null) {
      entity.setId(entityId);
    }
    entity.setActive(record.isActive());
      return legacyDataAdapter.save(entity);
  }

  protected final boolean nullSafeEquals (final Object obj0, final Object obj1) {
    if (obj0 == null) {
      return (obj1 == null);
    } else {
      return obj0.equals(obj1);
    }
  }
  
  protected final boolean equals (final KcPersonExtendedAttributes oldAttrs, final KCExtendedAttributes newAttrs) {
    if (!nullSafeEquals(oldAttrs.getAgeByFiscalYear(),newAttrs.getAgeByFiscalYear())) return false;
    if (!nullSafeEquals(oldAttrs.getCitizenshipTypeCode(),newAttrs.getCitizenshipType())) return false;
    if (!nullSafeEquals(oldAttrs.getCounty(),newAttrs.getCounty())) return false;
    if (!nullSafeEquals(oldAttrs.getDegree(),newAttrs.getDegree())) return false;
    if (!nullSafeEquals(oldAttrs.getDirectoryDepartment(),newAttrs.getDirectoryDepartment())) return false;
    if (!nullSafeEquals(oldAttrs.getDirectoryTitle(),newAttrs.getDirectoryTitle())) return false;
    if (!nullSafeEquals(oldAttrs.getEducationLevel(),newAttrs.getEducationLevel())) return false;
    if (oldAttrs.getHandicappedFlag() != newAttrs.isHandicapped()) return false;
    if (!nullSafeEquals(oldAttrs.getHandicapType(),newAttrs.getHandicapType())) return false;
    if (oldAttrs.getHasVisa() != newAttrs.isVisa()) return false;
    if (!nullSafeEquals(oldAttrs.getIdProvided(),newAttrs.getIdProvided())) return false;
    if (!nullSafeEquals(oldAttrs.getIdVerified(),newAttrs.getIdVerified())) return false;
    if (!nullSafeEquals(oldAttrs.getMajor(),newAttrs.getMajor())) return false;
    if (!nullSafeEquals(oldAttrs.getMultiCampusPrincipalId(),newAttrs.getMultiCampusPrincipalId())) return false;
    if (!nullSafeEquals(oldAttrs.getMultiCampusPrincipalName(),newAttrs.getMultiCampusPrincipalName())) return false;
    if (!nullSafeEquals(oldAttrs.getOfficeLocation(),newAttrs.getOfficeLocation())) return false;
    if (oldAttrs.getOnSabbaticalFlag() != newAttrs.isOnSabbatical()) return false;
    if (!nullSafeEquals(oldAttrs.getPrimaryTitle(),newAttrs.getPrimaryTitle())) return false;
    if (!nullSafeEquals(oldAttrs.getRace(),newAttrs.getRace())) return false;

    final Date oldSalDate = oldAttrs.getSalaryAnniversaryDate();
    final Date newSalDate = newAttrs.getSalaryAnniversaryDate();
    if (oldSalDate != null) {
      if (newSalDate == null || oldSalDate.getTime() != newSalDate.getTime()) {
        return false;
      }
    } else {
      if (newSalDate != null) {
        return false;
      }
    }
    
    if (!nullSafeEquals(oldAttrs.getSchool(),newAttrs.getSchool())) return false;
    if (!nullSafeEquals(oldAttrs.getSecondaryOfficeLocation(),newAttrs.getSecondaryOfficeLocation())) return false;
    if (oldAttrs.getVacationAccrualFlag() != newAttrs.isVacationAccrual()) return false;
    if (oldAttrs.getVeteranFlag() != newAttrs.isVeteran()) return false;
    if (!nullSafeEquals(oldAttrs.getVeteranType(),newAttrs.getVeteranType())) return false;
    if (!nullSafeEquals(oldAttrs.getVisaCode(),newAttrs.getVisaCode())) return false;

    final Date oldVisaDate = oldAttrs.getVisaRenewalDate();
    final Date newVisaDate = newAttrs.getVisaRenewalDate();
    if (oldVisaDate != null) {
      if (newVisaDate == null || oldVisaDate.getTime() != newVisaDate.getTime()) {
        return false;
      }
    } else {
      if (newVisaDate != null) {
        return false;
      }
    }

    if (!nullSafeEquals(oldAttrs.getVisaType(),newAttrs.getVisaType())) return false;
    final String oldGradYear = oldAttrs.getYearGraduated();
    final Integer newGradYearInt = newAttrs.getYearGraduated();
    final String newGradYear = newGradYearInt != null ? newGradYearInt.toString() : null;
    
    if (!nullSafeEquals(oldGradYear,newGradYear)) return false;
    return true;
  }
  
  protected boolean updateExtendedAttributes (final String principalId, final HRImportRecord record) {
    final KcPerson kcPerson = getKcPerson(principalId);
    
    final KCExtendedAttributes newAttrs = record.getKcExtendedAttributes();
      KcPersonExtendedAttributes attrs = getKcPersonExtendedAttributes(kcPerson);
    boolean modified = false;
    attrs.setPersonId(principalId);

    if (!equals (attrs, newAttrs)) {
      attrs.setAgeByFiscalYear(newAttrs.getAgeByFiscalYear());
      attrs.setCitizenshipTypeCode(newAttrs.getCitizenshipType());
      attrs.setCitizenshipType(legacyDataAdapter.findBySinglePrimaryKey(CitizenshipType.class, newAttrs.getCitizenshipType()));
      attrs.setCounty(newAttrs.getCounty());
      attrs.setDegree(newAttrs.getDegree());
      attrs.setDirectoryDepartment(newAttrs.getDirectoryDepartment());
      attrs.setDirectoryTitle(newAttrs.getDirectoryTitle());
      attrs.setEducationLevel(newAttrs.getEducationLevel());
      attrs.setHandicappedFlag(newAttrs.isHandicapped());
      attrs.setHandicapType(newAttrs.getHandicapType());
      attrs.setHasVisa(newAttrs.isVisa());
      attrs.setIdProvided(newAttrs.getIdProvided());
      attrs.setIdVerified(newAttrs.getIdVerified());
      attrs.setMajor(newAttrs.getMajor());
      attrs.setMultiCampusPrincipalId(newAttrs.getMultiCampusPrincipalId());
      attrs.setMultiCampusPrincipalName(newAttrs.getMultiCampusPrincipalName());
      attrs.setOfficeLocation(newAttrs.getOfficeLocation());
      attrs.setOnSabbaticalFlag(newAttrs.isOnSabbatical());
      attrs.setPrimaryTitle(newAttrs.getPrimaryTitle());
      attrs.setRace(newAttrs.getRace());
      final Date annvDate = newAttrs.getSalaryAnniversaryDate();
      if (annvDate != null) {
        attrs.setSalaryAnniversaryDate(new java.sql.Date(annvDate.getTime()));        
      }
      attrs.setSchool(newAttrs.getSchool());
      attrs.setSecondaryOfficeLocation(newAttrs.getSecondaryOfficeLocation());
      attrs.setVacationAccrualFlag(newAttrs.isVacationAccrual());
      attrs.setVeteranFlag(newAttrs.isVeteran());
      attrs.setVeteranType(newAttrs.getVeteranType());
      attrs.setVisaCode(newAttrs.getVisaCode());
      final Date visaRenewDate = newAttrs.getVisaRenewalDate();
      if (visaRenewDate != null) {
        attrs.setVisaRenewalDate(new java.sql.Date(visaRenewDate.getTime()));        
      }
      attrs.setVisaType(newAttrs.getVisaType());
      final Integer gradYear = newAttrs.getYearGraduated();
      if (gradYear != null) {
        attrs.setYearGraduated(gradYear.toString());
      }

        attrs = legacyDataAdapter.save(attrs);

      modified =  true;
    }
    final AppointmentCollection apptColl = record.getAppointmentCollection();
    if (apptColl != null && mergeImportedBOs(apptColl.getAppointments(), 
        attrs.getPersonAppointments(), appointmentAdapter, principalId)) {
      modified = true;
    }
    final DegreeCollection degColl = record.getDegreeCollection();
    if (degColl != null && mergeImportedBOs(degColl.getDegrees(),
        attrs.getPersonDegrees(), degreeAdapter, principalId)) {
      modified = true;
    }
    
    return modified;
  }

  public void setIdentityService(final IdentityService identityService) {
    this.identityService = identityService;
  }

  public void setLegacyDataAdapter(final LegacyDataAdapter legacyDataAdapter) {
    this.legacyDataAdapter = legacyDataAdapter;
  }

  public void setUnitService(final UnitService unitService) {
    appointmentAdapter.setUnitService(unitService);
  }
  
  public void setCacheManagerRegistry(final CacheManagerRegistry registry) {
    cacheManagerRegistry = registry;
  }
  
  @Override
  public void abort(final String importId) {
    stopRunningImport(importId);
  }

public AuthServicePushService getAuthServicePushService() {
	return authServicePushService;
}

public void setAuthServicePushService(AuthServicePushService authServicePushService) {
	this.authServicePushService = authServicePushService;
}


}
