/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2020 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package co.kuali.coeus.data.migration.custom;

import org.apache.commons.lang3.StringUtils;
import org.flywaydb.core.api.MigrationType;
import org.flywaydb.core.api.MigrationVersion;
import org.flywaydb.core.api.resolver.MigrationExecutor;
import org.flywaydb.core.api.resolver.MigrationResolver;
import org.flywaydb.core.api.resolver.ResolvedMigration;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.core.type.filter.RegexPatternTypeFilter;

import javax.sql.DataSource;
import java.lang.reflect.InvocationTargetException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

public class CoeusMigrationResolver implements MigrationResolver {

    protected DataSource riceDataSource;
    protected String javaMigrationPath;

    @Override
    public Set<ResolvedMigration> resolveMigrations() {
        Set<ResolvedMigration> migrations = new HashSet<>();

        final ClassPathScanningCandidateComponentProvider provider = new ClassPathScanningCandidateComponentProvider(true);
        // add include filters which matches all the classes... without it it returns nothing
        provider.addIncludeFilter(new RegexPatternTypeFilter(Pattern.compile(".*")));
        final Set<BeanDefinition> components = provider.findCandidateComponents(pathToPackage(javaMigrationPath));
        for (BeanDefinition component : components) {
            try {
                Class<?> clazz = Class.forName(component.getBeanClassName());
                if (SqlExecutor.class.isAssignableFrom(clazz)) {
                    final SqlExecutor migration = (SqlExecutor) clazz.getDeclaredConstructor().newInstance();
                    if (migration instanceof RiceDataSourceAware) {
                        ((RiceDataSourceAware) migration).setRiceDataSource(riceDataSource);
                    }
                    migrations.add(new SqlExecutorMigrationAdapter(migration));
                }
            } catch (InstantiationException|IllegalAccessException|ClassNotFoundException|NoSuchMethodException|InvocationTargetException e) {
                throw new RuntimeException(e);
            }
        }
        return migrations;
    }

    private String pathToPackage(String path) {
        String pkg =  path.replaceAll("/", ".");

        while (pkg.endsWith(".")) {
            pkg = StringUtils.removeEnd(pkg, ".");
        }
        return pkg;
    }

    public CoeusMigrationResolver() { }
    public CoeusMigrationResolver(DataSource riceDataSource) {
        this.riceDataSource = riceDataSource;
    }

    public DataSource getRiceDataSource() {
        return riceDataSource;
    }

    public void setRiceDataSource(DataSource riceDataSource) {
        this.riceDataSource = riceDataSource;
    }

    public String getJavaMigrationPath() {
        return javaMigrationPath;
    }

    public void setJavaMigrationPath(String javaMigrationPath) {
        this.javaMigrationPath = javaMigrationPath;
    }

    private static class SqlExecutorMigrationAdapter implements ResolvedMigration, MigrationExecutor {

        private final SqlExecutor migration;

        private SqlExecutorMigrationAdapter(SqlExecutor o) {
            migration = o;
        }

        @Override
        public void execute(Connection connection) throws SQLException {
            migration.execute(connection);
        }

        @Override
        public boolean executeInTransaction() {
            return  migration.executeInTransaction();
        }

        @Override
        public Integer getChecksum() {
            return null;
        }

        @Override
        public String getPhysicalLocation() {
            return migration.getClass().getCanonicalName();
        }

        @Override
        public String getScript() {
            return migration.getClass().getSimpleName();
        }

        @Override
        public MigrationType getType() {
            return MigrationType.CUSTOM;
        }

        @Override
        public MigrationVersion getVersion() {
            return MigrationVersion.fromVersion(migration.getClass().getSimpleName().split("__")[0].substring(1));
        }

        @Override
        public String getDescription() {
            return migration.getClass().getSimpleName().split("__")[1];
        }

        @Override
        public MigrationExecutor getExecutor() {
            return this;
        }
    }
}
