/*
 * Decompiled with CFR 0.152.
 */
package org.kuali.coeus.elasticsearch;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Stopwatch;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.xcontent.XContentType;
import org.joda.time.LocalDateTime;
import org.joda.time.ReadablePartial;
import org.kuali.coeus.common.framework.auth.perm.DocumentLevelPermissionable;
import org.kuali.coeus.elasticsearch.ElasticsearchIndexService;
import org.kuali.coeus.elasticsearch.GrantsSearchDocument;
import org.kuali.coeus.elasticsearch.serializers.ElasticsearchDocumentSerializer;
import org.kuali.rice.kew.api.exception.WorkflowException;
import org.kuali.rice.kim.api.permission.PermissionService;
import org.kuali.rice.kim.api.role.Role;
import org.kuali.rice.kim.api.role.RoleMembership;
import org.kuali.rice.kim.api.role.RoleService;
import org.kuali.rice.krad.document.Document;
import org.kuali.rice.krad.service.DocumentService;
import org.springframework.beans.factory.annotation.Autowire;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Configurable;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

@Configurable(autowire=Autowire.BY_TYPE)
@Transactional(propagation=Propagation.REQUIRES_NEW)
public class ElasticsearchIndexServiceImpl
implements ElasticsearchIndexService {
    private static final Logger LOG = LogManager.getLogger(ElasticsearchIndexServiceImpl.class);
    private static final String DEFAULT_KIM_TYPE = "1";
    private static final int INDEX_TIME_WARN_THRESHOLD = 3000;
    private static final int INDEX_PROGRESS_SIZE = 100;
    @Autowired
    private List<ElasticsearchDocumentSerializer> serializers;
    @Autowired
    private DocumentService documentService;
    @Autowired
    @Qualifier(value="elasticsearchClient")
    private RestHighLevelClient elasticsearchClient;
    @Autowired
    @Qualifier(value="elasticsearchObjectMapper")
    private ObjectMapper objectMapper;
    @Autowired
    private PermissionService permissionService;
    @Autowired
    private RoleService roleService;
    @Autowired
    @Qualifier(value="elasticsearchForkJoinPool")
    private ForkJoinPool indexThreadPool;
    private String indexName = "documents";

    @Override
    public List<CompletableFuture<Boolean>> bulkIndex(List<String> documentIds) {
        int numDocs = documentIds.size();
        AtomicInteger progress = new AtomicInteger();
        AtomicInteger successfullyIndexed = new AtomicInteger();
        Stopwatch timer = Stopwatch.createStarted();
        LOG.info(String.format("Bulk indexing %d documents using %d threads...", numDocs, this.indexThreadPool.getParallelism()));
        List<CompletableFuture<Boolean>> results = documentIds.stream().map(docId -> CompletableFuture.supplyAsync(() -> this.retrieveDocument((String)docId).map(document -> {
            int numProcessed;
            boolean success = this.doIndex((Document)document);
            if (success) {
                successfullyIndexed.incrementAndGet();
            }
            if ((numProcessed = progress.incrementAndGet()) % 100 == 0) {
                LOG.info(String.format("%d / %d documents processed for indexing", numProcessed, numDocs));
            }
            return success;
        }).orElse(false), this.indexThreadPool)).collect(Collectors.toList());
        CompletableFuture.allOf(results.toArray(new CompletableFuture[0])).whenCompleteAsync((result, e) -> {
            timer.stop();
            LOG.info(String.format("Took %d seconds to index %d / %d documents", timer.elapsed(TimeUnit.SECONDS), successfullyIndexed.get(), progress.get()));
        });
        return results;
    }

    @Override
    public Future<Boolean> index(String docId) {
        return this.retrieveDocument(docId).map(this::index).orElse(CompletableFuture.completedFuture(false));
    }

    @Override
    public Future<Boolean> index(Document document) {
        return this.indexThreadPool.submit(() -> this.doIndex(document));
    }

    protected boolean doIndex(final Document document) {
        Stopwatch timer = Stopwatch.createStarted();
        return this.buildIndexRequestForDocument(document).map(request -> {
            this.elasticsearchClient.indexAsync(request, RequestOptions.DEFAULT, (ActionListener)new ActionListener<IndexResponse>(){

                public void onResponse(IndexResponse indexResponse) {
                    LOG.trace(String.format("Document %s indexed into Elasticsearch successfully", document.getDocumentNumber()));
                }

                public void onFailure(Exception e) {
                    LOG.error(String.format("Failed to index document %s into Elasticsearch", document.getDocumentNumber()), (Throwable)e);
                }
            });
            timer.stop();
            long elapsed = timer.elapsed(TimeUnit.MILLISECONDS);
            if (elapsed > 3000L) {
                LOG.warn(String.format("Took %d ms to index document %s into Elasticsearch", elapsed, document.getDocumentNumber()));
            }
            return true;
        }).orElse(false);
    }

    protected Optional<Document> retrieveDocument(String docId) {
        try {
            return Optional.ofNullable(this.documentService.getByDocumentHeaderId(docId));
        }
        catch (WorkflowException e) {
            LOG.error(String.format("Failed to retrieve document #%s", docId), (Throwable)e);
            return Optional.empty();
        }
    }

    protected Optional<IndexRequest> buildIndexRequestForDocument(Document document) {
        if (document instanceof DocumentLevelPermissionable) {
            return this.translateDocument((DocumentLevelPermissionable)document).map(esDoc -> {
                try {
                    IndexRequest indexRequest = new IndexRequest(this.indexName, "_doc", document.getDocumentNumber());
                    indexRequest.source(this.objectMapper.writeValueAsBytes(esDoc), XContentType.JSON);
                    return indexRequest;
                }
                catch (JsonProcessingException e) {
                    LOG.error(String.format("Exception encountered preparing document %s for indexing", document.getDocumentNumber()), (Throwable)e);
                    return null;
                }
            });
        }
        return Optional.empty();
    }

    protected Optional<GrantsSearchDocument> translateDocument(DocumentLevelPermissionable document) {
        if (StringUtils.isBlank((CharSequence)document.getDocumentNumberForPermission())) {
            return Optional.empty();
        }
        return this.serializers.stream().filter(serializer -> serializer.supports(document)).findFirst().map(serializer -> {
            try {
                GrantsSearchDocument esDoc = serializer.translateDocument(document);
                esDoc.setViewers(this.getDocumentViewers(document, serializer.getViewPermissions()));
                return esDoc;
            }
            catch (RuntimeException e) {
                LOG.error(String.format("Failed to serialize document %s using %s", document.getDocumentNumber(), serializer.getClass().getSimpleName()), (Throwable)e);
                return null;
            }
        });
    }

    protected Set<String> getDocumentViewers(DocumentLevelPermissionable document, Set<String> viewPerms) {
        if (document == null || viewPerms == null || viewPerms.isEmpty()) {
            return Collections.emptySet();
        }
        HashMap<String, String> qualifiers = new HashMap<String, String>();
        qualifiers.put(document.getDocumentKey(), document.getDocumentNumberForPermission());
        qualifiers.put("documentNumber", document.getDocumentNumber());
        qualifiers.put("unitNumber", document.getLeadUnitNumber());
        Set viewRoles = viewPerms.stream().flatMap(perm -> this.permissionService.getRoleIdsForPermission(document.getNamespace(), perm).stream()).collect(Collectors.toSet());
        Set defaultViewRoles = this.roleService.getRoles(new ArrayList(viewRoles)).stream().filter(role -> DEFAULT_KIM_TYPE.equals(role.getKimTypeId())).map(Role::getId).collect(Collectors.toSet());
        Set qualifiedViewRoles = viewRoles.stream().filter(roleId -> !defaultViewRoles.contains(roleId)).collect(Collectors.toSet());
        return Stream.concat(this.roleService.getRoleMembers(new ArrayList(qualifiedViewRoles), qualifiers).stream(), this.roleService.getRoleMembers(new ArrayList(defaultViewRoles), Collections.emptyMap()).stream()).filter(this::isMembershipActive).map(RoleMembership::getMemberId).collect(Collectors.toSet());
    }

    protected boolean isMembershipActive(RoleMembership membership) {
        LocalDateTime now = LocalDateTime.now();
        LocalDateTime startDate = membership.getActiveFromDate() != null ? membership.getActiveFromDate().toLocalDateTime() : null;
        LocalDateTime endDate = membership.getActiveToDate() != null ? membership.getActiveToDate().toLocalDateTime() : null;
        boolean hasStarted = startDate == null || startDate.isBefore((ReadablePartial)now);
        boolean hasEnded = endDate != null && endDate.isBefore((ReadablePartial)now);
        return hasStarted && !hasEnded;
    }

    @Override
    public boolean isIdle() {
        return this.indexThreadPool.isQuiescent();
    }

    public void setSerializers(List<ElasticsearchDocumentSerializer> serializers) {
        this.serializers = serializers;
    }

    public void setDocumentService(DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setElasticsearchClient(RestHighLevelClient elasticsearchClient) {
        this.elasticsearchClient = elasticsearchClient;
    }

    public void setObjectMapper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    public void setPermissionService(PermissionService permissionService) {
        this.permissionService = permissionService;
    }

    public void setRoleService(RoleService roleService) {
        this.roleService = roleService;
    }

    public void setIndexThreadPool(ForkJoinPool indexThreadPool) {
        this.indexThreadPool = indexThreadPool;
    }

    public void setIndexName(String indexName) {
        this.indexName = indexName;
    }
}

