/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate;


import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType.FileLocation;
import gov.grants.apply.system.globalV10.HashValueDocument.HashValue;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlObject;
import org.kuali.coeus.common.api.sponsor.hierarchy.SponsorHierarchyService;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerContract;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerHeaderContract;
import org.kuali.coeus.common.questionnaire.api.core.QuestionAnswerService;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeService;
import org.kuali.coeus.propdev.api.core.DevelopmentProposalContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.person.attachment.ProposalPersonBiographyContract;
import org.kuali.coeus.propdev.api.questionnaire.PropDevQuestionAnswerService;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.api.core.InfrastructureConstants;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.api.hash.GrantApplicationHashService;
import org.kuali.coeus.s2sgen.impl.generate.support.GlobalLibraryV2_0Generator;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.io.Resource;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public abstract class S2SBaseFormGenerator<T extends XmlObject> implements S2SFormGenerator<T>, BeanNameAware {

    protected static final String NOT_ANSWERED = "No";
    protected static final String AREAS_AFFECTED_ABSTRACT_TYPE_CODE="16";
    protected static final int ORGANIZATON_NAME_MAX_LENGTH = 60;
    protected static final int ORGANIZATION_UEI_MAX_LENGTH = 12;
    protected static final int DUNS_NUMBER_MAX_LENGTH = 13;
    protected static final int PRIMARY_TITLE_MAX_LENGTH = 45;
    protected static final int CONGRESSIONAL_DISTRICT_MAX_LENGTH = 6;
    protected static final String DEFAULT_SORT_INDEX = "1000";
    private static final String REPLACEMENT_CHARACTER = "_";
    private static final String REGEX_TITLE_FILENAME_PATTERN = "([^0-9a-zA-Z\\.\\-_])";

    protected ProposalDevelopmentDocumentContract pdDoc = null;
    private List<AuditError> auditErrors = new ArrayList<>();
    private List<AttachmentData> attachments = new ArrayList<>();

    private String beanName;

    @Autowired
    @Qualifier("narrativeService")
    private NarrativeService narrativeService;

    @Autowired
    @Qualifier("propDevQuestionAnswerService")
    private PropDevQuestionAnswerService propDevQuestionAnswerService;

    @Autowired
    @Qualifier("questionAnswerService")
    private QuestionAnswerService questionAnswerService;

    @Autowired
    @Qualifier("sponsorHierarchyService")
    private SponsorHierarchyService sponsorHierarchyService;

    @Autowired
    @Qualifier("grantApplicationHashService")
    private GrantApplicationHashService grantApplicationHashService;

    /*
     * Reference to global library generators are defined here. The actual form generator will decide which object to be used for
     * respective implementations.
     */

    @Autowired
    @Qualifier("GlobalLibraryV2_0Generator")
    protected GlobalLibraryV2_0Generator globLibV20Generator;

    /**
     * Gets the list of attachments associated with a form. As the form generator fills the form data, the attachment information is
     * stored into the instance variable
     * 
     * @return List&lt;AttachementData&gt; List of attachments associated with the the form. Returns an empty list if no attachment is
     *         available.
     */

    @Override
    public List<AttachmentData> getAttachments() {
        return attachments;
    }

    /**
     * 
     * This is for adding attachment for the forms during form generation.
     * 
     * @param attachment - The attachment data to add.
     */
    protected void addAttachment(AttachmentData attachment) {
        attachments.add(attachment);
    }

    /**
     * 
     * This method is used to generate the HashValue for a particular file stream. The hashing algorithm is defined in constant
     * S2SConstants.HASH_ALGORITHM
     * 
     * @param fileData - They byte[] containing the file data.
     * @return HashValue - The hash value calculated for the fileData input.
     */
    protected HashValue getHashValue(byte[] fileData) {
        return createHashValue(grantApplicationHashService.computeAttachmentHash(fileData));
    }
    /**
     * 
     * Generates the contentId or href for narrative attachments in S2S
     */
    protected String createContentId(NarrativeContract narrative) {
        String retVal = narrative.getProposalNumber() + "-N-" + narrative.getModuleNumber();
        if (narrative.getNarrativeType() != null) {
            if (narrative.getNarrativeType().isAllowMultiple() &&
                    StringUtils.isNotBlank(narrative.getModuleTitle())) {
                retVal += "_" + narrative.getModuleTitle().trim();
            } else {
                retVal += "_" + narrative.getNarrativeType().getDescription().trim();
            }
        }
        retVal = cleanFileName(retVal);
        int index = getCountOfAttachmentAlreadyAdded(retVal);
        if (index > 0) {
            retVal += "-" + index;
        }
        return retVal;
    }

    private int getCountOfAttachmentAlreadyAdded(String contentId) {
        return (int)getAttachments().stream().filter(attachmentData -> attachmentData.getContentId().startsWith(contentId)).count();
    }

    protected String createContentId(ProposalPersonBiographyContract biography) {
        String retVal = biography.getProposalNumber() + "-B-" + biography.getProposalPersonNumber() + "_" + biography.getBiographyNumber();
        if (biography.getPropPerDocType() != null)
            retVal += "_" + biography.getPropPerDocType().getDescription().trim();
        if (StringUtils.isNotBlank(biography.getDescription())) {
            retVal += "_" + biography.getDescription().trim();
        }
        retVal = cleanFileName(retVal);
        int index = getCountOfAttachmentAlreadyAdded(retVal);
        if (index > 0) {
            retVal += "-" + index;
        }
        return retVal;
    }



    /**
     * 
     * This method creates and returns Hash Value for particular form.
     * @return hashValue (HashValue)
     * 
     */
    private synchronized static HashValue createHashValue(String hashValueStr) {
        HashValue hashValue = HashValue.Factory.newInstance();
        hashValue.setHashAlgorithm(InfrastructureConstants.HASH_ALGORITHM);
        hashValue.setStringValue(hashValueStr);
        return hashValue;
    }

    protected AttachedFileDataType addAttachedFileType(NarrativeContract narrative) {
        AttachedFileDataType attachedFileDataType = null;
        byte[] attachementContent = null;
        if(narrative.getNarrativeAttachment()!= null){
        	attachementContent = narrative.getNarrativeAttachment().getData();
        }
	    if(attachementContent != null && attachementContent.length > 0 ){
	    	
	        FileLocation fileLocation = FileLocation.Factory.newInstance();
	        String contentId = createContentId(narrative);
	        fileLocation.setHref(contentId);
	    	
	        attachedFileDataType = AttachedFileDataType.Factory.newInstance();
	        attachedFileDataType.setFileLocation(fileLocation);
	        attachedFileDataType.setFileName(getS2sNarrativeFileName(narrative));
	        attachedFileDataType.setMimeType(InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM);
            final HashValue hashValue = getHashValue(attachementContent);
            attachedFileDataType.setHashValue(hashValue);
	        addAttachment(new AttachmentData(narrative.getNarrativeAttachment().getFileDataId(), getS2sNarrativeFileName(narrative), contentId, attachementContent, InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM, hashValue.getHashAlgorithm(), hashValue.getStringValue(), narrative.getNarrativeAttachment().getUploadUser(), narrative.getNarrativeAttachment().getUploadTimestamp()));
	    }
        return attachedFileDataType;
    }
    /**
     * 
     * This method is used to get List of Other attachments from NarrativeAttachment
     * 
     * @return AttachedFileDataType[] based on the narrative type code.
     */
    protected AttachedFileDataType[] getAttachedFileDataTypes(String narrativeTypeCode) {

        int size = 0;
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            if (narrative.getNarrativeType() != null
                    && narrative.getNarrativeType().getCode().equals(narrativeTypeCode)) {
                size++;
            }
        }
        AttachedFileDataType[] attachedFileDataTypes = new AttachedFileDataType[size];
        int attachments = 0;
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            if (narrative.getNarrativeType() != null
                    && narrative.getNarrativeType().getCode().equals(narrativeTypeCode)) {
                attachedFileDataTypes[attachments] = addAttachedFileType(narrative);
                attachments++;
            }
        }
        return attachedFileDataTypes;
    }
    /**
     * 
     * This method is used to get List of Other attachments from NarrativeAttachment
     * 
     * @return AttachedFileDataType[] based on the narrative type code.
     */
    protected AttachedFileDataType getAttachedFileDataType(String narrativeTypeCode) {

        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            if (narrative.getNarrativeType() != null
                    && narrative.getNarrativeType().getCode().equals(narrativeTypeCode)) {
                return addAttachedFileType(narrative);
            }
        }
        return null;
    }

    /**
     * 
     * This method fetches the attachments for {@link org.kuali.coeus.propdev.api.person.ProposalPersonContract}. For a given person or rolodex ID, it will fetch the document
     * of required type, also passed alongside as documentType
     * 
     * @param pdDoc {@link ProposalDevelopmentDocumentContract} from which the attachments are to be fetched
     * @param personId ID of the proposal person
     * @param rolodexId Rolodex ID of the person
     * @param documentType type of document thats to be fetched
     * @return {@link AttachedFileDataType} containing the required document
     */
    protected AttachedFileDataType getPersonnelAttachments(ProposalDevelopmentDocumentContract pdDoc, String personId, Integer rolodexId,
                                                           String documentType) {
        boolean personBiographyFound = false;
        for (ProposalPersonBiographyContract proposalPersonBiography : pdDoc.getDevelopmentProposal().getPropPersonBios()) {
            if (personId != null && proposalPersonBiography.getPersonId() != null
                    && proposalPersonBiography.getPersonId().equals(personId)
                    && documentType.equals(proposalPersonBiography.getPropPerDocType().getCode())) {
                personBiographyFound = true;
            }
            else if (rolodexId != null && proposalPersonBiography.getRolodexId() != null
                    && proposalPersonBiography.getRolodexId().toString().equals(rolodexId.toString())
                    && proposalPersonBiography.getPropPerDocType() != null
                    && documentType.equals(proposalPersonBiography.getPropPerDocType().getCode())) {
                personBiographyFound = true;
            }
            byte[] attachmentContent = null;
            if(proposalPersonBiography.getPersonnelAttachment() != null){
                attachmentContent = proposalPersonBiography.getPersonnelAttachment().getData();
            }

            if (personBiographyFound && attachmentContent != null && attachmentContent.length > 0) {
                FileLocation fileLocation = FileLocation.Factory.newInstance();
                String contentId = createContentId(proposalPersonBiography);
                fileLocation.setHref(contentId);
                AttachedFileDataType attachedFileDataType = AttachedFileDataType.Factory.newInstance();
                attachedFileDataType.setFileLocation(fileLocation);
                attachedFileDataType.setFileName(getS2sPersonnelAttachmentFileName(pdDoc.getDevelopmentProposal(),proposalPersonBiography));
                attachedFileDataType.setMimeType(InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM);
                final HashValue hashValue = getHashValue(proposalPersonBiography.getPersonnelAttachment().getData());
                attachedFileDataType.setHashValue(hashValue);
                addAttachment(new AttachmentData(proposalPersonBiography.getPersonnelAttachment().getFileDataId(), getS2sPersonnelAttachmentFileName(pdDoc.getDevelopmentProposal(),proposalPersonBiography), contentId, proposalPersonBiography.getPersonnelAttachment().getData(), InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM, hashValue.getHashAlgorithm(), hashValue.getStringValue(), proposalPersonBiography.getPersonnelAttachment().getUploadUser(), proposalPersonBiography.getPersonnelAttachment().getUploadTimestamp()));
                return attachedFileDataType;
            }
        }
        return null;
    }

    private int getCountOfAttachmentAlreadyAddedByFileName(String fileName) {
        return (int)getAttachments().stream().filter(attachmentData -> attachmentData.getFileName().startsWith(fileName)).count();
    }

    protected String getS2sNarrativeFileName(NarrativeContract narrative){
        String fileName;
        if (narrative.getNarrativeType().isAllowMultiple()) {
            fileName = narrative.getModuleTitle();
        } else {
            fileName = narrative.getNarrativeType().getDescription();
        }
        fileName = cleanFileName(fileName);
        int attachmentCount = getCountOfAttachmentAlreadyAddedByFileName(fileName);
        if (attachmentCount > 0) {
            fileName += "-" + attachmentCount;
        }
        String extension = StringUtils.substringAfterLast(narrative.getNarrativeAttachment().getName(),".");
        return fileName + "." + extension;
    }

    protected String getS2sPersonnelAttachmentFileName(DevelopmentProposalContract developmentProposal, ProposalPersonBiographyContract biography) {
        String extension = StringUtils.substringAfterLast(biography.getName(),".");
        String fullName = getPerson(developmentProposal,biography.getProposalPersonNumber()).getFullName();
        String docType = biography.getPropPerDocType().getDescription();
        String fileName = cleanFileName(fullName + "_" + docType);
        int attachmentCount = getCountOfAttachmentAlreadyAddedByFileName(fileName);
        if (attachmentCount > 0) {
            fileName += "-" + attachmentCount;
        }

        return fileName + "." + extension;

    }

    protected String  cleanFileName(String fileName) {
        Pattern pattern = Pattern.compile(REGEX_TITLE_FILENAME_PATTERN);
        Matcher matcher = pattern.matcher(fileName);
        return matcher.replaceAll(REPLACEMENT_CHARACTER);
    }

    protected ProposalPersonContract getPerson(DevelopmentProposalContract developmentProposal, Integer proposalPersonNumber) {
        for (ProposalPersonContract person : developmentProposal.getProposalPersons()) {
            if (proposalPersonNumber.equals(person.getProposalPersonNumber())) {
                return person;
            }
        }
        return null;
    }


    @Override
    public List<AuditError> getAuditErrors() {
        return auditErrors;
    }

    public void setAuditErrors(List<AuditError> auditErrors) {
        this.auditErrors = auditErrors;
    }
    
    protected boolean isSponsorNIH(ProposalDevelopmentDocumentContract document) {
		return sponsorHierarchyService.isSponsorNihMultiplePi(document.getDevelopmentProposal().getSponsor().getSponsorCode());
	}
    
	protected NarrativeContract saveNarrative(byte[] attachment, String narrativeTypeCode,String fileName,String comment) {
        return narrativeService.createSystemGeneratedNarrative(pdDoc.getDevelopmentProposal().getProposalNumber(), narrativeTypeCode, attachment, fileName, comment);
	}

    public void setAttachments(List<AttachmentData> attachments) {
        this.attachments = attachments;
    }

    /**
     *
     * This method is used to get the answer for a particular Questionnaire question
     * question based on the question id.
     *
     * @param questionSeqId
     *            the question seq id to be passed.
     * @return returns the answer for a particular
     *         question based on the question id passed.
     */
    protected String getAnswer(Integer questionSeqId, List<? extends AnswerHeaderContract> answerHeaders) {
        for(AnswerHeaderContract answerHeader:answerHeaders){
            if(answerHeader!=null){
                List<? extends AnswerContract> answerDetails = answerHeader.getAnswers();
                for(AnswerContract answers:answerDetails){
                    if(questionSeqId.equals(getQuestionAnswerService().findQuestionById(answers.getQuestionId()).getQuestionSeqId())){
                        return answers.getAnswer();
                    }
                }
            }
        }

        return null;
    }

    /**
     *
     * This method is used to get the answerId for a particular Questionnaire question
     * question based on the question id.
     *
     * @param questionSeqId
     *            the question seq id to be passed.
     * @return returns the answer for a particular
     *         question based on the question id passed.
     */
    protected Long getAnswerId(Integer questionSeqId, List<? extends AnswerHeaderContract> answerHeaders) {
        if (answerHeaders != null && !answerHeaders.isEmpty()) {
            for (AnswerHeaderContract answerHeader : answerHeaders) {
                List<? extends AnswerContract> answerDetails = answerHeader.getAnswers();
                for (AnswerContract answers : answerDetails) {
                    if (answers.getAnswer() != null && questionSeqId.equals(getQuestionAnswerService().findQuestionById(answers.getQuestionId()).getQuestionSeqId())) {
                        return answers.getId();
                    }
                }
            }
        }
        return null;
    }
    /**
     *
     * This method is used to get the child question answer for a particular Questionnaire question
     * question based on the question id.
     * @param parentQuestionSeqId
     *            the parentQuestion id to be passed.
     * @param questionSeqId
     *            the question id to be passed.
     * @return returns the answer for a particular
     *         question based on the question id passed.
     */
    protected String getChildQuestionAnswer(Integer parentQuestionSeqId,Integer questionSeqId, List<? extends AnswerHeaderContract> answerHeaders) {
        for(AnswerHeaderContract answerHeader:answerHeaders){
            if(answerHeader!=null){
                List<? extends AnswerContract> answerDetails = answerHeader.getAnswers();
                for(AnswerContract answers:answerDetails){
                    if(answers.getParentAnswers()!= null){
                        AnswerContract parentAnswer =  answers.getParentAnswers().get(0);
                        if(questionSeqId.equals(getQuestionAnswerService().findQuestionById(answers.getQuestionId()).getQuestionSeqId()) &&
                                parentQuestionSeqId.equals(getQuestionAnswerService().findQuestionById(parentAnswer.getQuestionId()).getQuestionSeqId()) ){
                            return answers.getAnswer();
                        }
                    }
                }
            }
        }

        return null;

    }

    /*
  * This method will get the childAnswer for sub question
  */
    protected String getAnswers(Integer questionSeqId, List<? extends AnswerHeaderContract> answerHeaders) {

        String answer;
        String childAnswer = null;
        StringBuilder stringBuilder = new StringBuilder();
        if (answerHeaders != null && !answerHeaders.isEmpty()) {
            for (AnswerHeaderContract answerHeader : answerHeaders) {
                List<? extends AnswerContract> answerDetails = answerHeader.getAnswers();
                for (AnswerContract answers : answerDetails) {
                    if (questionSeqId.equals(getQuestionAnswerService().findQuestionById(answers.getQuestionId()).getQuestionSeqId())) {
                        answer = answers.getAnswer();
                        if (answer != null) {
                            if (!answer.equals(NOT_ANSWERED)) {
                                stringBuilder.append(answer);
                                stringBuilder.append(",");
                            }
                        }
                        childAnswer = stringBuilder.toString();
                    }
                }
            }
        }
        return childAnswer;
    }

    @Override
    public String getBeanName() {
        return beanName;
    }

    @Override
    public void setBeanName(String beanName) {
        this.beanName = beanName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return Collections.emptyList();
    }

    @Override
    public boolean supportsPdfFilling() {
        return this instanceof S2SFormGeneratorPdfFillable<?>;
    }

    @Override
    public boolean supportsXslTransform() {
        return true;
    }

    public NarrativeService getNarrativeService() {
        return narrativeService;
    }

    public PropDevQuestionAnswerService getPropDevQuestionAnswerService() {
        return propDevQuestionAnswerService;
    }

    public QuestionAnswerService getQuestionAnswerService() {
        return questionAnswerService;
    }

    public void setNarrativeService(NarrativeService narrativeService) {
        this.narrativeService = narrativeService;
    }

    public void setPropDevQuestionAnswerService(PropDevQuestionAnswerService propDevQuestionAnswerService) {
        this.propDevQuestionAnswerService = propDevQuestionAnswerService;
    }

    public void setQuestionAnswerService(QuestionAnswerService questionAnswerService) {
        this.questionAnswerService = questionAnswerService;
    }

    public SponsorHierarchyService getSponsorHierarchyService() {
        return sponsorHierarchyService;
    }

    public void setSponsorHierarchyService(SponsorHierarchyService sponsorHierarchyService) {
        this.sponsorHierarchyService = sponsorHierarchyService;
    }

    public GrantApplicationHashService getGrantApplicationHashService() {
        return grantApplicationHashService;
    }

    public void setGrantApplicationHashService(GrantApplicationHashService grantApplicationHashService) {
        this.grantApplicationHashService = grantApplicationHashService;
    }

    public GlobalLibraryV2_0Generator getGlobLibV20Generator() {
        return globLibV20Generator;
    }

    public void setGlobLibV20Generator(GlobalLibraryV2_0Generator globLibV20Generator) {
        this.globLibV20Generator = globLibV20Generator;
    }
}
