/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;


import gov.grants.apply.forms.phsFellowshipSupplemental31V31.DegreeTypeDataType;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.FieldOfTrainingDataType;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.NonUSCitizenDataType;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.*;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.AdditionalInformation.ConcurrentSupportDescription;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.AdditionalInformation.CurrentPriorNRSASupport;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.Budget.InstitutionalBaseSalary;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.Budget.SupplementationFromOtherSources;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.FellowshipApplicant.BackgroundandGoals;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.InstitutionalEnvironment.InstitutionalEnvironmentCommitmenttoTraining;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.Introduction.IntroductionToApplication;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.OtherResearchTrainingPlan.*;
import gov.grants.apply.forms.phsFellowshipSupplemental31V31.PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.ResearchTrainingPlan.ProgressReportPublicationList;
import gov.grants.apply.forms.phsFellowshipSupplementalV10.CitizenshipDataType;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.attachmentsV10.AttachmentGroupMin0Max100DataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.person.attr.CitizenshipType;
import org.kuali.coeus.common.api.ynq.YnqConstant;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemContract;
import org.kuali.coeus.common.budget.api.period.BudgetPeriodContract;
import org.kuali.coeus.common.budget.api.personnel.BudgetPersonnelDetailsContract;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerContract;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerHeaderContract;
import org.kuali.coeus.common.questionnaire.api.core.QuestionnaireContract;
import org.kuali.coeus.common.questionnaire.api.core.QuestionnaireQuestionContract;
import org.kuali.coeus.common.questionnaire.api.question.QuestionContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.budget.ProposalDevelopmentBudgetExtContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.specialreview.ProposalSpecialReviewContract;
import org.kuali.coeus.s2sgen.api.core.ConfigurationConstants;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.util.FieldValueConstants;
import org.kuali.coeus.sys.api.model.ScaleTwoDecimal;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

@FormGenerator("PHS398FellowshipSupplementalV3_1Generator")
public class PHS398FellowshipSupplementalV3_1Generator extends PHS398FellowshipSupplementalBaseGenerator<PHSFellowshipSupplemental31Document> implements S2SFormGeneratorPdfFillable<PHSFellowshipSupplemental31Document> {

    private static final Logger LOG = LogManager.getLogger(PHS398FellowshipSupplementalV3_1Generator.class);

    private static final int DEGREE_TYPE_SOUGHT = 99;
    private static final int OTHER_DBOTH = 100;

    private static final String SPONSOR_COSPONSOR = "134";
    private static final String LETTER_COLLAB_CONTRIB_CONSULT = "157";
    private static final String PHS_FELLOW_INSTITUTION_ENVIRON_COMMITMENT = "158";
    private static final String DATA_SAFETY_MONITORING_PLAN = "159";
    private static final String PHS_FELLOW_AUTH_KEY_BIO_CHEM_RESOURCES = "160";
    private static final int ARE_VERTEBRATE_ANIMALS_EUTHANISED = 146;
    private static final String INCLUSION_OF_CHILDREN = "107";
    private static final String INTRODUCTION_TO_APPLICATION = "97";
    private static final String SPECIFIC_AIMS = "98";
    private static final String RESEARCH_STRATEGY = "127";
    private static final String RESPECTIVE_CONTRIBUTIONS = "88";
    private static final String SELECTION_OF_SPONSOR_AND_INSTITUTION = "89";
    private static final String PROGRESS_REPORT_PUBLICATION_LIST = "103";
    private static final String RESPONSIBLE_CONDUCT_OF_RESEARCH = "90";
    private static final String PROTECTION_OF_HUMAN_SUBJECTS = "104";
    private static final String INCLUSION_OF_WOMEN_AND_MINORITIES = "105";
    private static final String CONCURRENT_SUPPORT = "91";
    private static final String VERTEBRATE_ANIMALS = "108";
    private static final String SELECT_AGENT_RESEARCH = "109";
    private static final String RESOURCE_SHARING_PLANS = "110";


    // Is method consistent with American Veterinary Medical Association (AVMA) guidelines?
    private static final int CONSISTENT_AVMA_GUIDELINES = 147;
    // If NO to AVMA Guidelines, describe method and provide scientific justification in 1000 characters or less
    private static final int NO_AVMA_METHOD_SCIENTIFIC_JUSTIFICATION = 148;
    private static final int FIELD_OF_TRAINING = 200;
    private static final String FELLOWSHIP_BACKGROUND_AND_GOALS = "156";

    private static final String TEMPORARY_VISA_ALSO_APPLIED_FOR_PERMANENT_RESIDENT_STATUS = "Temporary Visa also applied for permanent resident status";

    private static final ScaleTwoDecimal POINT_ZERO_ONE = new ScaleTwoDecimal(.01);

    @Value("http://apply.grants.gov/forms/PHS_Fellowship_Supplemental_3_1-V3.1")
    private String namespace;

    @Value("PHS_Fellowship_Supplemental_3_1")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/PHS_FellowshipSupplemental-V3.1.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/PHS_Fellowship_Supplemental_3_1-V3.1.pdf")
    private Resource pdfForm;

    @Value("211")
    private int sortIndex;

    private PHSFellowshipSupplemental31Document getPHSFellowshipSupplemental31() {
        PHSFellowshipSupplemental31Document phsFellowshipSupplementalDocument = PHSFellowshipSupplemental31Document.Factory
                .newInstance();
        PHSFellowshipSupplemental31 phsFellowshipSupplemental = phsFellowshipSupplementalDocument
                .addNewPHSFellowshipSupplemental31();
        ResearchTrainingPlan researchTrainingPlan = phsFellowshipSupplemental.addNewResearchTrainingPlan();
        setNarrativeDataForResearchTrainingPlan(phsFellowshipSupplemental, researchTrainingPlan);
        setOtherResearchTrainingPlanVertebrate(phsFellowshipSupplemental);
        phsFellowshipSupplemental.setFormVersion(FormVersion.v3_1.getVersion());
        final AttachmentGroupMin0Max100DataType appendix = getAppendix();
        if (appendix != null) {
            phsFellowshipSupplemental.setAppendix(appendix);
        }
        setQuestionnaireData(phsFellowshipSupplemental);
        return phsFellowshipSupplementalDocument;
    }

    private void setOtherResearchTrainingPlanVertebrate(PHSFellowshipSupplemental31 phsFellowshipSupplemental) {
        OtherResearchTrainingPlan otherResearchTrainingPlan = phsFellowshipSupplemental.getOtherResearchTrainingPlan();
        if (otherResearchTrainingPlan == null) {
            otherResearchTrainingPlan = phsFellowshipSupplemental.addNewOtherResearchTrainingPlan();
        }
        List<? extends AnswerHeaderContract> answers = findQuestionnaireWithAnswers(pdDoc.getDevelopmentProposal());
        for (AnswerHeaderContract answerHeader : answers) {
            QuestionnaireContract questionnaire = questionAnswerService.findQuestionnaireById(answerHeader.getQuestionnaireId());
            List<? extends QuestionnaireQuestionContract> questionnaireQuestions = questionnaire.getQuestionnaireQuestions();
            for (QuestionnaireQuestionContract questionnaireQuestion : questionnaireQuestions) {
                AnswerContract answerBO = getAnswer(questionnaireQuestion, answerHeader);
                String answer = answerBO != null ? answerBO.getAnswer() : null;
                QuestionContract question = questionnaireQuestion.getQuestion();
                Integer questionId = question.getQuestionSeqId();
                if (answer != null) {
                    switch (questionId) {
                        case VERT:
                            // will the inclusion of vertebrate animals use be indefinite
                            otherResearchTrainingPlan.setVertebrateAnimalsIndefinite(getYesNoEnum(answer));
                            break;
                        case ARE_VERTEBRATE_ANIMALS_EUTHANISED:
                            // Are vertebrate animals euthanized
                            otherResearchTrainingPlan.setAreAnimalsEuthanized(getYesNoEnum(answer));
                            break;
                        case CONSISTENT_AVMA_GUIDELINES:
                            otherResearchTrainingPlan.setAVMAConsistentIndicator(getYesNoEnum(answer));
                            break;
                        case NO_AVMA_METHOD_SCIENTIFIC_JUSTIFICATION:
                            otherResearchTrainingPlan.setEuthanasiaMethodDescription(answer);
                            break;
                    }
                }
            }
        }

        AttachedFileDataType attachedFileDataType;
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            final String code = narrative.getNarrativeType().getCode();
            if (code != null) {
                if (code.equalsIgnoreCase(VERTEBRATE_ANIMALS)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        VertebrateAnimals vertebrateAnimals = VertebrateAnimals.Factory.newInstance();
                        vertebrateAnimals.setAttFile(attachedFileDataType);
                        if (otherResearchTrainingPlan == null) {
                            otherResearchTrainingPlan = phsFellowshipSupplemental.addNewOtherResearchTrainingPlan();
                        }
                        otherResearchTrainingPlan.setVertebrateAnimals(vertebrateAnimals);
                    }
                } else if (code.equalsIgnoreCase(SELECT_AGENT_RESEARCH)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        SelectAgentResearch selectAgentResearch = SelectAgentResearch.Factory.newInstance();
                        selectAgentResearch.setAttFile(attachedFileDataType);
                        if (otherResearchTrainingPlan == null) {
                            otherResearchTrainingPlan = phsFellowshipSupplemental.addNewOtherResearchTrainingPlan();
                        }
                        otherResearchTrainingPlan.setSelectAgentResearch(selectAgentResearch);
                    }
                } else if (code.equalsIgnoreCase(RESOURCE_SHARING_PLANS)) {
                    ResourceSharingPlan resourceSharingPlan = ResourceSharingPlan.Factory.newInstance();
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        resourceSharingPlan.setAttFile(attachedFileDataType);
                        if (otherResearchTrainingPlan == null) {
                            otherResearchTrainingPlan = phsFellowshipSupplemental.addNewOtherResearchTrainingPlan();
                        }
                        otherResearchTrainingPlan.setResourceSharingPlan(resourceSharingPlan);
                    }
                } else if (code.equalsIgnoreCase(PHS_FELLOW_AUTH_KEY_BIO_CHEM_RESOURCES)) {
                    KeyBiologicalAndOrChemicalResources keyBiologicalAndOrChemicalResources = KeyBiologicalAndOrChemicalResources.Factory.newInstance();
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        keyBiologicalAndOrChemicalResources.setAttFile(attachedFileDataType);
                        if (otherResearchTrainingPlan == null) {
                            otherResearchTrainingPlan = phsFellowshipSupplemental.addNewOtherResearchTrainingPlan();
                        }
                        otherResearchTrainingPlan.setKeyBiologicalAndOrChemicalResources(keyBiologicalAndOrChemicalResources);
                    }
                }
            }
        }
    }

    private void setQuestionnaireData(PHSFellowshipSupplemental31 phsFellowshipSupplemental) {
        Map<Integer, String> hmBudgetQuestions = new HashMap<>();
        List<? extends AnswerHeaderContract> answers = findQuestionnaireWithAnswers(pdDoc.getDevelopmentProposal());
        OtherResearchTrainingPlan otherResearchTrainingPlan = phsFellowshipSupplemental.getOtherResearchTrainingPlan();
        if (otherResearchTrainingPlan == null) {
            otherResearchTrainingPlan = phsFellowshipSupplemental.addNewOtherResearchTrainingPlan();
        }
        setHumanSubjectInvolvedAndVertebrateAnimalUsed(otherResearchTrainingPlan);
        AdditionalInformation additionalInfoType = phsFellowshipSupplemental.addNewAdditionalInformation();
        AdditionalInformation.GraduateDegreeSought graduateDegreeSought = AdditionalInformation.GraduateDegreeSought.Factory.newInstance();
        AdditionalInformation.StemCells stemCellstype = AdditionalInformation.StemCells.Factory.newInstance();
        List<KirschsteinBean> cvKirsch = new ArrayList<>();
        for (AnswerHeaderContract answerHeader : answers) {
            QuestionnaireContract questionnaire = questionAnswerService.findQuestionnaireById(answerHeader.getQuestionnaireId());
            List<? extends QuestionnaireQuestionContract> questionnaireQuestions = questionnaire.getQuestionnaireQuestions();
            for (QuestionnaireQuestionContract questionnaireQuestion : questionnaireQuestions) {
                AnswerContract answerBO = getAnswer(questionnaireQuestion, answerHeader);
                String answer = answerBO != null ? answerBO.getAnswer() : null;
                QuestionContract question = questionnaireQuestion.getQuestion();
                Integer questionNumber = questionnaireQuestion.getQuestionNumber();
                Integer parentQuestionNumber = questionnaireQuestion.getParentQuestionNumber();
                Integer questionId = question.getQuestionSeqId();
                if (answer != null) {
                    switch (questionId) {
                        case HUMAN:
                            otherResearchTrainingPlan.setHumanSubjectsIndefinite(getYesNoEnum(answer));
                            break;
                        case CLINICAL:
                            // clinical trial
                            otherResearchTrainingPlan.setClinicalTrial(getYesNoEnum(answer));
                            break;
                        case PHASE3CLINICAL:
                            // phase 3 clinical trial
                            otherResearchTrainingPlan.setPhase3ClinicalTrial(getYesNoEnum(answer));
                            break;
                        case STEMCELLS:
                            // stem cells used
                            stemCellstype.setIsHumanStemCellsInvolved(getYesNoEnum(answer));
                            break;
                        case CELLLINEIND:
                            // stem cell line indicator
                            stemCellstype.setStemCellsIndicator(answer
                                    .equals(YnqConstant.NO.code()) ? YesNoDataType.Y_YES
                                    : YesNoDataType.N_NO);
                            break;
                        case STEMCELLLINES:
                            List<? extends AnswerContract> answerList = getAnswers(questionnaireQuestion, answerHeader);
                            for (AnswerContract questionnaireAnswerBO : answerList) {
                                String questionnaireSubAnswer = questionnaireAnswerBO.getAnswer();
                                if (questionnaireSubAnswer != null) {
                                    stemCellstype.addCellLines(questionnaireAnswerBO.getAnswer());
                                }
                            }
                            break;
                        case DEGREE_TYPE_SOUGHT:
                            graduateDegreeSought.setDegreeType(DegreeTypeDataType.Enum.forString(answer));
                            break;
                        case DEG_EXP_COMP_DATE:
                            graduateDegreeSought.setDegreeDate(answer.substring(6, 10) + STRING_SEPRATOR + answer.substring(0, 2));
                            break;
                        case OTHER_MASTERS:
                            graduateDegreeSought.setOtherDegreeTypeText(answer);
                            break;
                        case OTHER_DDOT:
                            graduateDegreeSought.setDegreeType(DegreeTypeDataType.DDOT_OTHER_DOCTOR_OF_MEDICAL_DENTISTRY);
                            graduateDegreeSought.setOtherDegreeTypeText(answer);
                            break;
                        case OTHER_VDOT:
                            graduateDegreeSought.setDegreeType(DegreeTypeDataType.VDOT_OTHER_DOCTOR_OF_VETERINARY_MEDICINE);
                            graduateDegreeSought.setOtherDegreeTypeText(answer);
                            break;
                        case OTHER_MDOT:
                            graduateDegreeSought.setDegreeType(DegreeTypeDataType.MDOT_OTHER_DOCTOR_OF_MEDICINE);
                            graduateDegreeSought.setOtherDegreeTypeText(answer);
                            break;
                        case OTHER_DBOTH:
                            if (graduateDegreeSought.getDegreeType().equals(DegreeTypeDataType.OTH_OTHER)) {
                                graduateDegreeSought.setOtherDegreeTypeText(answer);
                            }
                            break;
                        case OTHER_DOCT:
                            graduateDegreeSought.setDegreeType(DegreeTypeDataType.DOTH_OTHER_DOCTORATE);
                            graduateDegreeSought.setOtherDegreeTypeText(answer);
                            break;
                        case BROAD_TRAINING:
                        case FIELD_OF_TRAINING:
                            if (!answer.toUpperCase().equals(SUB_CATEGORY_NOT_FOUND)) {
                                final FieldOfTrainingDataType.Enum fieldOfTraining = FieldOfTrainingDataType.Enum.forString(answer);
                                additionalInfoType.setFieldOfTraining(fieldOfTraining);
                            }
                            break;
                        case NRSA_SUPPORT:
                            additionalInfoType.setCurrentPriorNRSASupportIndicator(getYesNoEnum(answer));
                            break;
                        case KIRST_START_KNOWN:
                        case KIRST_END_KNOWN:
                        case KIRST_START_DT:
                        case KIRST_END_DT:
                        case KIRST_GRANT_KNOWN:
                        case KIRST_GRANT_NUM:
                        case PRE_OR_POST:
                        case IND_OR_INST:
                            if (questionId == KIRST_START_KNOWN) {
                                if (answer.equals(QUESTION_ANSWER_NO)) {
                                    answer = FieldValueConstants.VALUE_UNKNOWN;
                                    questionId = KIRST_START_DT;
                                } else
                                    break;
                            }
                            if (questionId == KIRST_END_KNOWN) {
                                if (answer.equals(QUESTION_ANSWER_NO)) {
                                    answer = FieldValueConstants.VALUE_UNKNOWN;
                                    questionId = KIRST_END_DT;
                                } else
                                    break;
                            }
                            if (questionId == KIRST_GRANT_KNOWN) {
                                if (answer.equals(QUESTION_ANSWER_NO)) {
                                    answer = FieldValueConstants.VALUE_UNKNOWN;
                                    questionId = KIRST_GRANT_NUM;
                                } else
                                    break;
                            }
                            KirschsteinBean cbKirschstein = new KirschsteinBean();
                            cbKirschstein.setAnswer(answer);
                            cbKirschstein.setQuestionId(questionId);
                            cbKirschstein.setQuestionNumber(questionNumber);
                            cbKirschstein.setParentQuestionNumber(parentQuestionNumber);
                            cvKirsch.add(cbKirschstein);
                            break;
                        case SUBMITTED_DIFF_INST:
                            additionalInfoType.setChangeOfInstitution(getYesNoEnum(answer));
                            break;
                        case FORMER_INST:
                            additionalInfoType.setFormerInstitution(answer);
                            break;
                        case SENIOR_FELL:
                            hmBudgetQuestions.put(SENIOR_FELL, answer);
                            break;
                        case OTHER_SUPP_SOURCE:
                            hmBudgetQuestions.put(OTHER_SUPP_SOURCE, answer);
                            break;
                        case SUPP_SOURCE:
                            hmBudgetQuestions.put(SUPP_SOURCE, answer);
                            break;
                        case SUPP_FUNDING_AMT:
                            hmBudgetQuestions.put(SUPP_FUNDING_AMT, answer);
                            break;
                        case SUPP_MONTHS:
                            hmBudgetQuestions.put(SUPP_MONTHS, answer);
                            break;
                        case SUPP_TYPE:
                            hmBudgetQuestions.put(SUPP_TYPE, answer);
                            break;
                        case SALARY_MONTHS:
                            hmBudgetQuestions.put(SALARY_MONTHS, answer);
                            break;
                        case ACAD_PERIOD:
                            hmBudgetQuestions.put(ACAD_PERIOD, answer);
                            break;
                        case BASE_SALARY:
                            hmBudgetQuestions.put(BASE_SALARY, answer);
                            break;
                        default:
                            break;

                    }
                } else if (answer == null) {
                    switch (questionId) {
                        case HUMAN:
                            otherResearchTrainingPlan.setHumanSubjectsIndefinite(null);
                            otherResearchTrainingPlan.setHumanSubjectsInvolved(null);
                            break;
                        case VERT:
                            otherResearchTrainingPlan.setVertebrateAnimalsIndefinite(null);
                            otherResearchTrainingPlan.setVertebrateAnimalsUsed(null);
                            break;
                        case CLINICAL:
                            otherResearchTrainingPlan.setClinicalTrial(null);
                            break;
                        case PHASE3CLINICAL:
                            if (otherResearchTrainingPlan.getClinicalTrial() == (YesNoDataType.Y_YES)) {
                                otherResearchTrainingPlan.setPhase3ClinicalTrial(null);
                            }
                            break;
                        case FIELD_OF_TRAINING:
                            additionalInfoType.setFieldOfTraining(null);
                            break;
                        case STEMCELLS:
                            stemCellstype.setIsHumanStemCellsInvolved(null);
                            break;
                        case NRSA_SUPPORT:
                            additionalInfoType.setCurrentPriorNRSASupportIndicator(null);
                            break;
                        default:
                            break;
                    }
                }
            }
        }
        if (stemCellstype != null)
            additionalInfoType.setStemCells(stemCellstype);
        if (graduateDegreeSought.getDegreeType() != null)
            additionalInfoType.setGraduateDegreeSought(graduateDegreeSought);
        List<KirschsteinBean> cvType = new ArrayList<>();
        List<KirschsteinBean> cvStart = new ArrayList<>();
        List<KirschsteinBean> cvEnd = new ArrayList<>();
        List<KirschsteinBean> cvLevel = new ArrayList<>();
        List<KirschsteinBean> cvGrant = new ArrayList<>();

        KirschsteinBean kbBean1;
        KirschsteinBean kbBean2 = null;
        KirschsteinBean kbBean3 = null;
        KirschsteinBean kbBean4 = null;
        KirschsteinBean kbBean5 = null;

        if (additionalInfoType.getCurrentPriorNRSASupportIndicator() != null) {
            if (additionalInfoType.getCurrentPriorNRSASupportIndicator().equals(YesNoDataType.Y_YES)) {
                KirschsteinBean kbBean;
                cvKirsch.sort(BY_QUESTION_NUMBER);
                for (KirschsteinBean aCvKirsch : cvKirsch) {
                    kbBean = aCvKirsch;
                    switch (kbBean.getQuestionId()) {
                        case PRE_OR_POST:
                            cvLevel.add(kbBean);
                            break;
                        case IND_OR_INST:
                            cvType.add(kbBean);
                            break;
                        case KIRST_START_DT:
                            cvStart.add(kbBean);
                            break;
                        case KIRST_END_DT:
                            cvEnd.add(kbBean);
                            break;
                        case KIRST_GRANT_NUM:
                            cvGrant.add(kbBean);
                            break;
                    }
                }
            }
            List<CurrentPriorNRSASupport> currentPriorNRSASupportList = new ArrayList<>();
            int numberRepeats = cvLevel.size();
            if (numberRepeats > 0) {
                for (int j = 0; j < numberRepeats; j++) {
                    kbBean1 = cvLevel.get(j);
                    if (cvType.size() - 1 >= j) {
                        kbBean2 = cvType.get(j);
                    }
                    if (cvStart.size() - 1 >= j) {
                        kbBean3 = cvStart.get(j);
                    }
                    if (cvEnd.size() - 1 >= j) {
                        kbBean4 = cvEnd.get(j);
                    }
                    if (cvGrant.size() - 1 >= j) {
                        kbBean5 = cvGrant.get(j);
                    }
                    CurrentPriorNRSASupport nrsaSupportType = CurrentPriorNRSASupport.Factory.newInstance();
                    if (kbBean1 != null) {
                        nrsaSupportType.setLevel(CurrentPriorNRSASupport.Level.Enum.forString(kbBean1.getAnswer()));
                    }
                    if (kbBean2 != null) {
                        nrsaSupportType.setType(CurrentPriorNRSASupport.Type.Enum.forString(kbBean2.getAnswer()));
                    }
                    if (kbBean3 != null && !kbBean3.getAnswer().equals(FieldValueConstants.VALUE_UNKNOWN)) {
                        nrsaSupportType.setStartDate(s2SDateTimeService.convertDateStringToCalendar(kbBean3.getAnswer()));
                    }
                    if (kbBean4 != null && !kbBean4.getAnswer().equals(FieldValueConstants.VALUE_UNKNOWN)) {
                        nrsaSupportType.setEndDate(s2SDateTimeService.convertDateStringToCalendar(kbBean4.getAnswer()));
                    }
                    if (kbBean5 != null) {
                        nrsaSupportType.setGrantNumber(kbBean5.getAnswer());
                    }
                    currentPriorNRSASupportList.add(nrsaSupportType);
                }

            }
            additionalInfoType.setCurrentPriorNRSASupportArray(currentPriorNRSASupportList.toArray(new CurrentPriorNRSASupport[0]));
        }
        phsFellowshipSupplemental.setBudget(createBudgetElements(hmBudgetQuestions));
        setAdditionalInformation(additionalInfoType);
    }

    protected Budget createBudgetElements(Map<Integer, String> budgetMap) {
        Budget budget = Budget.Factory.newInstance();
        budget.setTuitionAndFeesRequested(YesNoDataType.N_NO);
        getInstitutionalBaseSalary(budget, budgetMap);
        getFederalStipendRequested(budget);
        getSupplementationFromOtherSources(budget, budgetMap);
        setTuitionRequestedYears(budget);
        return budget;
    }

    /*
     * This method is used to get TuitionRequestedYears data to Budget XMLObject from List of BudgetLineItem based on CostElement
     * value of TUITION_COST_ELEMENTS
     */
    protected void setTuitionRequestedYears(Budget budget) {
        ProposalDevelopmentBudgetExtContract pBudget = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());
        if (pBudget == null) {
            return;
        }
        ScaleTwoDecimal tuitionTotal = ScaleTwoDecimal.ZERO;
        for (BudgetPeriodContract budgetPeriod : pBudget.getBudgetPeriods()) {
            ScaleTwoDecimal tuition = ScaleTwoDecimal.ZERO;
            for (BudgetLineItemContract budgetLineItem : budgetPeriod.getBudgetLineItems()) {
                if (getCostElementsByParam(ConfigurationConstants.TUITION_COST_ELEMENTS).contains(budgetLineItem.getCostElementBO().getCostElement())) {
                    tuition = tuition.add(budgetLineItem.getLineItemCost());
                }
            }
            tuitionTotal = tuitionTotal.add(tuition);
            switch (budgetPeriod.getBudgetPeriod()) {
                case 1:
                    budget.setTuitionRequestedYear1(tuition.bigDecimalValue());
                    break;
                case 2:
                    budget.setTuitionRequestedYear2(tuition.bigDecimalValue());
                    break;
                case 3:
                    budget.setTuitionRequestedYear3(tuition.bigDecimalValue());
                    break;
                case 4:
                    budget.setTuitionRequestedYear4(tuition.bigDecimalValue());
                    break;
                case 5:
                    budget.setTuitionRequestedYear5(tuition.bigDecimalValue());
                    break;
                case 6:
                    budget.setTuitionRequestedYear6(tuition.bigDecimalValue());
                    break;
                default:
                    break;
            }
        }
        budget.setTuitionRequestedTotal(tuitionTotal.bigDecimalValue());
        if (!tuitionTotal.equals(ScaleTwoDecimal.ZERO)) {
            budget.setTuitionAndFeesRequested(YesNoDataType.Y_YES);
        }
    }

    /*
     * This method is used to set data to SupplementationFromOtherSources XMLObject from budgetMap data for Budget
     */
    protected void getSupplementationFromOtherSources(Budget budget, Map<Integer, String> hmBudgetQuestions) {

        if (!hmBudgetQuestions.isEmpty()) {
            if (hmBudgetQuestions.get(OTHER_SUPP_SOURCE) != null) {
                if (hmBudgetQuestions.get(OTHER_SUPP_SOURCE).toUpperCase().equals(QUESTION_ANSWER_YES)) {
                    SupplementationFromOtherSources supplementationFromOtherSources = budget
                            .addNewSupplementationFromOtherSources();
                    if (hmBudgetQuestions.get(SUPP_SOURCE) != null) {
                        supplementationFromOtherSources.setSource(hmBudgetQuestions.get(SUPP_SOURCE));
                        supplementationFromOtherSources.setAmount(new BigDecimal(hmBudgetQuestions.get(SUPP_FUNDING_AMT)));
                        try {
                            supplementationFromOtherSources.setNumberOfMonths(new BigDecimal(hmBudgetQuestions.get(SUPP_MONTHS)));
                        } catch (Exception ex) {
                            LOG.error(ex);
                        }
                        supplementationFromOtherSources.setType(hmBudgetQuestions.get(SUPP_TYPE));

                    }
                }
            }
        }
    }

    protected void getFederalStipendRequested(PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.Budget budget) {
        PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.Budget.FederalStipendRequested federalStipendRequested = PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.Budget.FederalStipendRequested.Factory.newInstance();
        ProposalDevelopmentBudgetExtContract pBudget = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());
        if (pBudget != null) {
            ScaleTwoDecimal sumOfLineItemCost = ScaleTwoDecimal.ZERO;
            ScaleTwoDecimal numberOfMonths = ScaleTwoDecimal.ZERO;

            for (BudgetPeriodContract budgetPeriod : pBudget.getBudgetPeriods()) {
                if (budgetPeriod.getBudgetPeriod() == 1) {
                    for (BudgetLineItemContract budgetLineItem : budgetPeriod.getBudgetLineItems()) {
                        if (getCostElementsByParam(ConfigurationConstants.STIPEND_COST_ELEMENTS).contains(budgetLineItem.getCostElementBO().getCostElement())) {
                            sumOfLineItemCost = sumOfLineItemCost.add(budgetLineItem.getLineItemCost());

                            if (CollectionUtils.isNotEmpty(budgetLineItem.getBudgetPersonnelDetailsList())) {
                                for (BudgetPersonnelDetailsContract details : budgetLineItem.getBudgetPersonnelDetailsList()){
                                    final ScaleTwoDecimal detailsNumberOfMonths = details.getPercentEffort() != null ?
                                            getNumberOfMonths(details.getStartDate(), details.getEndDate()).multiply(details.getPercentEffort().multiply(POINT_ZERO_ONE)) :
                                            getNumberOfMonths(details.getStartDate(), details.getEndDate());

                                    numberOfMonths = numberOfMonths.add(detailsNumberOfMonths);
                                }
                            } else {
                                numberOfMonths = numberOfMonths.add(getNumberOfMonths(budgetLineItem.getStartDate(), budgetLineItem.getEndDate()));
                            }
                        }
                    }
                }
            }
            federalStipendRequested.setAmount(sumOfLineItemCost.bigDecimalValue());
            federalStipendRequested.setNumberOfMonths(numberOfMonths.bigDecimalValue());
            budget.setFederalStipendRequested(federalStipendRequested);

        }
    }

    /*
     * This method is used to set data to InstitutionalBaseSalary XMLObject from budgetMap data for Budget
     */
    protected void getInstitutionalBaseSalary(Budget budget, Map<Integer, String> budgetMap) {
        InstitutionalBaseSalary institutionalBaseSalary = InstitutionalBaseSalary.Factory.newInstance();
        if (budgetMap.get(SENIOR_FELL) != null && budgetMap.get(SENIOR_FELL).equals(YnqConstant.YES.code())) {
            if (budgetMap.get(BASE_SALARY) != null) {
                institutionalBaseSalary.setAmount(new BigDecimal(budgetMap.get(BASE_SALARY)));
            }
            if (budgetMap.get(ACAD_PERIOD) != null) {
                institutionalBaseSalary.setAcademicPeriod(PHSFellowshipSupplemental31Document.PHSFellowshipSupplemental31.Budget.InstitutionalBaseSalary.AcademicPeriod.Enum.forString(budgetMap.get(ACAD_PERIOD).toString()));
            }
            if (budgetMap.get(SALARY_MONTHS) != null) {
                institutionalBaseSalary.setNumberOfMonths(new BigDecimal(budgetMap.get(SALARY_MONTHS)));
            }
            budget.setInstitutionalBaseSalary(institutionalBaseSalary);
        }
    }

    /**
     * This method is used to set Narrative Data to ResearchTrainingPlan XMLObject based on NarrativeTypeCode.
     **/
    protected void setNarrativeDataForResearchTrainingPlan(PHSFellowshipSupplemental31 phsFellowshipSupplemental,
                                                           ResearchTrainingPlan researchTrainingPlan) {
        AttachedFileDataType attachedFileDataType;
        Sponsors sponsors = phsFellowshipSupplemental.addNewSponsors();
        OtherResearchTrainingPlan otherResearchTrainingPlan = null;
        PHSFellowshipSupplemental31.InstitutionalEnvironment institutionalEnvironment = phsFellowshipSupplemental.addNewInstitutionalEnvironment();
        PHSFellowshipSupplemental31.FellowshipApplicant fellowshipApplicant = phsFellowshipSupplemental.addNewFellowshipApplicant();
        PHSFellowshipSupplemental31.Introduction introduction = phsFellowshipSupplemental.addNewIntroduction();

        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            final String code = narrative.getNarrativeType().getCode();
            if (code != null) {
                if (code.equalsIgnoreCase(INTRODUCTION_TO_APPLICATION)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        IntroductionToApplication introductionToApplication = IntroductionToApplication.Factory.newInstance();
                        introductionToApplication.setAttFile(attachedFileDataType);
                        introduction.setIntroductionToApplication(introductionToApplication);
                    }
                } else if (code.equalsIgnoreCase(FELLOWSHIP_BACKGROUND_AND_GOALS)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        BackgroundandGoals backgroundandGoals = BackgroundandGoals.Factory.newInstance();
                        backgroundandGoals.setAttFile(attachedFileDataType);
                        fellowshipApplicant.setBackgroundandGoals(backgroundandGoals);
                    }
                } else if (code.equalsIgnoreCase(SPECIFIC_AIMS)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        ResearchTrainingPlan.SpecificAims specificAims = ResearchTrainingPlan.SpecificAims.Factory.newInstance();
                        specificAims.setAttFile(attachedFileDataType);
                        researchTrainingPlan.setSpecificAims(specificAims);
                    }
                } else if (code.equalsIgnoreCase(RESEARCH_STRATEGY)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        ResearchTrainingPlan.ResearchStrategy researchStrategy = ResearchTrainingPlan.ResearchStrategy.Factory.newInstance();
                        researchStrategy.setAttFile(attachedFileDataType);
                        researchTrainingPlan.setResearchStrategy(researchStrategy);
                    }
                } else if (code.equalsIgnoreCase(RESPECTIVE_CONTRIBUTIONS)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        ResearchTrainingPlan.RespectiveContribution respectiveContribution = ResearchTrainingPlan.RespectiveContribution.Factory.newInstance();
                        respectiveContribution.setAttFile(attachedFileDataType);
                        researchTrainingPlan.setRespectiveContribution(respectiveContribution);
                    }
                } else if (code.equalsIgnoreCase(SELECTION_OF_SPONSOR_AND_INSTITUTION)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        ResearchTrainingPlan.SponsorandInstitution sponsorAndInstitution = ResearchTrainingPlan.SponsorandInstitution.Factory
                                .newInstance();
                        sponsorAndInstitution.setAttFile(attachedFileDataType);
                        researchTrainingPlan.setSponsorandInstitution(sponsorAndInstitution);
                    }
                } else if (code.equalsIgnoreCase(PROGRESS_REPORT_PUBLICATION_LIST)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        ResearchTrainingPlan.ProgressReportPublicationList progressReportPublicationList = ProgressReportPublicationList.Factory
                                .newInstance();
                        progressReportPublicationList.setAttFile(attachedFileDataType);
                        researchTrainingPlan.setProgressReportPublicationList(progressReportPublicationList);
                    }
                } else if (code.equalsIgnoreCase(RESPONSIBLE_CONDUCT_OF_RESEARCH)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        ResearchTrainingPlan.TrainingInResponsibleConductOfResearch responsibleConductOfResearch = ResearchTrainingPlan.TrainingInResponsibleConductOfResearch.Factory
                                .newInstance();
                        responsibleConductOfResearch.setAttFile(attachedFileDataType);
                        researchTrainingPlan.setTrainingInResponsibleConductOfResearch(responsibleConductOfResearch);
                    }
                }
                // -- Sponsor(s), Collaborator(s), and Consultant(s) Section
                else if (code.equalsIgnoreCase(SPONSOR_COSPONSOR)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        Sponsors.SponsorAndCoSponsorStatements sponsorCosponsorInfo = sponsors.addNewSponsorAndCoSponsorStatements();
                        sponsorCosponsorInfo.setAttFile(attachedFileDataType);
                    }
                } else if (code.equalsIgnoreCase(LETTER_COLLAB_CONTRIB_CONSULT)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        Sponsors.LettersOfSupport lettersOfSupport = Sponsors.LettersOfSupport.Factory.newInstance();
                        lettersOfSupport.setAttFile(attachedFileDataType);
                        sponsors.setLettersOfSupport(lettersOfSupport);
                    }
                }
                // Institutional Environment and Commitment to Training Section
                else if (code.equalsIgnoreCase(PHS_FELLOW_INSTITUTION_ENVIRON_COMMITMENT)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        InstitutionalEnvironmentCommitmenttoTraining institutionalEnvironmentCommitmenttoTraining =
                                InstitutionalEnvironmentCommitmenttoTraining.Factory.newInstance();
                        institutionalEnvironmentCommitmenttoTraining.setAttFile(attachedFileDataType);
                        institutionalEnvironment.setInstitutionalEnvironmentCommitmenttoTraining(institutionalEnvironmentCommitmenttoTraining);
                    }
                }
                // Other research training plan section
                else if (code.equalsIgnoreCase(PROTECTION_OF_HUMAN_SUBJECTS)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        ProtectionOfHumanSubjects protectionOfHumanSubjects = ProtectionOfHumanSubjects.Factory.newInstance();
                        protectionOfHumanSubjects.setAttFile(attachedFileDataType);
                        if (otherResearchTrainingPlan == null) {
                            otherResearchTrainingPlan = phsFellowshipSupplemental.addNewOtherResearchTrainingPlan();
                        }
                        otherResearchTrainingPlan.setProtectionOfHumanSubjects(protectionOfHumanSubjects);
                    }
                } else if (code.equalsIgnoreCase(DATA_SAFETY_MONITORING_PLAN)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        DataSafetyMonitoringPlan dataSafetyMonitoringPlan = DataSafetyMonitoringPlan.Factory.newInstance();
                        dataSafetyMonitoringPlan.setAttFile(attachedFileDataType);
                        if (otherResearchTrainingPlan == null) {
                            otherResearchTrainingPlan = phsFellowshipSupplemental.addNewOtherResearchTrainingPlan();
                        }
                        otherResearchTrainingPlan.setDataSafetyMonitoringPlan(dataSafetyMonitoringPlan);
                    }
                } else if (code.equalsIgnoreCase(INCLUSION_OF_WOMEN_AND_MINORITIES)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        InclusionOfWomenAndMinorities inclusionOfWomenAndMinorities = InclusionOfWomenAndMinorities.Factory
                                .newInstance();
                        inclusionOfWomenAndMinorities.setAttFile(attachedFileDataType);
                        if (otherResearchTrainingPlan == null) {
                            otherResearchTrainingPlan = phsFellowshipSupplemental.addNewOtherResearchTrainingPlan();
                        }
                        otherResearchTrainingPlan.setInclusionOfWomenAndMinorities(inclusionOfWomenAndMinorities);
                    }
                } else if (code.equalsIgnoreCase(INCLUSION_OF_CHILDREN)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        InclusionOfChildren inclusionOfChildren = InclusionOfChildren.Factory.newInstance();
                        inclusionOfChildren.setAttFile(attachedFileDataType);
                        if (otherResearchTrainingPlan == null) {
                            otherResearchTrainingPlan = phsFellowshipSupplemental.addNewOtherResearchTrainingPlan();
                        }
                        otherResearchTrainingPlan.setInclusionOfChildren(inclusionOfChildren);
                    }
                }
            }
        }

        setMandatoryAttachmentsOnResearchTrainingPlan(researchTrainingPlan);
        setMandatoryAttachmentsOnFellowshipApplicant(fellowshipApplicant);
    }

    private void setMandatoryAttachmentsOnFellowshipApplicant(PHSFellowshipSupplemental31.FellowshipApplicant fellowshipApplicant) {
        if(fellowshipApplicant.getBackgroundandGoals() == null) {
            fellowshipApplicant.setBackgroundandGoals(BackgroundandGoals.Factory.newInstance());
        }
    }

    private void setMandatoryAttachmentsOnResearchTrainingPlan(ResearchTrainingPlan researchTrainingPlan) {
        if(researchTrainingPlan.getSpecificAims() == null) {
            researchTrainingPlan.setSpecificAims(ResearchTrainingPlan.SpecificAims.Factory.newInstance());
        }
        if(researchTrainingPlan.getResearchStrategy() == null) {
            researchTrainingPlan.setResearchStrategy(ResearchTrainingPlan.ResearchStrategy.Factory.newInstance());
        }
        if(researchTrainingPlan.getRespectiveContribution() == null) {
            researchTrainingPlan.setRespectiveContribution(ResearchTrainingPlan.RespectiveContribution.Factory.newInstance());
        }
        if(researchTrainingPlan.getSponsorandInstitution() == null) {
            researchTrainingPlan.setSponsorandInstitution(ResearchTrainingPlan.SponsorandInstitution.Factory.newInstance());
        }
        if(researchTrainingPlan.getTrainingInResponsibleConductOfResearch() == null) {
            researchTrainingPlan.setTrainingInResponsibleConductOfResearch(ResearchTrainingPlan.TrainingInResponsibleConductOfResearch.Factory.newInstance());
        }
    }



    /**
     * This method is used to set HumanSubjectInvoved and VertebrateAnimalUsed XMLObject Data.
     *
     * @param researchTrainingPlan
     */
    protected void setHumanSubjectInvolvedAndVertebrateAnimalUsed(OtherResearchTrainingPlan researchTrainingPlan) {
        researchTrainingPlan.setHumanSubjectsInvolved(YesNoDataType.N_NO);
        researchTrainingPlan.setVertebrateAnimalsUsed(YesNoDataType.N_NO);
        for (ProposalSpecialReviewContract propSpecialReview : pdDoc.getDevelopmentProposal().getPropSpecialReviews()) {
            if (SPECIAL_REVIEW_HUMAN_SUBJECTS_CODE.equals(propSpecialReview.getSpecialReviewType().getCode())) {
                researchTrainingPlan.setHumanSubjectsInvolved(YesNoDataType.Y_YES);
            } else if (SPECIAL_REVIEW_VERTEBRATE_ANIMALS_CODE.equals(propSpecialReview.getSpecialReviewType().getCode())) {
                researchTrainingPlan.setVertebrateAnimalsUsed(YesNoDataType.Y_YES);
            }
        }
    }

    /*
     * This method is used to set additional information data to AdditionalInformation XMLObject from DevelopmentProposal,
     * ProposalYnq
     */
    private void setAdditionalInformation(AdditionalInformation additionalInformation) {
        setCitizenshipAndAlternatePhoneNumber(additionalInformation);
        setAdditionalInformationConcurrentSupport(additionalInformation);
    }

    private void setCitizenshipAndAlternatePhoneNumber(AdditionalInformation additionalInformation) {
        ProposalPersonContract principalInvestigator = s2SProposalPersonService.getPrincipalInvestigator(pdDoc);
        for (ProposalPersonContract proposalPerson : pdDoc.getDevelopmentProposal().getProposalPersons()) {
            if (proposalPerson.isPrincipalInvestigator()) {
                CitizenshipType citizenShip = s2SProposalPersonService.getCitizenship(proposalPerson);
                if(citizenShip!=null && StringUtils.isNotBlank(citizenShip.getCitizenShip())){
                    if (citizenShip.getCitizenShip().trim().equals(CitizenshipDataType.NON_U_S_CITIZEN_WITH_TEMPORARY_VISA.toString())) {
                        additionalInformation.setUSCitizen(YesNoDataType.N_NO);
                        additionalInformation.setNonUSCitizen(NonUSCitizenDataType.WITH_A_TEMPORARY_U_S_VISA);
                    }
                    else if (citizenShip.getCitizenShip().trim().equals(CitizenshipDataType.PERMANENT_RESIDENT_OF_U_S.toString())) {
                        additionalInformation.setUSCitizen(YesNoDataType.N_NO);
                        additionalInformation.setNonUSCitizen(NonUSCitizenDataType.WITH_A_PERMANENT_U_S_RESIDENT_VISA);
                    }
                    else if (citizenShip.getCitizenShip().trim().equals(
                            CitizenshipDataType.U_S_CITIZEN_OR_NONCITIZEN_NATIONAL.toString())) {
                        additionalInformation.setUSCitizen(YesNoDataType.Y_YES);
                    }
                    else if (citizenShip.getCitizenShip().trim().equals(
                            TEMPORARY_VISA_ALSO_APPLIED_FOR_PERMANENT_RESIDENT_STATUS)) {
                        additionalInformation.setUSCitizen(YesNoDataType.N_NO);
                        additionalInformation.setNonUSCitizen(NonUSCitizenDataType.WITH_A_TEMPORARY_U_S_VISA);
                        additionalInformation.setPermanentResidentByAwardIndicator(YesNoDataType.Y_YES);
                    }
                }
            }
        }
        if (additionalInformation.getUSCitizen() == null && additionalInformation.getNonUSCitizen() == null) {
            additionalInformation.setUSCitizen(YesNoDataType.N_NO);
            additionalInformation.setNonUSCitizen(null);
        }

        if (principalInvestigator != null && StringUtils.isNotEmpty(principalInvestigator.getMobilePhoneNumber())) {
            additionalInformation.setAlernatePhoneNumber(principalInvestigator.getMobilePhoneNumber());
        }
    }

    private void setAdditionalInformationConcurrentSupport(AdditionalInformation additionalInformation) {
        additionalInformation.setConcurrentSupport(YesNoDataType.N_NO);
        AttachedFileDataType attachedFileDataType;
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            final String code = narrative.getNarrativeType().getCode();
            if (code != null) {
                if(code.equalsIgnoreCase(CONCURRENT_SUPPORT)) {
                    attachedFileDataType = addAttachedFileType(narrative);
                    if (attachedFileDataType != null) {
                        ConcurrentSupportDescription concurrentSupportDescription = ConcurrentSupportDescription.Factory
                                .newInstance();
                        concurrentSupportDescription.setAttFile(attachedFileDataType);
                        additionalInformation.setConcurrentSupport(YesNoDataType.Y_YES);
                        additionalInformation.setConcurrentSupportDescription(concurrentSupportDescription);
                    }
                }
            }
        }
    }

    @Override
    public PHSFellowshipSupplemental31Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getPHSFellowshipSupplemental31();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(PHSFellowshipSupplemental31Document form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {

            if (form.getPHSFellowshipSupplemental31().getIntroduction() != null) {
                final PHSFellowshipSupplemental31.Introduction introduction = form.getPHSFellowshipSupplemental31().getIntroduction();
                final PHSFellowshipSupplemental31.Introduction.IntroductionToApplication introductionToApplication = introduction.getIntroductionToApplication();
                if (introductionToApplication != null && introductionToApplication.getAttFile() != null && a.getContentId().equals(introductionToApplication.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.optionalFile0", a);
                }
            }

            if (form.getPHSFellowshipSupplemental31().getFellowshipApplicant() != null) {
                final PHSFellowshipSupplemental31.FellowshipApplicant fellowshipApplicant = form.getPHSFellowshipSupplemental31().getFellowshipApplicant();
                final PHSFellowshipSupplemental31.FellowshipApplicant.BackgroundandGoals backgroundandGoals = fellowshipApplicant.getBackgroundandGoals();
                if (backgroundandGoals != null && backgroundandGoals.getAttFile() != null && a.getContentId().equals(backgroundandGoals.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.mandatoryFile0", a);
                }
            }

            if (form.getPHSFellowshipSupplemental31().getResearchTrainingPlan() != null) {
                final ResearchTrainingPlan researchTrainingPlan = form.getPHSFellowshipSupplemental31().getResearchTrainingPlan();

                final ResearchTrainingPlan.SpecificAims specificAims = researchTrainingPlan.getSpecificAims();
                if (specificAims != null && specificAims.getAttFile() != null && a.getContentId().equals(specificAims.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.mandatoryFile1", a);
                }

                final ResearchTrainingPlan.ResearchStrategy researchStrategy = researchTrainingPlan.getResearchStrategy();
                if (researchStrategy != null && researchStrategy.getAttFile() != null && a.getContentId().equals(researchStrategy.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.mandatoryFile2", a);
                }

                final ResearchTrainingPlan.RespectiveContribution respectiveContribution = researchTrainingPlan.getRespectiveContribution();
                if (respectiveContribution != null && respectiveContribution.getAttFile() != null && a.getContentId().equals(respectiveContribution.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.mandatoryFile3", a);
                }

                final ResearchTrainingPlan.SponsorandInstitution sponsorandInstitution = researchTrainingPlan.getSponsorandInstitution();
                if (sponsorandInstitution != null && sponsorandInstitution.getAttFile() != null && a.getContentId().equals(sponsorandInstitution.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.mandatoryFile4", a);
                }

                final ResearchTrainingPlan.ProgressReportPublicationList progressReportPublicationList = researchTrainingPlan.getProgressReportPublicationList();
                if (progressReportPublicationList != null && progressReportPublicationList.getAttFile() != null && a.getContentId().equals(progressReportPublicationList.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.optionalFile1", a);
                }

                final ResearchTrainingPlan.TrainingInResponsibleConductOfResearch trainingInResponsibleConductOfResearch = researchTrainingPlan.getTrainingInResponsibleConductOfResearch();
                if (trainingInResponsibleConductOfResearch != null && trainingInResponsibleConductOfResearch.getAttFile() != null && a.getContentId().equals(trainingInResponsibleConductOfResearch.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.mandatoryFile5", a);
                }
            }

            if (form.getPHSFellowshipSupplemental31().getSponsors() != null) {
                final Sponsors sponsors = form.getPHSFellowshipSupplemental31().getSponsors();
                final Sponsors.SponsorAndCoSponsorStatements sponsorAndCoSponsorStatements = sponsors.getSponsorAndCoSponsorStatements();
                if (sponsorAndCoSponsorStatements != null && sponsorAndCoSponsorStatements.getAttFile() != null && a.getContentId().equals(sponsorAndCoSponsorStatements.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.optionalFile2", a);
                }

                final Sponsors.LettersOfSupport lettersOfSupport = sponsors.getLettersOfSupport();
                if (lettersOfSupport != null && lettersOfSupport.getAttFile() != null && a.getContentId().equals(lettersOfSupport.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.optionalFile3", a);
                }
            }

            if (form.getPHSFellowshipSupplemental31().getInstitutionalEnvironment() != null) {
                final PHSFellowshipSupplemental31.InstitutionalEnvironment institutionalEnvironment = form.getPHSFellowshipSupplemental31().getInstitutionalEnvironment();
                final PHSFellowshipSupplemental31.InstitutionalEnvironment.InstitutionalEnvironmentCommitmenttoTraining institutionalEnvironmentCommitmenttoTraining = institutionalEnvironment.getInstitutionalEnvironmentCommitmenttoTraining();
                if (institutionalEnvironmentCommitmenttoTraining != null && institutionalEnvironmentCommitmenttoTraining.getAttFile() != null && a.getContentId().equals(institutionalEnvironmentCommitmenttoTraining.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.optionalFile4", a);
                }
            }

            if (form.getPHSFellowshipSupplemental31().getOtherResearchTrainingPlan() != null) {
                final OtherResearchTrainingPlan otherResearchTrainingPlan = form.getPHSFellowshipSupplemental31().getOtherResearchTrainingPlan();
                final OtherResearchTrainingPlan.ProtectionOfHumanSubjects protectionOfHumanSubjects = otherResearchTrainingPlan.getProtectionOfHumanSubjects();
                if (protectionOfHumanSubjects != null && protectionOfHumanSubjects.getAttFile() != null && a.getContentId().equals(protectionOfHumanSubjects.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.optionalFile5", a);
                }

                final OtherResearchTrainingPlan.DataSafetyMonitoringPlan dataSafetyMonitoringPlan = otherResearchTrainingPlan.getDataSafetyMonitoringPlan();
                if (dataSafetyMonitoringPlan != null && dataSafetyMonitoringPlan.getAttFile() != null && a.getContentId().equals(dataSafetyMonitoringPlan.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.optionalFile6", a);
                }

                final OtherResearchTrainingPlan.InclusionOfWomenAndMinorities inclusionOfWomenAndMinorities = otherResearchTrainingPlan.getInclusionOfWomenAndMinorities();
                if (inclusionOfWomenAndMinorities != null && inclusionOfWomenAndMinorities.getAttFile() != null && a.getContentId().equals(inclusionOfWomenAndMinorities.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.optionalFile7", a);
                }

                final OtherResearchTrainingPlan.InclusionOfChildren inclusionOfChildren = otherResearchTrainingPlan.getInclusionOfChildren();
                if (inclusionOfChildren != null && inclusionOfChildren.getAttFile() != null && a.getContentId().equals(inclusionOfChildren.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P1.optionalFile8", a);
                }

                final OtherResearchTrainingPlan.VertebrateAnimals vertebrateAnimals = otherResearchTrainingPlan.getVertebrateAnimals();
                if (vertebrateAnimals != null && vertebrateAnimals.getAttFile() != null && a.getContentId().equals(vertebrateAnimals.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P2.optionalFile0", a);
                }

                final OtherResearchTrainingPlan.SelectAgentResearch selectAgentResearch = otherResearchTrainingPlan.getSelectAgentResearch();
                if (selectAgentResearch != null && selectAgentResearch.getAttFile() != null && a.getContentId().equals(selectAgentResearch.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P2.optionalFile1", a);
                }

                final OtherResearchTrainingPlan.ResourceSharingPlan resourceSharingPlan = otherResearchTrainingPlan.getResourceSharingPlan();
                if (resourceSharingPlan != null && resourceSharingPlan.getAttFile() != null && a.getContentId().equals(resourceSharingPlan.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P2.optionalFile2", a);
                }

                final OtherResearchTrainingPlan.KeyBiologicalAndOrChemicalResources keyBiologicalAndOrChemicalResources = otherResearchTrainingPlan.getKeyBiologicalAndOrChemicalResources();
                if (keyBiologicalAndOrChemicalResources != null && keyBiologicalAndOrChemicalResources.getAttFile() != null && a.getContentId().equals(keyBiologicalAndOrChemicalResources.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P2.optionalFile3", a);
                }
            }

            if (form.getPHSFellowshipSupplemental31().getAdditionalInformation() != null) {
                final AdditionalInformation additionalInformation = form.getPHSFellowshipSupplemental31().getAdditionalInformation();
                final AdditionalInformation.ConcurrentSupportDescription concurrentSupportDescription = additionalInformation.getConcurrentSupportDescription();
                if (concurrentSupportDescription != null && concurrentSupportDescription.getAttFile() != null && a.getContentId().equals(concurrentSupportDescription.getAttFile().getFileLocation().getHref())) {
                    return entry("PHS_Fellowship_Supplemental_3_1_P2.optionalFile4", a);
                }
            }

            if (form.getPHSFellowshipSupplemental31().getAppendix() != null && form.getPHSFellowshipSupplemental31().getAppendix().getAttachedFileList() != null) {
                final List<AttachedFileDataType> atts = form.getPHSFellowshipSupplemental31().getAppendix().getAttachedFileList();
                for (int i = 0; i < atts.size(); i++) {
                    final AttachedFileDataType att = atts.get(i);

                    if (att != null && a.getContentId().equals(att.getFileLocation().getHref())){
                        return entry("PHS_Fellowship_Supplemental_3_1_P2.optionalFile_5_" + i, a);
                    }
                }
            }

            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<PHSFellowshipSupplemental31Document> factory() {
        return PHSFellowshipSupplemental31Document.Factory;
    }
}
