/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.rrBudget20V20.BudgetTypeDataType;
import gov.grants.apply.forms.rrBudget20V20.BudgetYearDataType;
import gov.grants.apply.forms.rrBudget20V20.BudgetYearDataType.*;
import gov.grants.apply.forms.rrBudget20V20.BudgetYearDataType.Equipment.EquipmentList;
import gov.grants.apply.forms.rrBudget20V20.BudgetYearDataType.KeyPersons.KeyPerson;
import gov.grants.apply.forms.rrBudget20V20.BudgetYearDataType.OtherPersonnel.*;
import gov.grants.apply.forms.rrBudget20V20.RRBudget20Document;
import gov.grants.apply.forms.rrBudget20V20.RRBudget20Document.RRBudget20;
import gov.grants.apply.forms.rrBudget20V20.RRBudget20Document.RRBudget20.BudgetSummary;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType.FileLocation;
import gov.grants.apply.system.globalV10.HashValueDocument;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.core.DevelopmentProposalContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.s2sgen.api.budget.*;
import org.kuali.coeus.s2sgen.api.core.InfrastructureConstants;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.util.UnsupportedOtherCostsErrorBuilder;
import org.kuali.coeus.sys.api.model.ScaleTwoDecimal;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;


@FormGenerator("RRBudgetV2_0Generator")
public class RRBudgetV2_0Generator extends RRBudgetBaseGenerator<RRBudget20Document> implements S2SFormGeneratorPdfFillable<RRBudget20Document> {

    private static final Logger LOG = LogManager.getLogger(RRBudgetV2_0Generator.class);
    private static final int SUPPORTED_OTHER_COSTS = 3;

    @Value("http://apply.grants.gov/forms/RR_Budget_2_0-V2.0")
    private String namespace;

    @Value("RR_Budget_2_0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/RR_Budget-V2.0.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/RR_Budget_2_0-V2.0.pdf")
    private Resource pdfForm;

    @Value("165")
    private int sortIndex;

    private RRBudget20Document getRRBudget20() {

        deleteAutoGenNarratives();
        RRBudget20Document rrBudgetDocument = RRBudget20Document.Factory
                .newInstance();
        RRBudget20 rrBudget = RRBudget20.Factory.newInstance();
        rrBudget.setFormVersion(FormVersion.v2_0.getVersion());
        if (pdDoc.getDevelopmentProposal().getApplicantOrganization() != null) {
            rrBudget.setSAMUEI(pdDoc.getDevelopmentProposal()
                .getApplicantOrganization().getOrganization()
                .getUei());
            rrBudget.setOrganizationName(StringUtils.substring(pdDoc.getDevelopmentProposal().getApplicantOrganization().getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));

        }
        rrBudget.setBudgetType(BudgetTypeDataType.PROJECT);

        validateBudgetForForm(pdDoc);
        final S2SBudgetDto budgetSummary = s2sBudgetInfoService.getBudgetInfo(pdDoc);
        final List<S2SBudgetPeriodDto> budgetPeriodList = budgetSummary.getBudgetPeriods();

        rrBudget.setBudgetSummary(getBudgetSummary(budgetSummary));

        for (S2SBudgetPeriodDto budgetPeriodData : budgetPeriodList) {
            setBudgetYearDataType(rrBudget, budgetPeriodData);
        }
        AttachedFileDataType attachedFileDataType;
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            if (narrative.getNarrativeType().getCode() != null
                    && Integer.parseInt(narrative.getNarrativeType().getCode()) == 132) {
                attachedFileDataType = addAttachedFileType(narrative);
                if (attachedFileDataType != null) {
                    break;
                }
            }
        }
        rrBudget.setBudgetJustificationAttachment(getBudgetJustification());
        rrBudgetDocument.setRRBudget20(rrBudget);
        return rrBudgetDocument;
    }

    private void setBudgetYearDataType(RRBudget20 rrBudget, S2SBudgetPeriodDto periodInfo) {

        BudgetYearDataType budgetYear = rrBudget.addNewBudgetYear();
        if (periodInfo != null) {
            budgetYear.setBudgetPeriodStartDate(s2SDateTimeService.convertDateToCalendar(periodInfo.getStartDate()));
            budgetYear.setBudgetPeriodEndDate(s2SDateTimeService.convertDateToCalendar(periodInfo.getEndDate()));
            budgetYear.setKeyPersons(getKeyPersons(periodInfo));
            budgetYear.setOtherPersonnel(getOtherPersonnel(periodInfo));
            if (periodInfo.getTotalCompensation() != null) {
                budgetYear.setTotalCompensation(periodInfo
                        .getTotalCompensation().bigDecimalValue());
            }
            budgetYear.setEquipment(getEquipment(periodInfo));
            budgetYear.setTravel(getTravel(periodInfo));
            budgetYear
                    .setParticipantTraineeSupportCosts(getParticipantTraineeSupportCosts(periodInfo));
            budgetYear.setOtherDirectCosts(getOtherDirectCosts(periodInfo));
            BigDecimal directCosts = periodInfo.getDirectCostsTotal()
                    .bigDecimalValue();
            budgetYear.setDirectCosts(directCosts);
            IndirectCosts indirectCosts = getIndirectCosts(periodInfo);
            if (indirectCosts != null) {
                budgetYear.setIndirectCosts(indirectCosts);
                budgetYear.setTotalCosts(periodInfo.getDirectCostsTotal().bigDecimalValue().add(indirectCosts.getTotalIndirectCosts()));
            } else {
                budgetYear.setTotalCosts(periodInfo.getDirectCostsTotal().bigDecimalValue());
            }

            budgetYear.setTotalCostsFee(budgetYear.getFee() != null ? budgetYear.getFee().add(budgetYear.getTotalCosts()) : budgetYear.getTotalCosts());
            budgetYear.setCognizantFederalAgency(periodInfo
                    .getCognizantFedAgency());
        }
    }

    private BudgetSummary getBudgetSummary(S2SBudgetDto budgetSummaryData) {

        BudgetSummary budgetSummary = BudgetSummary.Factory.newInstance();
        S2SOtherDirectCostInfoDto otherDirectCosts = null;
        if (budgetSummaryData != null) {
            if (budgetSummaryData.getOtherDirectCosts() != null && budgetSummaryData.getOtherDirectCosts().size() > 0) {
                otherDirectCosts = budgetSummaryData.getOtherDirectCosts().get(0);
            }
            if (otherDirectCosts != null) {

                budgetSummary.setCumulativeTotalFundsRequestedSeniorKeyPerson(BigDecimal.ZERO);
                budgetSummary.setCumulativeTotalFundsRequestedPersonnel(BigDecimal.ZERO);

                if (budgetSummaryData.getCumTotalFundsForSrPersonnel() != null) {
                    budgetSummary
                            .setCumulativeTotalFundsRequestedSeniorKeyPerson(budgetSummaryData
                                    .getCumTotalFundsForSrPersonnel().bigDecimalValue());
                }
                if (budgetSummaryData.getCumTotalFundsForOtherPersonnel() != null
                        && budgetSummaryData.getCumTotalFundsForOtherPersonnel().isGreaterThan(ScaleTwoDecimal.ZERO)) {
                    budgetSummary
                            .setCumulativeTotalFundsRequestedOtherPersonnel(budgetSummaryData
                                    .getCumTotalFundsForOtherPersonnel()
                                    .bigDecimalValue());
                }
                if (budgetSummaryData.getCumNumOtherPersonnel() != null) {
                    budgetSummary.setCumulativeTotalNoOtherPersonnel(budgetSummaryData
                            .getCumNumOtherPersonnel().intValue());
                }
                if (budgetSummaryData.getCumTotalFundsForPersonnel() != null) {
                    budgetSummary.setCumulativeTotalFundsRequestedPersonnel(budgetSummaryData.getCumTotalFundsForPersonnel().bigDecimalValue());
                }
                budgetSummary.setCumulativeTotalFundsRequestedEquipment(budgetSummaryData.getCumEquipmentFunds().bigDecimalValue());
                budgetSummary.setCumulativeTotalFundsRequestedTravel(budgetSummaryData.getCumTravel().bigDecimalValue());
                budgetSummary.setCumulativeDomesticTravelCosts(budgetSummaryData.getCumDomesticTravel().bigDecimalValue());
                budgetSummary.setCumulativeForeignTravelCosts(budgetSummaryData.getCumForeignTravel().bigDecimalValue());
                budgetSummary.setCumulativeTotalFundsRequestedTraineeCosts(budgetSummaryData.getPartOtherCost()
                        .add(budgetSummaryData.getPartStipendCost()
                                .add(budgetSummaryData.getPartTravelCost()
                                        .add(budgetSummaryData.getPartTuition()
                                                .add(budgetSummaryData.getPartSubsistence())))).bigDecimalValue());
                budgetSummary.setCumulativeTraineeStipends(otherDirectCosts.getPartStipends().bigDecimalValue());
                budgetSummary.setCumulativeTraineeSubsistence(otherDirectCosts.getPartSubsistence().bigDecimalValue());
                budgetSummary.setCumulativeTraineeTravel(otherDirectCosts.getPartTravel().bigDecimalValue());
                budgetSummary.setCumulativeTraineeTuitionFeesHealthInsurance(otherDirectCosts.getPartTuition().bigDecimalValue());
                budgetSummary.setCumulativeOtherTraineeCost(budgetSummaryData.getPartOtherCost().bigDecimalValue());
                budgetSummary.setCumulativeNoofTrainees(budgetSummaryData.getParticipantCount());
                budgetSummary.setCumulativeTotalFundsRequestedOtherDirectCosts(otherDirectCosts.getTotalOtherDirect().bigDecimalValue());
                budgetSummary.setCumulativeMaterialAndSupplies(otherDirectCosts.getMaterials().bigDecimalValue());
                budgetSummary.setCumulativePublicationCosts(otherDirectCosts.getPublications().bigDecimalValue());
                budgetSummary.setCumulativeConsultantServices(otherDirectCosts.getConsultants().bigDecimalValue());
                budgetSummary.setCumulativeADPComputerServices(otherDirectCosts.getComputer().bigDecimalValue());
                budgetSummary.setCumulativeSubawardConsortiumContractualCosts(otherDirectCosts.getSubAwards().bigDecimalValue());
                budgetSummary.setCumulativeEquipmentFacilityRentalFees(otherDirectCosts.getEquipRental().bigDecimalValue());
                budgetSummary.setCumulativeAlterationsAndRenovations(otherDirectCosts.getAlterations().bigDecimalValue());
                List<S2SOtherCostDto> cvOthers = otherDirectCosts.getOtherCosts();
                if (cvOthers.size() > SUPPORTED_OTHER_COSTS) {
                    getAuditErrors().add(UnsupportedOtherCostsErrorBuilder.build(SUPPORTED_OTHER_COSTS));
                } else {
                    for (int j = 0; j < cvOthers.size(); j++) {
                        S2SOtherCostDto hmCosts = cvOthers.get(j);
                        if (j == 0) {
                            budgetSummary.setCumulativeOther1DirectCost(hmCosts.getCost().bigDecimalValue());
                        } else if (j == 1) {
                            budgetSummary.setCumulativeOther2DirectCost(hmCosts.getCost().bigDecimalValue());
                        } else {
                            budgetSummary.setCumulativeOther3DirectCost(hmCosts.getCost().bigDecimalValue());
                        }
                    }
                }
                budgetSummary.setCumulativeTotalFundsRequestedDirectCosts(budgetSummaryData
                        .getCumTotalDirectCosts().bigDecimalValue());
                budgetSummary.setCumulativeTotalFundsRequestedIndirectCost(budgetSummaryData
                        .getCumTotalIndirectCosts().bigDecimalValue());
                budgetSummary.setCumulativeTotalFundsRequestedDirectIndirectCosts(budgetSummaryData
                        .getCumTotalCosts().bigDecimalValue());
                if (budgetSummaryData.getCumFee() != null) {
                    budgetSummary.setCumulativeFee(budgetSummaryData.getCumFee()
                            .bigDecimalValue());
                }
                budgetSummary.setCumulativeTotalCostsFee(budgetSummary.getCumulativeFee() != null ?
                        budgetSummary.getCumulativeFee().add(budgetSummary.getCumulativeTotalFundsRequestedDirectIndirectCosts()) :
                        budgetSummary.getCumulativeTotalFundsRequestedDirectIndirectCosts());
            }
        }
        return budgetSummary;
    }

    private ParticipantTraineeSupportCosts getParticipantTraineeSupportCosts(
            S2SBudgetPeriodDto periodInfo) {

        ParticipantTraineeSupportCosts traineeSupportCosts = ParticipantTraineeSupportCosts.Factory
                .newInstance();
        if (periodInfo != null) {
            traineeSupportCosts.setTuitionFeeHealthInsurance(periodInfo
                    .getPartTuition().bigDecimalValue());
            traineeSupportCosts.setStipends(periodInfo.getPartStipendCost()
                    .bigDecimalValue());
            traineeSupportCosts.setTravel(periodInfo.getPartTravelCost()
                    .bigDecimalValue());
            traineeSupportCosts.setSubsistence(periodInfo.getPartSubsistence()
                    .bigDecimalValue());
            traineeSupportCosts.setOther(getOtherPTSupportCosts(periodInfo));
            traineeSupportCosts.setParticipantTraineeNumber(periodInfo
                    .getParticipantCount());
            traineeSupportCosts.setTotalCost(traineeSupportCosts.getTuitionFeeHealthInsurance()
                    .add(traineeSupportCosts.getStipends().add(traineeSupportCosts.getTravel()
                            .add(traineeSupportCosts.getSubsistence().add(traineeSupportCosts.getOther().getCost())))));
        }
        return traineeSupportCosts;
    }

    private ParticipantTraineeSupportCosts.Other getOtherPTSupportCosts(S2SBudgetPeriodDto periodInfo) {
        ParticipantTraineeSupportCosts.Other other = ParticipantTraineeSupportCosts.Other.Factory.newInstance();
        other.setDescription(OTHERCOST_DESCRIPTION);
        ScaleTwoDecimal otherCost = ScaleTwoDecimal.ZERO;
        if (periodInfo != null && periodInfo.getPartOtherCost() != null) {
            otherCost = periodInfo.getPartOtherCost();
        }
        other.setCost(otherCost.bigDecimalValue());
        return other;
    }

    private OtherDirectCosts getOtherDirectCosts(S2SBudgetPeriodDto periodInfo) {

        OtherDirectCosts otherDirectCosts = OtherDirectCosts.Factory
                .newInstance();
        if (periodInfo != null && periodInfo.getOtherDirectCosts().size() > 0) {
            if (periodInfo.getOtherDirectCosts().get(0).getPublications() != null) {
                otherDirectCosts.setPublicationCosts(periodInfo
                        .getOtherDirectCosts().get(0).getPublications()
                        .bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getMaterials() != null) {
                otherDirectCosts.setMaterialsSupplies(periodInfo
                        .getOtherDirectCosts().get(0).getMaterials()
                        .bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getConsultants() != null) {
                otherDirectCosts.setConsultantServices(periodInfo
                        .getOtherDirectCosts().get(0).getConsultants()
                        .bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getComputer() != null) {
                otherDirectCosts.setADPComputerServices(periodInfo
                        .getOtherDirectCosts().get(0).getComputer()
                        .bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getSubAwards() != null) {
                otherDirectCosts
                        .setSubawardConsortiumContractualCosts(periodInfo
                                .getOtherDirectCosts().get(0).getSubAwards()
                                .bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getAlterations() != null) {
                otherDirectCosts.setAlterationsRenovations(periodInfo
                        .getOtherDirectCosts().get(0).getAlterations()
                        .bigDecimalValue());
            }
            if (periodInfo.getOtherDirectCosts().get(0).getEquipRental() != null) {
                otherDirectCosts.setEquipmentRentalFee(periodInfo
                        .getOtherDirectCosts().get(0).getEquipRental()
                        .bigDecimalValue());
            }
            setOthersForOtherDirectCosts(otherDirectCosts, periodInfo);
            if (periodInfo.getOtherDirectCosts().get(0).getTotalOtherDirect() != null) {
                otherDirectCosts.setTotalOtherDirectCost(periodInfo
                        .getOtherDirectCosts().get(0).getTotalOtherDirect()
                        .bigDecimalValue());
            }
        }
        return otherDirectCosts;
    }

    private IndirectCosts getIndirectCosts(S2SBudgetPeriodDto periodInfo) {

        IndirectCosts indirectCosts = null;

        if (periodInfo != null
                && periodInfo.getIndirectCosts() != null
                && periodInfo.getIndirectCosts().getIndirectCostDetails() != null) {

            List<IndirectCosts.IndirectCost> indirectCostList = new ArrayList<>();
            int IndirectCostCount = 0;
            for (S2SIndirectCostDetailsDto indirectCostDetails : periodInfo
                    .getIndirectCosts().getIndirectCostDetails()) {
                IndirectCosts.IndirectCost indirectCost = IndirectCosts.IndirectCost.Factory
                        .newInstance();
                if (indirectCostDetails.getBase() != null) {
                    indirectCost.setBase(indirectCostDetails.getBase()
                            .bigDecimalValue());
                }
                indirectCost.setCostType(indirectCostDetails.getCostType());
                if (indirectCostDetails.getFunds() != null) {
                    indirectCost.setFundRequested(indirectCostDetails
                            .getFunds().bigDecimalValue());
                }
                if (indirectCostDetails.getRate() != null) {
                    indirectCost.setRate(indirectCostDetails.getRate()
                            .bigDecimalValue());
                }
                indirectCostList.add(indirectCost);
                IndirectCostCount++;
                if (IndirectCostCount == ARRAY_LIMIT_IN_SCHEMA) {
                    LOG.warn("Stopping iteration over indirect cost details because array limit in schema is only 4");
                    break;
                }
            }
            if (IndirectCostCount > 0) {
                indirectCosts = IndirectCosts.Factory.newInstance();

                indirectCosts.setIndirectCostArray(indirectCostList
                        .toArray(new IndirectCosts.IndirectCost[0]));
                if (periodInfo.getIndirectCosts().getTotalIndirectCosts() != null) {
                    indirectCosts.setTotalIndirectCosts(periodInfo
                            .getIndirectCosts().getTotalIndirectCosts()
                            .bigDecimalValue());
                }
            }
        }
        return indirectCosts;
    }

    private void setOthersForOtherDirectCosts(OtherDirectCosts otherDirectCosts, S2SBudgetPeriodDto periodInfo) {
        if (periodInfo != null && periodInfo.getOtherDirectCosts() != null) {
            for (S2SOtherDirectCostInfoDto otherDirectCostInfo : periodInfo.getOtherDirectCosts()) {
                if (CollectionUtils.isNotEmpty(otherDirectCostInfo.getOtherCosts())) {
                    for (S2SOtherCostDto oc : otherDirectCostInfo.getOtherCosts()) {
                        final OtherDirectCosts.Other other = otherDirectCosts.addNewOther();
                        other.setCost(oc.getCost().bigDecimalValue());
                        other.setDescription(oc.getDescription());
                    }
                }
            }
        }
    }

    private Travel getTravel(S2SBudgetPeriodDto periodInfo) {

        Travel travel = Travel.Factory.newInstance();
        if (periodInfo != null) {
            travel.setDomesticTravelCost(periodInfo.getDomesticTravelCost()
                    .bigDecimalValue());
            travel.setForeignTravelCost(periodInfo.getForeignTravelCost()
                    .bigDecimalValue());
            travel.setTotalTravelCost(periodInfo.getTotalTravelCost()
                    .bigDecimalValue());
        }
        return travel;
    }

    private Equipment getEquipment(S2SBudgetPeriodDto periodInfo) {
        Equipment equipment = Equipment.Factory.newInstance();
        NarrativeContract extraEquipmentNarr = null;
        if (periodInfo != null && periodInfo.getEquipment() != null
                && periodInfo.getEquipment().size() > 0) {
            // Evaluating Equipments.
            List<EquipmentList> equipmentArrayList = new ArrayList<>();
            ScaleTwoDecimal totalFund = ScaleTwoDecimal.ZERO;
            for (S2SCostDto costInfo : periodInfo.getEquipment().get(0)
                    .getEquipmentList()) {
                EquipmentList equipmentList = EquipmentList.Factory.newInstance();
                equipmentList.setEquipmentItem(costInfo.getDescription());
                if (costInfo.getCost() != null) {
                    equipmentList.setFundsRequested(costInfo.getCost().bigDecimalValue());
                }
                totalFund = totalFund.add(costInfo.getCost());
                equipmentArrayList.add(equipmentList);
            }

            // Evaluating Extra Equipments.
            List<S2SCostDto> extraEquipmentArrayList = new ArrayList<>();
            ScaleTwoDecimal totalExtraEquipFund = ScaleTwoDecimal.ZERO;
            for (S2SCostDto costInfo : periodInfo.getEquipment().get(0).getExtraEquipmentList()) {
                extraEquipmentArrayList.add(costInfo);
                totalExtraEquipFund = totalExtraEquipFund.add(costInfo.getCost());
            }

            EquipmentList[] equipmentArray = equipmentArrayList.toArray(new EquipmentList[0]);
            equipment.setEquipmentListArray(equipmentArray);
            totalFund = totalFund.add(totalExtraEquipFund);
            equipment.setTotalFund(totalFund.bigDecimalValue());
            if (equipmentArray.length > 0) {
                equipment.setTotalFundForAttachedEquipment(totalExtraEquipFund.bigDecimalValue());
            }
            extraEquipmentNarr = saveAdditionalEquipments(periodInfo, extraEquipmentArrayList);
        }
        if (extraEquipmentNarr != null) {
            AttachedFileDataType equipmentAttachment = AttachedFileDataType.Factory.newInstance();
            FileLocation fileLocation = FileLocation.Factory.newInstance();
            equipmentAttachment.setFileLocation(fileLocation);
            String contentId = createContentId(extraEquipmentNarr);
            fileLocation.setHref(contentId);
            equipmentAttachment.setFileLocation(fileLocation);
            equipmentAttachment.setFileName(extraEquipmentNarr.getNarrativeAttachment().getName());
            equipmentAttachment.setMimeType(InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM);
            if (extraEquipmentNarr.getNarrativeAttachment() != null) {
                final HashValueDocument.HashValue hashValue = getHashValue(extraEquipmentNarr.getNarrativeAttachment().getData());
                equipmentAttachment.setHashValue(hashValue);

                addAttachment(new AttachmentData(extraEquipmentNarr.getNarrativeAttachment().getFileDataId(), extraEquipmentNarr.getNarrativeAttachment().getName(), contentId, extraEquipmentNarr
                        .getNarrativeAttachment().getData(), InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM, hashValue.getHashAlgorithm(), hashValue.getStringValue(), extraEquipmentNarr.getNarrativeAttachment().getUploadUser(), extraEquipmentNarr.getNarrativeAttachment().getUploadTimestamp()));
                equipment.setAdditionalEquipmentsAttachment(equipmentAttachment);
            }
        }
        return equipment;
    }

    private OtherPersonnel getOtherPersonnel(S2SBudgetPeriodDto periodInfo) {
        OtherPersonnel otherPersonnel = OtherPersonnel.Factory.newInstance();
        int otherPersonnelCount = 0;
        List<Other> otherPersonnelList = new ArrayList<>();

        if (periodInfo != null) {
            for (S2SOtherPersonnelDto otherPersonnelInfo : periodInfo
                    .getOtherPersonnel()) {

                if (OTHERPERSONNEL_POSTDOC.equals(otherPersonnelInfo
                        .getPersonnelType())) {
                    otherPersonnel
                            .setPostDocAssociates(getPostDocAssociates(otherPersonnelInfo));
                } else if (OTHERPERSONNEL_GRADUATE.equals(otherPersonnelInfo
                        .getPersonnelType())) {
                    otherPersonnel
                            .setGraduateStudents(getGraduateStudents(otherPersonnelInfo));
                } else if (OTHERPERSONNEL_UNDERGRADUATE
                        .equals(otherPersonnelInfo.getPersonnelType())) {
                    otherPersonnel
                            .setUndergraduateStudents(getUndergraduateStudents(otherPersonnelInfo));
                } else if (OTHERPERSONNEL_SECRETARIAL.equals(otherPersonnelInfo
                        .getPersonnelType())) {
                    otherPersonnel.setSecretarialClerical(getSecretarialClerical(otherPersonnelInfo));
                } else if (otherPersonnelCount < OTHERPERSONNEL_MAX_ALLOWED) {// Max
                    // allowed
                    // is 6
                    S2SCompensationDto sectBCompType = otherPersonnelInfo.getCompensation();
                    Other otherPersonnelDataType = otherPersonnel.addNewOther();
                    otherPersonnelDataType.setNumberOfPersonnel(otherPersonnelInfo.getNumberPersonnel());
                    otherPersonnelDataType.setProjectRole(otherPersonnelInfo.getRole());
                    otherPersonnelDataType.setRequestedSalary(sectBCompType.getRequestedSalary().bigDecimalValue());
                    otherPersonnelDataType.setFringeBenefits(sectBCompType.getFringe().bigDecimalValue());
                    otherPersonnelDataType.setAcademicMonths(sectBCompType.getAcademicMonths().bigDecimalValue());
                    otherPersonnelDataType.setCalendarMonths(sectBCompType.getCalendarMonths().bigDecimalValue());
                    otherPersonnelDataType.setFundsRequested(sectBCompType.getFundsRequested().bigDecimalValue());
                    otherPersonnelDataType.setSummerMonths(sectBCompType.getSummerMonths().bigDecimalValue());
                    otherPersonnelList.add(otherPersonnelDataType);
                    otherPersonnelCount++;
                }
            }

            otherPersonnel.setOtherArray(otherPersonnelList.toArray(new Other[0]));

            if (periodInfo.getOtherPersonnelTotalNumber() != null) {
                otherPersonnel.setOtherPersonnelTotalNumber(periodInfo
                        .getOtherPersonnelTotalNumber().intValue());
            }
            if (periodInfo.getTotalOtherPersonnelFunds() != null) {
                otherPersonnel.setTotalOtherPersonnelFund(periodInfo
                        .getTotalOtherPersonnelFunds().bigDecimalValue());
            }
        }
        return otherPersonnel;
    }

    private PostDocAssociates getPostDocAssociates(
            S2SOtherPersonnelDto otherPersonnel) {

        PostDocAssociates postDocAssociates = PostDocAssociates.Factory
                .newInstance();
        if (otherPersonnel != null) {
            postDocAssociates.setNumberOfPersonnel(otherPersonnel
                    .getNumberPersonnel());
            postDocAssociates.setProjectRole(otherPersonnel.getRole());
            S2SCompensationDto sectBCompType = otherPersonnel.getCompensation();

            postDocAssociates.setRequestedSalary(sectBCompType.getRequestedSalary().bigDecimalValue());
            postDocAssociates.setFringeBenefits(sectBCompType.getFringe().bigDecimalValue());
            postDocAssociates.setAcademicMonths(sectBCompType.getAcademicMonths().bigDecimalValue());
            postDocAssociates.setCalendarMonths(sectBCompType.getCalendarMonths().bigDecimalValue());
            postDocAssociates.setFundsRequested(sectBCompType.getFundsRequested().bigDecimalValue());
            postDocAssociates.setSummerMonths(sectBCompType.getSummerMonths().bigDecimalValue());


        }
        return postDocAssociates;
    }

    private GraduateStudents getGraduateStudents(
            S2SOtherPersonnelDto otherPersonnel) {

        GraduateStudents graduate = GraduateStudents.Factory.newInstance();
        if (otherPersonnel != null) {
            graduate.setNumberOfPersonnel(otherPersonnel.getNumberPersonnel());
            graduate.setProjectRole(otherPersonnel.getRole());
            S2SCompensationDto sectBCompType = otherPersonnel.getCompensation();

            graduate.setRequestedSalary(sectBCompType.getRequestedSalary().bigDecimalValue());
            graduate.setFringeBenefits(sectBCompType.getFringe().bigDecimalValue());
            graduate.setAcademicMonths(sectBCompType.getAcademicMonths().bigDecimalValue());
            graduate.setCalendarMonths(sectBCompType.getCalendarMonths().bigDecimalValue());
            graduate.setFundsRequested(sectBCompType.getFundsRequested().bigDecimalValue());
            graduate.setSummerMonths(sectBCompType.getSummerMonths().bigDecimalValue());

        }
        return graduate;
    }

    private UndergraduateStudents getUndergraduateStudents(
            S2SOtherPersonnelDto otherPersonnel) {

        UndergraduateStudents undergraduate = UndergraduateStudents.Factory
                .newInstance();
        if (otherPersonnel != null) {
            undergraduate.setNumberOfPersonnel(otherPersonnel
                    .getNumberPersonnel());
            undergraduate.setProjectRole(otherPersonnel.getRole());
            S2SCompensationDto sectBCompType = otherPersonnel.getCompensation();

            undergraduate.setRequestedSalary(sectBCompType.getRequestedSalary().bigDecimalValue());
            undergraduate.setFringeBenefits(sectBCompType.getFringe().bigDecimalValue());
            undergraduate.setAcademicMonths(sectBCompType.getAcademicMonths().bigDecimalValue());
            undergraduate.setCalendarMonths(sectBCompType.getCalendarMonths().bigDecimalValue());
            undergraduate.setFundsRequested(sectBCompType.getFundsRequested().bigDecimalValue());
            undergraduate.setSummerMonths(sectBCompType.getSummerMonths().bigDecimalValue());


        }
        return undergraduate;
    }

    private SecretarialClerical getSecretarialClerical(
            S2SOtherPersonnelDto otherPersonnel) {

        SecretarialClerical secretarialClerical = SecretarialClerical.Factory
                .newInstance();
        if (otherPersonnel != null) {
            secretarialClerical.setNumberOfPersonnel(otherPersonnel
                    .getNumberPersonnel());
            secretarialClerical.setProjectRole(otherPersonnel.getRole());
            S2SCompensationDto sectBCompType = otherPersonnel.getCompensation();

            secretarialClerical.setRequestedSalary(sectBCompType.getRequestedSalary().bigDecimalValue());
            secretarialClerical.setFringeBenefits(sectBCompType.getFringe().bigDecimalValue());
            secretarialClerical.setAcademicMonths(sectBCompType.getAcademicMonths().bigDecimalValue());
            secretarialClerical.setCalendarMonths(sectBCompType.getCalendarMonths().bigDecimalValue());
            secretarialClerical.setFundsRequested(sectBCompType.getFundsRequested().bigDecimalValue());
            secretarialClerical.setSummerMonths(sectBCompType.getSummerMonths().bigDecimalValue());

        }
        return secretarialClerical;
    }

    private KeyPersons getKeyPersons(S2SBudgetPeriodDto periodInfo) {

        KeyPersons keyPersons = KeyPersons.Factory.newInstance();
        ScaleTwoDecimal extraFunds = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal baseSalaryByPeriod;

        if (periodInfo != null) {
            if (periodInfo.getKeyPersons() != null) {
                List<KeyPerson> keyPersonList = new ArrayList<>();
                int keyPersonCount = 0;
                for (S2SKeyPersonDto keyPerson : periodInfo.getKeyPersons()) {
                    if (keyPerson.getRole().equals(NID_PD_PI) || hasPersonnelBudget(keyPerson, periodInfo.getBudgetPeriod())) {
                        KeyPerson keyPersonDataType = KeyPerson.Factory.newInstance();
                        keyPersonDataType.setName(globLibV20Generator
                                .getHumanNameDataType(keyPerson));
                        if (isSponsorNIH(pdDoc)
                                && KEYPERSON_CO_PD_PI.equals(keyPerson.getRole())) {
                            DevelopmentProposalContract developmentProposal = pdDoc.getDevelopmentProposal();

                            for (ProposalPersonContract proposalPerson : developmentProposal.getInvestigators()) {
                                if (isProposalPersonEqualsKeyPerson(proposalPerson, keyPerson)) {
                                    if (proposalPerson.isMultiplePi())
                                        keyPersonDataType.setProjectRole(NID_PD_PI);
                                    else
                                        keyPersonDataType.setProjectRole(NID_CO_PD_PI);
                                }
                            }
                        } else if (keyPerson.getKeyPersonRole() != null) {
                            keyPersonDataType.setProjectRole(keyPerson.getKeyPersonRole());
                        } else {
                            keyPersonDataType.setProjectRole(keyPerson.getRole());
                        }
                        if (CollectionUtils.isNotEmpty(pdDoc.getDevelopmentProposal().getBudgets())) {
                            baseSalaryByPeriod = keyPerson.getCompensation().getBaseSalary();
                            if (baseSalaryByPeriod != null && baseSalaryByPeriod.isGreaterThan(ScaleTwoDecimal.ZERO)) {
                                keyPersonDataType.setBaseSalary(baseSalaryByPeriod.bigDecimalValue());
                            } else {
                                if (keyPerson.getCompensation().getBaseSalary() != null) {
                                    keyPersonDataType.setBaseSalary(keyPerson.getCompensation().getBaseSalary().bigDecimalValue());
                                }
                            }

                        } else {
                            if (keyPerson.getCompensation().getBaseSalary() != null) {
                                keyPersonDataType.setBaseSalary(keyPerson.getCompensation().getBaseSalary().bigDecimalValue());
                            }
                        }
                        keyPersonDataType.setRequestedSalary(keyPerson.getCompensation().getRequestedSalary().bigDecimalValue());
                        keyPersonDataType.setFringeBenefits(keyPerson.getCompensation().getFringe().bigDecimalValue());
                        keyPersonDataType.setAcademicMonths(keyPerson.getCompensation().getAcademicMonths().bigDecimalValue());
                        keyPersonDataType.setCalendarMonths(keyPerson.getCompensation().getCalendarMonths().bigDecimalValue());
                        keyPersonDataType.setFundsRequested(keyPerson.getCompensation().getFundsRequested().bigDecimalValue());
                        keyPersonDataType.setSummerMonths(keyPerson.getCompensation().getSummerMonths().bigDecimalValue());

                        keyPersonList.add(keyPersonDataType);
                        keyPersonCount++;
                        LOG.info("keyPersonCount:" + keyPersonCount);
                    }
                }
                keyPersons.setKeyPersonArray(keyPersonList.toArray(new KeyPerson[0]));
            }
            if (periodInfo.getTotalFundsKeyPersons() != null) {
                keyPersons.setTotalFundForKeyPersons(periodInfo
                        .getTotalFundsKeyPersons().bigDecimalValue());
            }
            for (S2SKeyPersonDto keyPerson : periodInfo.getExtraKeyPersons()) {
                extraFunds = extraFunds.add(keyPerson.getCompensation().getFundsRequested());
            }
        }
        keyPersons.setTotalFundForAttachedKeyPersons(extraFunds.bigDecimalValue());

        NarrativeContract extraKeyPersonNarr = saveExtraKeyPersons(periodInfo);
        setKeyPersonNarrative(keyPersons, extraKeyPersonNarr);

        return keyPersons;
    }

    private void setKeyPersonNarrative(KeyPersons keyPersons, NarrativeContract extraKeyPersonNarr) {
        if (extraKeyPersonNarr != null) {
            AttachedFileDataType attachedKeyPersons = AttachedFileDataType.Factory.newInstance();
            FileLocation fileLocation = FileLocation.Factory.newInstance();
            attachedKeyPersons.setFileLocation(fileLocation);
            String contentId = createContentId(extraKeyPersonNarr);
            fileLocation.setHref(contentId);
            attachedKeyPersons.setFileLocation(fileLocation);
            attachedKeyPersons.setFileName(extraKeyPersonNarr.getNarrativeAttachment().getName());
            attachedKeyPersons.setMimeType(InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM);
            byte[] narrativeContent = null;
            if (extraKeyPersonNarr.getNarrativeAttachment() != null) {
                narrativeContent = extraKeyPersonNarr.getNarrativeAttachment().getData();
            }
            if (narrativeContent != null && narrativeContent.length > 0) {
                final HashValueDocument.HashValue hashValue = getHashValue(narrativeContent);
                attachedKeyPersons.setHashValue(hashValue);
                addAttachment(new AttachmentData(extraKeyPersonNarr.getNarrativeAttachment().getFileDataId(), extraKeyPersonNarr.getNarrativeAttachment().getName(), contentId, narrativeContent, InfrastructureConstants.CONTENT_TYPE_OCTET_STREAM, hashValue.getHashAlgorithm(), hashValue.getStringValue(), extraKeyPersonNarr.getNarrativeAttachment().getUploadUser(), extraKeyPersonNarr.getNarrativeAttachment().getUploadTimestamp()));
                keyPersons.setAttachedKeyPersons(attachedKeyPersons);
            }
        }
    }

    private AttachedFileDataType getBudgetJustification() {
        AttachedFileDataType attachedFileDataType = AttachedFileDataType.Factory.newInstance();
        for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
            if (narrative.getNarrativeType().getCode() != null
                    && Integer.parseInt(narrative.getNarrativeType().getCode()) == BUDGET_JUSTIFICATION_ATTACHMENT) {
                attachedFileDataType = addAttachedFileType(narrative);
                if (attachedFileDataType != null) {
                    break;
                }
            }
        }
        return attachedFileDataType;
    }

    @Override
    public RRBudget20Document getFormObject(
            ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getRRBudget20();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(RRBudget20Document form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {
            final AttachedFileDataType budgetJustificationAttachment = form.getRRBudget20().getBudgetJustificationAttachment();
            if (budgetJustificationAttachment != null && a.getContentId().equals(budgetJustificationAttachment.getFileLocation().getHref())) {
                return entry("RR_Budget_2_0_P4.optionalFile", a);
            }

            final List<BudgetYearDataType> budgets = form.getRRBudget20().getBudgetYearList();
            if (budgets != null) {;
                for (int i = 0; i < budgets.size(); i++) {
                    final BudgetYearDataType budget = budgets.get(i);
                    if (budget != null) {
                        final BudgetYearDataType.KeyPersons keyPersons = budget.getKeyPersons();
                        if (keyPersons != null && keyPersons.getAttachedKeyPersons() != null && a.getContentId().equals(keyPersons.getAttachedKeyPersons().getFileLocation().getHref())){
                            return entry("RR_Budget_2_0_P1.BudgetYear[" + i + "].P1.optionalFile", a);
                        }

                        final BudgetYearDataType.Equipment equipment = budget.getEquipment();
                        if (equipment != null && equipment.getAdditionalEquipmentsAttachment() != null && a.getContentId().equals(equipment.getAdditionalEquipmentsAttachment().getFileLocation().getHref())){
                            return entry("RR_Budget_2_0_P1.BudgetYear[" + i + "].P2.optionalFile", a);
                        }
                    }
                }
            }

            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<RRBudget20Document> factory() {
        return RRBudget20Document.Factory;
    }
}
