/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import org.apache.xmlbeans.XmlObject;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemContract;
import org.kuali.coeus.common.budget.api.personnel.BudgetPersonnelDetailsContract;
import org.kuali.coeus.propdev.api.budget.ProposalDevelopmentBudgetExtContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.s2sgen.api.budget.S2SBudgetInfoService;
import org.kuali.coeus.s2sgen.api.budget.S2SKeyPersonDto;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.impl.budget.S2SBudgetValidationService;
import org.kuali.coeus.s2sgen.impl.budget.S2SCommonBudgetService;
import org.kuali.coeus.s2sgen.impl.datetime.S2SDateTimeService;
import org.kuali.coeus.s2sgen.impl.generate.S2SBaseFormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.List;

/**
 * This abstract class has methods that are common to all the versions of RRFedNonFedBudget form.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
public abstract class RRFedNonFedBudgetBaseGenerator<T extends XmlObject> extends S2SBaseFormGenerator<T> implements S2SFormGeneratorPdfFillable<T> {

    public static final String OTHERPERSONNEL_POSTDOC = "PostDoc";
    public static final String OTHERPERSONNEL_GRADUATE = "Grad";
    public static final String OTHERPERSONNEL_UNDERGRADUATE = "UnderGrad";
    public static final String OTHERPERSONNEL_SECRETARIAL = "Sec";
    public static final String OTHERCOST_DESCRIPTION = "Other";
    protected static final String NID_PD_PI = "PD/PI";
    public static final int BUDGET_JUSTIFICATION_ATTACHMENT = 131;
    
    protected static final int OTHERPERSONNEL_MAX_ALLOWED = 6;
    protected static final int ARRAY_LIMIT_IN_SCHEMA = 4;

    @Autowired
    @Qualifier("s2SBudgetInfoService")
    protected S2SBudgetInfoService s2sBudgetInfoService;

    @Autowired
    @Qualifier("s2SDateTimeService")
    protected S2SDateTimeService s2SDateTimeService;

    @Autowired
    @Qualifier("s2SCommonBudgetService")
    protected S2SCommonBudgetService s2SCommonBudgetService;

    @Autowired
    @Qualifier("s2SBudgetValidationService")
    protected S2SBudgetValidationService s2SBudgetValidationService;

    /**
     * This method check whether the key person has a personnel budget  
     * 
     * @param keyPerson
     *            (KeyPersonInfo) key person entry.
     * @param period
     *            budget period
     * @return true if key person has personnel budget else false.
     */
    protected Boolean hasPersonnelBudget(S2SKeyPersonDto keyPerson, int period){
        ProposalDevelopmentBudgetExtContract budget = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());
        List<? extends BudgetLineItemContract> budgetLineItemList = budget.getBudgetPeriods().get(period - 1).getBudgetLineItems();
           
        for(BudgetLineItemContract budgetLineItem : budgetLineItemList) {
            for(BudgetPersonnelDetailsContract budgetPersonnelDetails : budgetLineItem.getBudgetPersonnelDetailsList()){
                if( budgetPersonnelDetails.getPersonId().equals(keyPerson.getPersonId())){
                    return true;
                } else if (keyPerson.getRolodexId() != null && budgetPersonnelDetails.getPersonId().equals(keyPerson.getRolodexId().toString())) {
                    return true;
                }
            }
        }
        return false;       
    }
    
    /**
     * Perform manual validations on the budget. Similarly done in RRBudgetBaseGenerator.
     */
    protected boolean validateBudgetForForm(ProposalDevelopmentDocumentContract pdDoc) {
        final List<AuditError> errors = s2SBudgetValidationService.validateBudgetForForm(pdDoc, getFormName());
        getAuditErrors().addAll(errors);
        return errors.isEmpty();
    }

    public S2SBudgetInfoService getS2sBudgetInfoService() {
        return s2sBudgetInfoService;
    }

    public void setS2sBudgetInfoService(S2SBudgetInfoService s2sBudgetInfoService) {
        this.s2sBudgetInfoService = s2sBudgetInfoService;
    }

    public S2SBudgetValidationService getS2SBudgetValidationService() {
        return s2SBudgetValidationService;
    }

    public void setS2SBudgetValidationService(S2SBudgetValidationService s2SBudgetValidationService) {
        this.s2SBudgetValidationService = s2SBudgetValidationService;
    }

    public S2SDateTimeService getS2SDateTimeService() {
        return s2SDateTimeService;
    }

    public void setS2SDateTimeService(S2SDateTimeService s2SDateTimeService) {
        this.s2SDateTimeService = s2SDateTimeService;
    }

    public S2SCommonBudgetService getS2SCommonBudgetService() {
        return s2SCommonBudgetService;
    }

    public void setS2SCommonBudgetService(S2SCommonBudgetService s2SCommonBudgetService) {
        this.s2SCommonBudgetService = s2SCommonBudgetService;
    }
}
