/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.rrPersonalData12V12.*;
import gov.grants.apply.forms.rrPersonalData12V12.RRPersonalData12Document.RRPersonalData12;
import gov.grants.apply.forms.rrPersonalData12V12.impl.CitizenshipTypeImpl;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Class for generating the XML object for grants.gov RRPersonalDataV1.2. Form
 * is generated using XMLBean classes and is based on RRPersonalData schema.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@FormGenerator("RRPersonalDataV1_2Generator")
public class RRPersonalDataV1_2Generator extends RRPersonalDataBaseGenerator<RRPersonalData12Document> {

    @Value("http://apply.grants.gov/forms/RR_PersonalData_1_2-V1.2")
    private String namespace;

    @Value("RR_PersonalData_1_2-V1.2")
    private String formName;

	@FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/RR_PersonalData-V1.2.xsl")
    private List<Resource> stylesheets;

	@Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/RR_PersonalData_1_2-V1.2.pdf")
	private Resource pdfForm;

    @Value("157")
    private int sortIndex;

	/**
	 * 
	 * This method gives the personal information of ProjectDirector and
	 * CoProjectDirector
	 * 
	 * @return rrPersonalDataDocument {@link XmlObject} of type
	 *         RRPersonalDataDocument.
	 */
	private RRPersonalData12Document getRRPersonalData() {
		RRPersonalData12Document rrPersonalDataDocument = RRPersonalData12Document.Factory
				.newInstance();
		RRPersonalData12 rrPersonalData = RRPersonalData12.Factory
				.newInstance();
		rrPersonalData.setFormVersion(FormVersion.v1_2.getVersion());
		rrPersonalData.setProjectDirector(getProjectDirectorType());
		rrPersonalData.setCoProjectDirectorArray(getCoProjectDirectoryType());
		rrPersonalDataDocument.setRRPersonalData12(rrPersonalData);
		return rrPersonalDataDocument;
	}

	/**
	 * 
	 * This method is used to get Personal details of Principal Investigator
	 * 
	 * @return DirectorType principal investigator details.
	 */
	private DirectorType getProjectDirectorType() {
		DirectorType directorType = DirectorType.Factory.newInstance();
		ProposalPersonContract PI = s2SProposalPersonService.getPrincipalInvestigator(pdDoc);
		if (PI != null) {
			directorType.setName(globLibV20Generator.getHumanNameDataType(PI));
		}
		setDirectorOptionals(directorType);
		return directorType;
	}

	private void setDirectorOptionals(DirectorType directorType) {
		directorType.setCitizenship(CitizenshipTypeImpl.DO_NOT_WISH_TO_PROVIDE);
		directorType.setGender(GenderType.DO_NOT_WISH_TO_PROVIDE);
		directorType.setRaceArray(new RaceType.Enum[]{RaceType.DO_NOT_WISH_TO_PROVIDE});
		directorType.setEthnicity(EthnicityType.DO_NOT_WISH_TO_PROVIDE);
		directorType.setDisabilityStatusArray(new DisablilityStatusType.Enum[]{DisablilityStatusType.DO_NOT_WISH_TO_PROVIDE});
	}

	/**
	 * 
	 * This method is used to get List of Personal details of Co-Investigator
	 * 
	 * @return DirectorType[] Array of director types.
	 */
	private DirectorType[] getCoProjectDirectoryType() {

		List<DirectorType> directorTypeList = new ArrayList<>();
		if (pdDoc.getDevelopmentProposal().getProposalPersons() != null) {
			ProposalPersonContract CoPI = null;
			for (ProposalPersonContract proposalPerson : pdDoc.getDevelopmentProposal()
					.getProposalPersons()) {
				DirectorType coDirectorType = DirectorType.Factory
						.newInstance();
				if (proposalPerson.getProposalPersonRoleId() != null) {
					if (KEYPERSON_TYPE_C0_INVESTIGATOR.equals(proposalPerson
							.getProposalPersonRoleId())) {
						CoPI = proposalPerson;
						coDirectorType.setName(globLibV20Generator
								.getHumanNameDataType(CoPI));
						setDirectorOptionals(coDirectorType);
						directorTypeList.add(coDirectorType);
					}
				}
			}
		}
		return directorTypeList.toArray(new DirectorType[0]);
	}

	/**
	 * This method creates {@link XmlObject} of type
	 * {@link RRPersonalData12Document} by populating data from the given
	 * {@link ProposalDevelopmentDocumentContract}
	 * 
	 * @param proposalDevelopmentDocument
	 *            for which the {@link XmlObject} needs to be created
	 * @return {@link XmlObject} which is generated using the given
	 *         {@link ProposalDevelopmentDocumentContract}
	 */
	@Override
	public RRPersonalData12Document getFormObject(
			ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
		this.pdDoc = proposalDevelopmentDocument;
		return getRRPersonalData();
	}

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

	@Override
	public List<Resource> getStylesheets() {
		return stylesheets;
	}

	public void setStylesheets(List<Resource> stylesheets) {
		this.stylesheets = stylesheets;
	}

	@Override
	public Resource getPdfForm() {
		return pdfForm;
	}

	public void setPdfForm(Resource pdfForm) {
		this.pdfForm = pdfForm;
	}

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

	@Override
	public Attachments getMappedAttachments(RRPersonalData12Document form, List<AttachmentData> attachments) {
		return new Attachments(Collections.emptyMap(), attachments);
	}

	@Override
	public DocumentFactory<RRPersonalData12Document> factory() {
		return RRPersonalData12Document.Factory;
	}
}
