/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;


import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlObject;
import org.kuali.coeus.common.budget.api.core.BudgetContract;
import org.kuali.coeus.propdev.api.budget.subaward.BudgetSubAwardAttachmentContract;
import org.kuali.coeus.propdev.api.budget.subaward.BudgetSubAwardsContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.s2sgen.api.core.InfrastructureConstants;
import org.kuali.coeus.s2sgen.api.core.S2SException;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.budget.S2SCommonBudgetService;
import org.kuali.coeus.s2sgen.impl.generate.S2SBaseFormGenerator;
import org.kuali.coeus.s2sgen.impl.util.SafeXmlUtils;
import org.kuali.coeus.s2sgen.impl.validate.S2SErrorHandlerService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * This abstract class has methods that are common to all the versions of RRSubAwardBudget form.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */

public abstract class S2SAdobeFormAttachmentBaseGenerator<T extends XmlObject> extends S2SBaseFormGenerator<T> {

    protected static final String RR_BUDGET_11_NAMESPACE_URI = "http://apply.grants.gov/forms/RR_Budget-V1.1";
    protected static final String LOCAL_NAME = "RR_Budget";
    private static final String SUB_AWARD_BUDGET_NOT_FOUND = "budget.subaward.notfound";

    private static final String REPLACEMENT_CHARACTER = "_";
    //Exclude everything but numbers, alphabets, dots, hyphens and underscores
    private static final String REGEX_TITLE_FILENAME_PATTERN = "([^0-9a-zA-Z\\.\\-_])";

    public final ArrayList <String> attachmentList = new ArrayList<>();
    public final ArrayList <String> budgetIdList = new ArrayList<>();
    public final ArrayList <String> budgetSubawardNumberList = new ArrayList<>();

    @Autowired
    @Qualifier("s2SErrorHandlerService")
    protected S2SErrorHandlerService s2SErrorHandlerService;

    @Autowired
    @Qualifier("s2SCommonBudgetService")
    protected S2SCommonBudgetService s2SCommonBudgetService;

    /**
     * This method convert node of form in to a Document
     * 
     * @param node n {Node} node entry.
     * @return Document containing doc information
     */

    public Document nodeToDom(org.w3c.dom.Node node) throws S2SException {
        try {
            javax.xml.transform.TransformerFactory tf = SafeXmlUtils.safeTransformerFactory();
            javax.xml.transform.Transformer xf = tf.newTransformer();
            javax.xml.transform.dom.DOMResult dr = new javax.xml.transform.dom.DOMResult();
            xf.transform(new javax.xml.transform.dom.DOMSource(node), dr);
            return (Document) dr.getNode();

        }
        catch (javax.xml.transform.TransformerException ex) {
            throw new S2SException(ex.getMessage());
        }
    }


    /**
     * This method convert xml string in to a Document
     * 
     * @param xmlSource {xml String} xml source entry.
     * @return Document containing doc information
     */
    public Document stringToDom(String xmlSource) throws S2SException {
        try {
            DocumentBuilderFactory factory = SafeXmlUtils.safeDocumentBuilderFactory();
            factory.setNamespaceAware(true);
            DocumentBuilder builder = factory.newDocumentBuilder();
            return builder.parse(new InputSource(new StringReader(xmlSource)));
        }
        catch (SAXException | IOException | ParserConfigurationException ex) {
            throw new S2SException(ex.getMessage(), ex);
        }
    }

    /**
     * This method convert Document to a byte Array
     * 
     * @param node {Document} node entry.
     * @return byte Array containing doc information
     */
    public byte[] docToBytes(Document node) throws S2SException {
        try {
            return docToString(node).getBytes(StandardCharsets.UTF_8.name());
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * This method convert Document to a String
     * 
     * @param node {Document} node entry.
     * @return String containing doc information
     */
    public String docToString(Document node) throws S2SException {
        try {
            DOMSource domSource = new DOMSource(node);
            StringWriter writer = new StringWriter();
            StreamResult result = new StreamResult(writer);
            TransformerFactory tf = SafeXmlUtils.safeTransformerFactory();
            Transformer transformer = tf.newTransformer();
            transformer.transform(domSource, result);
            return writer.toString();
        }
        catch (Exception e) {
            throw new S2SException(e.getMessage(),e);
        }
    }

    /**
     * 
     * This method is used to return the attachment name which comes from BudgetSubawards
     * 
     * @param budgetSubAwards(BudgetSubAwards) budget sub award entry.
     * @return String attachment name for the budget sub awards.
     */
    protected String prepareAttName(BudgetSubAwardsContract budgetSubAwards) {
        StringBuilder attachmentName = new StringBuilder();
        boolean hasSameFileName = false;
        boolean isAlreadyprinted = false;
        int attachmentCount=0;
        int suffix=1;
       
        int index =0;
        for(String budgetId : budgetIdList){
            
            if(budgetSubAwards.getBudgetId().toString().equals(budgetId)){
                if(budgetSubawardNumberList.get(index).equals(budgetSubAwards.getSubAwardNumber().toString())){
                    attachmentList.clear();
                    isAlreadyprinted = true;
                    break;
                }
            }
            index++;
        }
        if(isAlreadyprinted){
            budgetIdList.clear();
            budgetSubawardNumberList.clear();
        }
        
        //checking organization name and replacing invalid characters
        // with underscores.
        String cleanSubAwardOrganizationName = checkAndReplaceInvalidCharacters(budgetSubAwards.getOrganizationName());
        attachmentName.append(cleanSubAwardOrganizationName);
        BudgetContract budget = findBudgetFromProposal();
        List<? extends BudgetSubAwardsContract> budgetSubAwardsList = budget.getBudgetSubAwards();
        ArrayList<String> attachments = new ArrayList<>();
        for (BudgetSubAwardsContract budgetSubAward: budgetSubAwardsList) {
            StringBuilder existingAttachmentName = new StringBuilder();
            String subAward_OrganizationName = checkAndReplaceInvalidCharacters(budgetSubAward.getOrganizationName());
            existingAttachmentName.append(subAward_OrganizationName);
            attachments.add(existingAttachmentName.toString());                 
        }
        for (String attachment : attachments) {
            if (attachment.equals(attachmentName.toString())) {
                attachmentCount++;
            }
        }        
        if (attachmentCount>1 && !attachmentList.contains(attachmentName.toString())) {
            attachmentList.add(attachmentName.toString());
            if(attachmentName.length() > 49){
                attachmentName.delete(49, attachmentName.length());               
             }
            attachmentName.append(1);
            hasSameFileName = true;                
        } else {
            for (String attachment:attachmentList) {                  
                if (attachment.equals(attachmentName.toString())) {
                    suffix++;
                }                    
            }
        }            
        if (attachmentList.contains(attachmentName.toString()) && !hasSameFileName) {
            attachmentList.add(attachmentName.toString());
            if(attachmentName.length() > 49){
                attachmentName.delete(49, attachmentName.length());               
             }
            attachmentName.append(suffix);
        } else {
            attachmentList.add(attachmentName.toString());             
        }                           
        budgetIdList.add(budgetSubAwards.getBudgetId().toString());
        budgetSubawardNumberList.add(budgetSubAwards.getSubAwardNumber().toString());           
        return attachmentName.toString();
    }

    public String checkAndReplaceInvalidCharacters(String text) {
        String cleanText = text;
        if (text != null) {
            Pattern pattern = Pattern.compile(REGEX_TITLE_FILENAME_PATTERN);
            Matcher matcher = pattern.matcher(text);
            cleanText = matcher.replaceAll(REPLACEMENT_CHARACTER);
            if(cleanText.length() > 50){
                cleanText = cleanText.substring(0, 50);
            }
        }
        return cleanText;
    }

    /**
     * Adding attachments to subaward
     */
    protected void addSubAwdAttachments(BudgetSubAwardsContract budgetSubAwards) {
        List<? extends BudgetSubAwardAttachmentContract> subAwardAttachments = budgetSubAwards.getBudgetSubAwardAttachments();
        for (BudgetSubAwardAttachmentContract budgetSubAwardAttachment : subAwardAttachments) {
            final String hash = getGrantApplicationHashService().computeAttachmentHash(budgetSubAwardAttachment.getData());
            addAttachment(new AttachmentData(budgetSubAwardAttachment.getFileDataId(), budgetSubAwardAttachment.getName(), budgetSubAwardAttachment.getName(), budgetSubAwardAttachment.getData(), budgetSubAwardAttachment.getType(), InfrastructureConstants.HASH_ALGORITHM, hash, budgetSubAwardAttachment.getUploadUser(), budgetSubAwardAttachment.getUploadTimestamp()));
        }
    }
    /**
     * 
     * This method is used to get BudgetSubAwrads from ProposalDevelopmentDocumentContract
     * 
     * @param proposalDevelopmentDocument (ProposalDevelopmentDocumentContract)
     * @return List&lt;BudgetSubAwards&gt; list of budget sub awards.
     */
    protected List<BudgetSubAwardsContract> getBudgetSubAwards(ProposalDevelopmentDocumentContract proposalDevelopmentDocument,
            String namespace,boolean checkNull) {
        List<BudgetSubAwardsContract> budgetSubAwardsList = new ArrayList<>();
        BudgetContract budget = findBudgetFromProposal();
        if(budget==null){
            getAuditErrors().add(s2SErrorHandlerService.getError(SUB_AWARD_BUDGET_NOT_FOUND, getFormName()));
        }else{
            budgetSubAwardsList = findBudgetSubawards(namespace, budget,checkNull);
            if(budgetSubAwardsList.isEmpty()){
                getAuditErrors().add(s2SErrorHandlerService.getError(SUB_AWARD_BUDGET_NOT_FOUND, getFormName()));
            }
        }
        return budgetSubAwardsList;
    }


    /**
     * This method is to find the subaward budget BOs for the given namespace.
     */
    @SuppressWarnings("unchecked")
    private List<BudgetSubAwardsContract> findBudgetSubawards(String namespace, BudgetContract budget,boolean checkNull) {
        List<BudgetSubAwardsContract> budgetSubAwardsList = new ArrayList<>();
        for (BudgetSubAwardsContract subAwards : budget.getBudgetSubAwards()) {
        	if (StringUtils.equals(namespace, subAwards.getNamespace())
        			|| (checkNull && StringUtils.isBlank(subAwards.getNamespace()))) {
        		budgetSubAwardsList.add(subAwards);
        	}
        }
        return budgetSubAwardsList;
    }

    private BudgetContract findBudgetFromProposal() {
        return s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());
    }

    public S2SErrorHandlerService getS2SErrorHandlerService() {
        return s2SErrorHandlerService;
    }

    public void setS2SErrorHandlerService(S2SErrorHandlerService s2SErrorHandlerService) {
        this.s2SErrorHandlerService = s2SErrorHandlerService;
    }

    public S2SCommonBudgetService getS2SCommonBudgetService() {
        return s2SCommonBudgetService;
    }

    public void setS2SCommonBudgetService(S2SCommonBudgetService s2SCommonBudgetService) {
        this.s2SCommonBudgetService = s2SCommonBudgetService;
    }
}
