/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.person;


import org.kuali.coeus.common.api.person.attr.CitizenshipType;
import org.kuali.coeus.common.api.person.attr.CitizenshipTypeService;
import org.kuali.coeus.common.api.rolodex.RolodexService;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

@Component("s2SProposalPersonService")
public class S2SProposalPersonServiceImpl implements S2SProposalPersonService {

    @Autowired
    @Qualifier("citizenshipTypeService")
    private CitizenshipTypeService citizenshipTypeService;

    @Autowired
    @Qualifier("rolodexService")
    private RolodexService rolodexService;

    /**
     * This method limits the number of key persons to n, returns list of key persons.
     *
     * @param proposalPersons list of {@link org.kuali.coeus.propdev.api.person.ProposalPersonContract}
     * @param n number of key persons that are considered as not extra persons
     * @return list of {@link org.kuali.coeus.propdev.api.person.ProposalPersonContract}
     */
    @Override
    public List<ProposalPersonContract> getNKeyPersons(List<? extends ProposalPersonContract> proposalPersons, int n) {
        ProposalPersonContract proposalPerson, previousProposalPerson;
        int size = proposalPersons.size();

        for (int i = size - 1; i > 0; i--) {
            proposalPerson = proposalPersons.get(i);
            previousProposalPerson = proposalPersons.get(i - 1);
            if (proposalPerson.getPersonId() != null && previousProposalPerson.getPersonId() != null
                    && proposalPerson.getPersonId().equals(previousProposalPerson.getPersonId())) {
                proposalPersons.remove(i);
            }
            else if (proposalPerson.getRolodexId() != null && previousProposalPerson.getRolodexId() != null
                    && proposalPerson.getRolodexId().equals(previousProposalPerson.getRolodexId())) {
                proposalPersons.remove(i);
            }
        }

        size = proposalPersons.size();
        List<ProposalPersonContract> firstNPersons = new ArrayList<>();

        // Make sure we don't exceed the size of the list.
        if (size > n) {
            size = n;
        }
        // remove extras
        for (int i = 0; i < size; i++) {
            firstNPersons.add(proposalPersons.get(i));
        }
        return firstNPersons;
    }

    /**
     * This method is to get PrincipalInvestigator from person list
     *
     * @param pdDoc Proposal development document.
     * @return ProposalPerson PrincipalInvestigator for the proposal.
     */
    @Override
    public ProposalPersonContract getPrincipalInvestigator(ProposalDevelopmentDocumentContract pdDoc) {
        ProposalPersonContract proposalPerson = null;
        if (pdDoc != null) {
            for (ProposalPersonContract person : pdDoc.getDevelopmentProposal().getProposalPersons()) {
                if (person.isPrincipalInvestigator()) {
                    proposalPerson = person;
                }
            }
        }
        return proposalPerson;
    }

    /**
     * Finds all the Investigators associated with the provided pdDoc.
     */
    @Override
    public List<ProposalPersonContract> getCoInvestigators(ProposalDevelopmentDocumentContract pdDoc) {
        List<ProposalPersonContract> investigators = new ArrayList<>();
        if (pdDoc != null) {
            for (ProposalPersonContract person : pdDoc.getDevelopmentProposal().getProposalPersons()) {
                //multi-pis are still considered co-i within S2S.
                if (person.isCoInvestigator() || person.isMultiplePi()) {
                    investigators.add(person);
                }
            }
        }
        return investigators;
    }

    /**
     * Finds all the key Person associated with the provided pdDoc.
     */
    @Override
    public List<ProposalPersonContract> getKeyPersons(ProposalDevelopmentDocumentContract pdDoc) {
        List<ProposalPersonContract> keyPersons = new ArrayList<>();
        if (pdDoc != null) {
            for (ProposalPersonContract person : pdDoc.getDevelopmentProposal().getProposalPersons()) {
                if (person.isKeyPerson()) {
                    keyPersons.add(person);
                }
            }
        }
        return keyPersons;
    }

    /**
     * Implementation should return one of the enums defined in PHS398CareerDevelopmentAwardSup11V11 form schema. For now, it
     * returns US RESIDENT as default
     */
    @Override
    public CitizenshipType getCitizenship(ProposalPersonContract proposalPerson) {
    	return citizenshipTypeService.getPersonCitizenshipType(proposalPerson);
    }

    @Override
    public String getPersonUei(ProposalPersonContract proposalPerson) {
        final var rolodexId = proposalPerson.getRolodexId();
        if (rolodexId == null) return null;
        final var rolodex = rolodexService.getRolodex(rolodexId);
        if (rolodex == null || rolodex.getOrganizationUei() == null || rolodex.getOrganizationUei().isEmpty()) return null;
        return rolodex.getOrganizationUei();
    }
}
