/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.coeus.s2sgen.impl.budget;

import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemContract;
import org.kuali.coeus.common.budget.api.period.BudgetPeriodContract;
import org.kuali.coeus.propdev.api.budget.ProposalDevelopmentBudgetExtContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.s2s.S2SConfigurationService;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.api.core.ConfigurationConstants;
import org.kuali.coeus.s2sgen.impl.validate.S2SErrorHandlerService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

@Component("s2SBudgetValidationService")
public class S2SBudgetValidationServiceImpl implements S2SBudgetValidationService {

    private static final String PARTICIPANT_COUNT_REQUIRED = "s2s.budget.participantcount.required";
    private static final String PARTICIPANT_COSTS_REQUIRED = "s2s.budget.participantcost.required";

    @Autowired
    @Qualifier("s2SCommonBudgetService")
    private S2SCommonBudgetService s2SCommonBudgetService;

    @Autowired
    @Qualifier("s2SConfigurationService")
    private S2SConfigurationService s2SConfigurationService;

    @Autowired
    @Qualifier("s2SErrorHandlerService")
    protected S2SErrorHandlerService s2SErrorHandlerService;

    private String getParticipantSupportCategoryCode() {
        return s2SConfigurationService.getValueAsString(ConfigurationConstants.BUDGET_CATEGORY_TYPE_PARTICIPANT_SUPPORT);
    }

    private List<? extends BudgetLineItemContract> getMatchingLineItems(List<? extends BudgetLineItemContract> lineItems, List<String> budgetCategoryType) {
        List<BudgetLineItemContract> result = new ArrayList<>();
        for (BudgetLineItemContract lineItem : lineItems) {
            if (budgetCategoryType.contains(lineItem.getBudgetCategory().getBudgetCategoryType().getCode())) {
                result.add(lineItem);
            }
        }
        return result;
    }

    @Override
    public List<AuditError> validateBudgetForForm(ProposalDevelopmentDocumentContract pdDoc, String formName) {
        final List<AuditError> auditErrors = new ArrayList<>();

        ProposalDevelopmentBudgetExtContract budget = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());
        if(budget != null){
            for (BudgetPeriodContract period : budget.getBudgetPeriods()) {
                List<String> participantSupportCode = new ArrayList<>();
                participantSupportCode.add(getParticipantSupportCategoryCode());
                List<? extends BudgetLineItemContract> participantSupportLineItems =
                        getMatchingLineItems(period.getBudgetLineItems(), participantSupportCode);
                int numberOfParticipants = period.getNumberOfParticipants() == null ? 0 : period.getNumberOfParticipants();
                if (!participantSupportLineItems.isEmpty() && numberOfParticipants == 0) {
                    AuditError auditError= s2SErrorHandlerService.getError(PARTICIPANT_COUNT_REQUIRED, formName);
                    AuditError error= new AuditError(auditError.getErrorKey(),
                            auditError.getMessageKey()+period.getBudgetPeriod(),auditError.getLink());
                    auditErrors.add(error);
                } else if (numberOfParticipants > 0 && participantSupportLineItems.isEmpty()) {
                    auditErrors.add(s2SErrorHandlerService.getError(PARTICIPANT_COSTS_REQUIRED, formName));
                }
            }
        }
        return auditErrors;
    }
}
