/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate;

import gov.grants.apply.system.header20V20.Header20Document.Header20;
import gov.grants.apply.system.metaGrantApplication.GrantApplicationDocument;
import gov.grants.apply.system.metaGrantApplication.GrantApplicationDocument.GrantApplication.Forms;
import gov.grants.apply.system.metaMultiGrantApplication.ApplicationDocument;
import gov.grants.apply.system.metaMultiGrantApplication.ApplicationDocument.Application;
import gov.grants.apply.system.metaMultiGrantApplication.ApplicationDocument.Application.ApplicationHeader;
import gov.grants.apply.system.metaMultiGrantApplication.ApplicationDocument.Application.ApplicationPackage;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.kuali.coeus.propdev.api.attachment.NarrativeService;
import org.kuali.coeus.propdev.api.core.DevelopmentProposalContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.s2s.S2sOppFormsContract;
import org.kuali.coeus.propdev.api.s2s.S2sUserAttachedFormFileContract;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.api.core.InfrastructureConstants;
import org.kuali.coeus.s2sgen.api.core.S2SException;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.api.generate.FormGenerationResult;
import org.kuali.coeus.s2sgen.api.generate.FormGeneratorService;
import org.kuali.coeus.s2sgen.api.generate.MultiProjectFormGenerationResult;
import org.kuali.coeus.s2sgen.impl.datetime.S2SDateTimeService;
import org.kuali.coeus.s2sgen.impl.util.XmlBeansUtils;
import org.kuali.coeus.s2sgen.impl.validate.S2SValidatorService;
import org.kuali.coeus.s2sgen.impl.validate.ValidationResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.io.ResourceLoader;
import org.springframework.stereotype.Component;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.xml.namespace.QName;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.kuali.coeus.s2sgen.impl.util.MultiProjectConstants.*;

@Component("formGeneratorService")
public class FormGeneratorServiceImpl implements FormGeneratorService {

    private static final String IN_SUBMISSION_OVERRIDE = " in Submission Override.";
    @Autowired
    @Qualifier("s2SFormGeneratorRetrievalService")
	private S2SFormGeneratorRetrievalService s2SFormGeneratorRetrievalService;

    @Autowired
    @Qualifier("s2SValidatorService")
    private S2SValidatorService s2SValidatorService;

    @Autowired
    @Qualifier("narrativeService")
    private NarrativeService narrativeService;

    @Autowired
    @Qualifier("s2SDateTimeService")
    private S2SDateTimeService s2SDateTimeService;

    @Autowired
    @Qualifier("formApplicationService")
    private FormApplicationService formApplicationService;

    @Autowired
    private ResourceLoader resourceLoader;

    @Override
    public FormGenerationResult generateAndValidateForms(ProposalDevelopmentDocumentContract pdDocContract) throws S2SException {

        if (pdDocContract == null) {
            throw new IllegalArgumentException("pdDoc is null");
        }

        return isOverrideAvailable(pdDocContract.getDevelopmentProposal()) ? createOverrideResult(pdDocContract) : createDynamicResult(pdDocContract);
    }

    private boolean isOverrideAvailable(DevelopmentProposalContract proposal) {
        return proposal.getS2sOverride() != null && proposal.getS2sOverride().isActive() && proposal.getS2sOverride().getApplicationOverride() != null && proposal.getS2sOverride().getApplicationOverride().getApplication() != null;
    }

    private FormGenerationResult createOverrideResult(ProposalDevelopmentDocumentContract pdDocContract) throws S2SException {
        final String applicationXml = pdDocContract.getDevelopmentProposal().getS2sOverride().getApplicationOverride().getApplication();

        final List<AuditError> errors = new ArrayList<>();
        boolean valid = true;
        try {
            final GrantApplicationDocument app = GrantApplicationDocument.Factory.parse(applicationXml);
            final GrantApplicationDocument.GrantApplication.Forms forms = app.getGrantApplication().getForms();
            final NodeList formNodes = forms.newDomNode().getChildNodes();
            for (int i = 0; i < formNodes.getLength(); i++) {
                final Node formNode = formNodes.item(i).getFirstChild();
                final String namespace = formNode.getNamespaceURI();
                if (StringUtils.isNotBlank(namespace)) {
                    final S2SFormGenerator<?> s2SFormGenerator = s2SFormGeneratorRetrievalService.getS2SGenerator(namespace, pdDocContract.getDevelopmentProposal().getProposalNumber());
                    //if the form isn't available in the formMapping service, then just use the node name.
                    final String formName = s2SFormGenerator != null ? s2SFormGenerator.getFormName() : formNode.getLocalName();
                    final ValidationResult validationResult = s2SValidatorService.validateForm(XmlObject.Factory.parse(formNode), formName);
                    valid &= validationResult.isValid();
                    errors.addAll(validationResult.getErrors());
                }
            }

            final FormGenerationResult result;
            if (valid) {
                final ValidationResult validationResult = s2SValidatorService.validateApplication(applicationXml, resourceLoader.getResource(pdDocContract.getDevelopmentProposal().getS2sOpportunity().getSchemaUrl()));
                valid = validationResult.isValid();
                errors.addAll(validationResult.getErrors());

                errors.forEach(error -> error.setMessageKey(modifyMessageForOverride(error.getMessageKey())));
                if (valid) {
                    result = FormGenerationResult.valid(errors, applicationXml, pdDocContract.getDevelopmentProposal().getS2sOverride().getApplicationOverride().getAttachments()
                            .stream()
                            .map(attachment -> new AttachmentData(attachment.getFileDataId(), attachment.getName(), attachment.getContentId(), attachment.getData(), attachment.getType(), InfrastructureConstants.HASH_ALGORITHM, attachment.getSha1Hash(), attachment.getUploadUser(), attachment.getUploadTimestamp()))
                            .collect(Collectors.toList()));
                } else {
                    result = FormGenerationResult.invalid(errors);
                }
            } else {
                result = FormGenerationResult.invalid(errors);
            }

            return result;
        } catch (XmlException e) {
            throw new S2SException(e);
        }
    }

    private String modifyMessageForOverride(String msg) {
        return msg.endsWith(".") ? msg.substring(0,  msg.length() - 1) + IN_SUBMISSION_OVERRIDE : msg + IN_SUBMISSION_OVERRIDE;
    }

    private FormGenerationResult createDynamicResult(ProposalDevelopmentDocumentContract pdDocContract) throws S2SException {
        final GrantApplicationDocument.GrantApplication.Forms forms = GrantApplicationDocument.GrantApplication.Forms.Factory.newInstance();
        final List<AttachmentData> attList = new ArrayList<>();
        boolean validationSucceeded = true;
        final List<AuditError> auditErrors = new ArrayList<>();

        final DevelopmentProposalContract developmentProposal = pdDocContract.getDevelopmentProposal();
        getNarrativeService().deleteSystemGeneratedNarratives(developmentProposal.getNarratives());
        for (S2sOppFormsContract opportunityForm : developmentProposal.getS2sOppForms()) {
            if (opportunityForm.getInclude()) {
                final S2SFormGenerator<?> s2SFormGenerator = s2SFormGeneratorRetrievalService.getS2SGenerator(opportunityForm.getOppNameSpace(), pdDocContract.getDevelopmentProposal().getProposalNumber());
                if (s2SFormGenerator != null) {
                    try {
                        final XmlObject formObject = s2SFormGenerator.getFormObject(pdDocContract);
                        auditErrors.addAll(s2SFormGenerator.getAuditErrors());
                        final ValidationResult validationResult = s2SValidatorService.validateForm(formObject, s2SFormGenerator.getFormName());
                        auditErrors.addAll(validationResult.getErrors());
                        if (validationResult.isValid()) {
                            setFormObject(forms, formObject);
                        } else {
                            validationSucceeded = false;
                        }
                        attList.addAll(s2SFormGenerator.getAttachments());
                    } catch (RuntimeException ex) {
                        throw new S2SException("Could not generate form for " + opportunityForm.getFormName(), ex);
                    }
                }
            }
        }

        final FormGenerationResult result;
        if (validationSucceeded) {
            String applicationXml = getFormApplicationService().getGrantApplicationDocument(pdDocContract, forms);

            final ValidationResult validationResult = s2SValidatorService.validateApplication(applicationXml, resourceLoader.getResource(developmentProposal.getS2sOpportunity().getSchemaUrl()));
            validationSucceeded = validationResult.isValid();
            auditErrors.addAll(validationResult.getErrors());

            if (validationSucceeded) {
                result = FormGenerationResult.valid(auditErrors, applicationXml, attList);
            } else {
                result = FormGenerationResult.invalid(auditErrors);
            }
        } else {
            result = FormGenerationResult.invalid(auditErrors);
        }

        return result;
    }

    private String getXmlFromDocument(XmlObject grantApplicationDocument) {
        final String applicationXmlText = grantApplicationDocument.xmlText(s2SFormGeneratorRetrievalService.getXmlOptionsPrefixes());
        return s2SDateTimeService.removeTimezoneFactor(applicationXmlText);
    }

    /**
	 *
	 * This method is to set the formObject to MetaGrants Forms Object. The
	 * xmlbeans Schema compiled with xsd:any does not provide a direct method to
	 * add individual forms to the Forms object. In this method, an XML Cursor
	 * is created from the existing Forms object and use the moveXml method to
	 * add the form object to the Forms object.
	 *
	 * @param forms
	 *            Forms object to which the grants.gov form is added.
	 * @param formObject
	 *            xml object representing the grants.gov form.
	 */
    private void setFormObject(Forms forms, XmlObject formObject) {
		// Create a cursor from the grants.gov form
		XmlCursor formCursor = formObject.newCursor();
		formCursor.toStartDoc();
		formCursor.toNextToken();

		// Create a cursor from the Forms object
		XmlCursor metaGrantCursor = forms.newCursor();
		metaGrantCursor.toNextToken();

		// Add the form to the Forms object.
		formCursor.moveXml(metaGrantCursor);
	}

    /**
     * Generate and validate forms and application xml for a complex multi-project opportunity.  Parent and children
     * are required.
     */
    @Override
    public MultiProjectFormGenerationResult generateAndValidateMPForms(ProposalDevelopmentDocumentContract parent,
                                                           List<ProposalDevelopmentDocumentContract> children) throws S2SException {
        if (parent == null || CollectionUtils.isEmpty(children)) {
            throw new IllegalArgumentException("parent and at least one child is required for multi-project submissions");
        }

        return createMPDynamicResult(parent, children);
    }

    private MultiProjectFormGenerationResult createMPDynamicResult(ProposalDevelopmentDocumentContract parent,
                                                       List<ProposalDevelopmentDocumentContract> children) throws S2SException {

        List<ComplexFormGeneratedResult> formResults = Stream.concat(children.stream(), Stream.of(parent))
                .map(this::generateAndValidateMPForms)
                .collect(Collectors.toList());

        boolean validationSucceeded = formResults.stream().allMatch(result -> result.isValid);
        final Map<String, List<AuditError>> auditErrors = formResults.stream()
                .collect(Collectors.toMap(
                        r -> r.getDevelopmentDocumentContract().getDevelopmentProposal().getProposalNumber(),
                        ComplexFormGeneratedResult::getAuditErrors,
                        (e1, e2) -> Stream.concat(e1.stream(), e2.stream()).collect(Collectors.toList())
                ));
        final Map<String, List<AttachmentData>> attachments = formResults.stream()
                .collect(Collectors.toMap(
                        r -> r.getDevelopmentDocumentContract().getDevelopmentProposal().getProposalNumber(),
                        ComplexFormGeneratedResult::getAttachments,
                        (e1, e2) -> Stream.concat(e1.stream(), e2.stream()).collect(Collectors.toList())
                ));

        final MultiProjectFormGenerationResult result;
        if (validationSucceeded) {
            String applicationXml = getComplexGrantApplicationDocument(formResults);

            final DevelopmentProposalContract parentProposal = parent.getDevelopmentProposal();
            final ValidationResult validationResult = s2SValidatorService.validateApplication(applicationXml, resourceLoader.getResource(parentProposal.getS2sOpportunity().getSchemaUrl()));
            validationSucceeded = validationResult.isValid();

            if (!validationResult.isValid()) {
                if (auditErrors.containsKey(parentProposal.getProposalNumber())) {
                    auditErrors.get(parentProposal.getProposalNumber()).addAll(validationResult.getErrors());
                } else {
                    auditErrors.put(parentProposal.getProposalNumber(), validationResult.getErrors());
                }
            }

            if (validationSucceeded) {
                result = MultiProjectFormGenerationResult.valid(auditErrors, applicationXml, attachments);
            } else {
                result = MultiProjectFormGenerationResult.invalid(auditErrors);
            }
        } else {
            result = MultiProjectFormGenerationResult.invalid(auditErrors);
        }

        return result;
    }

    private ComplexFormGeneratedResult generateAndValidateMPForms(ProposalDevelopmentDocumentContract proposalContract) throws S2SException{
        ComplexFormGeneratedResult complexFormGeneratedResult = new ComplexFormGeneratedResult(proposalContract);

        final DevelopmentProposalContract parentProposal = proposalContract.getDevelopmentProposal();
        getNarrativeService().deleteSystemGeneratedNarratives(parentProposal.getNarratives());
        for (S2sOppFormsContract opportunityForm : parentProposal.getS2sOppForms()) {
            if (opportunityForm.getInclude()) {
                final S2SFormGenerator<?> s2sFormGenerator = s2SFormGeneratorRetrievalService.getS2SGenerator(opportunityForm.getOppNameSpace(), parentProposal.getProposalNumber());
                if (s2sFormGenerator != null) {
                    try {
                        final XmlObject formObject = s2sFormGenerator.getFormObject(proposalContract);
                        complexFormGeneratedResult.addAuditErrors(s2sFormGenerator.getAuditErrors());
                        final ValidationResult validationResult = s2SValidatorService.validateForm(formObject, s2sFormGenerator.getFormName());
                        complexFormGeneratedResult.addAuditErrors(validationResult.getErrors());
                        if (validationResult.isValid()) {
                            complexFormGeneratedResult.addForm(formObject);
                        } else {
                            complexFormGeneratedResult.setValid(false);
                        }
                        complexFormGeneratedResult.addAttachments(s2sFormGenerator.getAttachments());
                    } catch (RuntimeException ex) {
                        throw new S2SException("Could not generate form for " + opportunityForm.getFormName(), ex);
                    }
                }
            }
        }

        return complexFormGeneratedResult;
    }

    private String getComplexGrantApplicationDocument(final List<ComplexFormGeneratedResult> formResults) throws S2SException {
        final ApplicationDocument applicationDocument = ApplicationDocument.Factory.newInstance();
        final Application application = applicationDocument.addNewApplication();
        final ApplicationHeader applicationHeader = application.addNewApplicationHeader();
        final ApplicationPackage applicationPackage = application.addNewApplicationPackage();

        ComplexFormGeneratedResult overallFormResults = formResults.stream()
                .filter(r -> OVERALL_COMPONENT_TYPE.equalsIgnoreCase(r.getComponentType()))
                .findFirst()
                .orElseThrow(() -> new S2SException("Missing overall component"));
        final ProposalDevelopmentDocumentContract overallPdDoc = overallFormResults.getDevelopmentDocumentContract();

        applicationHeader.setSchemaVersion(FormVersion.v1_0.getVersion());
        setHeader2(applicationHeader, overallPdDoc);

        createOverallApplication(applicationPackage, overallFormResults);

        formResults.stream()
                .filter(r -> !OVERALL_COMPONENT_TYPE.equalsIgnoreCase(r.getComponentType()))
                .collect(Collectors.groupingBy(ComplexFormGeneratedResult::getComponentType))
                .entrySet()
                .stream()
                .sorted((e1, e2) -> compareRefs(e1.getKey(), e2.getKey(), overallPdDoc))
                .forEach(entry -> createSubApplicationGroup(applicationPackage, entry.getKey(), entry.getValue()));

        final XmlCursor cursor = applicationDocument.newCursor();

        cursor.toStartDoc();
        if (cursor.toFirstChild()) {
            final String defaultNameSpace = cursor.getName().getNamespaceURI();
            final String schemaLocation = defaultNameSpace + " " + overallPdDoc.getDevelopmentProposal().getS2sOpportunity().getSchemaUrl();
            cursor.setAttributeText(new QName("http://www.w3.org/2001/XMLSchema-instance", "schemaLocation"), schemaLocation);
        }
        cursor.dispose();

        fixNamespaces(applicationDocument);

        return getXmlFromDocument(applicationDocument);
    }

    /**
     * Grants.gov can't seem to parse proper xml... instead they require the component namespaces to appear in the root node
     * This method will add those declarations properly based on what was already generated.  Note that nih validates just
     * fine with these namespaces declared on the nodes they apply to.
     */
    private void fixNamespaces(ApplicationDocument applicationDocument) {
        Set<QName> componentNamespaces = new HashSet<>();
        XmlCursor cursor = applicationDocument.newCursor();
        cursor.toFirstChild(); //Application
        cursor.toFirstChild(); //ApplicationHeader
        cursor.toNextSibling(); //ApplicationPackage
        cursor.toFirstChild(); //OverallApplication
        do {
            componentNamespaces.add(cursor.getName());
        } while (cursor.toNextSibling());

        cursor.toStartDoc();
        cursor.toFirstChild();
        cursor.toNextToken(); //insert will insert before the cursor, not at the cursor :(
        componentNamespaces.forEach(ns -> cursor.insertNamespace(ns.getPrefix(), ns.getNamespaceURI()));
        cursor.dispose();
    }

    //We need to ensure that components are built in the order they are specified in the opportunity xsd
    //Elements look like this: <xsd:element ref="Admin-Core:SubApplicationGroup" minOccurs="0" maxOccurs="1"/>
    private int compareRefs(String component1, String component2, ProposalDevelopmentDocumentContract proposalDocument) {
        String opportunityXsd = proposalDocument.getDevelopmentProposal().getS2sOpportunity().getOpportunity();
        String subApplicationGroup1 = component1 + ":" + SUB_APPLICATION_GROUP;
        String subApplicationGroup2 = component2 + ":" + SUB_APPLICATION_GROUP;
        return Integer.valueOf(opportunityXsd.indexOf(subApplicationGroup1)).compareTo(Integer.valueOf(opportunityXsd.indexOf(subApplicationGroup2)));
    }

    /*
    The provided xsd for a MetaMultiGrantApplication provides the wrong namespaces.  Due to that, we can't just use the generated beans.
    Here we are manually creating the proper elements in the /Overall namespace
     */
    private void createOverallApplication(final ApplicationPackage applicationPackage, final ComplexFormGeneratedResult overallFormResults) {
        final XmlCursor packageCursor = applicationPackage.newCursor();
        packageCursor.toNextToken(); //take us to the inside of the ApplicationPackage element
        packageCursor.push();
        packageCursor.beginElement(XmlBeansUtils.createMPElement(OVERALL_COMPONENT_TYPE, OVERALL_APPLICATION));
        packageCursor.push();
        packageCursor.beginElement(XmlBeansUtils.createMPElement(OVERALL_COMPONENT_TYPE, OVERALL_APPLICATION_HEADER));
        packageCursor.beginElement(XmlBeansUtils.createMPElement(OVERALL_COMPONENT_TYPE, OVERALL_APPLICATION_ID));
        packageCursor.insertChars(OVERALL_COMPONENT_TYPE);
        packageCursor.pop();
        packageCursor.toNextSibling(); //now we're right after the OverallApplication element
        createGrantApplication(packageCursor, OVERALL_COMPONENT_TYPE, overallFormResults);
        packageCursor.dispose();
    }

    private void setHeader2(final ApplicationHeader header, final ProposalDevelopmentDocumentContract pdDoc) {
        final Header20 header20 = header.addNewHeader20();
        header20.setApplicationFilingName(pdDoc.getDevelopmentProposal().getTitle());
        header20.setPackageID(pdDoc.getDevelopmentProposal().getS2sOpportunity().getPackageId());
        header20.setSchemaVersion(FormVersion.v2_0.getVersion());
    }

    private void createSubApplicationGroup(final XmlObject applicationPackage, final String componentType,
                                           final List<ComplexFormGeneratedResult> groupFormResults) {
        final XmlCursor cursor = applicationPackage.newCursor();
        cursor.toEndToken();
        cursor.beginElement(XmlBeansUtils.createMPElement(componentType, SUB_APPLICATION_GROUP));
        cursor.push();
        cursor.beginElement(XmlBeansUtils.createMPElement(componentType, SUB_APPLICATION_GROUP_HEADER));
        cursor.beginElement(XmlBeansUtils.createMPElement(componentType, SUB_APPLICATION_GROUP_ID));
        cursor.insertChars(componentType);
        cursor.pop();
        cursor.toNextSibling();

        int i = 1;
        for (ComplexFormGeneratedResult groupFormResult : groupFormResults) {
            createSubApplication(cursor, groupFormResult, i, componentType);
            i++;
        }

        cursor.dispose();
    }

    private void createSubApplication(final XmlCursor cursor, final ComplexFormGeneratedResult formResults,
                                      int applicationIndex, String componentType) {
        cursor.push();
        cursor.beginElement(XmlBeansUtils.createMPElement(componentType, SUB_APPLICATION));
        cursor.push();
        cursor.beginElement(XmlBeansUtils.createMPElement(componentType, SUB_APPLICATION_HEADER));
        cursor.beginElement(XmlBeansUtils.createMPElement(componentType, SUB_APPLICATION_ID));
        cursor.insertChars(String.format("%03d-%s", applicationIndex, componentType));
        cursor.pop();
        cursor.toNextSibling(); //now we're right after the SubApplicationHeader element
        createGrantApplication(cursor, componentType, formResults);

        //set the cursor after our new SubApplication element for the next guy
        cursor.pop();
        cursor.toNextSibling();
    }

    private void createGrantApplication(final XmlCursor cursor, final String componentType, final ComplexFormGeneratedResult formResults) {
        cursor.beginElement(XmlBeansUtils.createMPElement(componentType, GRANT_APPLICATION));
        cursor.beginElement(XmlBeansUtils.createMPElement(componentType, FORMS));

        //Now insert our forms here
        formResults.getForms().forEach(formObject -> {
            XmlCursor formsCursor = formObject.newCursor();
            formsCursor.toNextToken();
            formsCursor.moveXml(cursor);
            formsCursor.dispose();
        });
    }

    /**
     * Generate and validate forms for a complex project component.  GG validation is skipped
     * @param child    The component to validate, must be a child node in a hierarchy
     */
    @Override
    public MultiProjectFormGenerationResult generateAndValidateMPComponentForms(ProposalDevelopmentDocumentContract child,
                                                                                String parentSchemaUrl) throws S2SException {
        if (child == null) {
            throw new IllegalArgumentException("a child is required for component validation");
        }

        return createMPComponentDynamicResult(child, parentSchemaUrl);
    }

    private MultiProjectFormGenerationResult createMPComponentDynamicResult(ProposalDevelopmentDocumentContract child,
                                                                            String parentSchemaUrl) throws S2SException {

        ComplexFormGeneratedResult formResults = generateAndValidateMPForms(child);

        final MultiProjectFormGenerationResult result;
        if (formResults.isValid()) {
            result = MultiProjectFormGenerationResult.valid(
                    Map.of(child.getDevelopmentProposal().getProposalNumber(), formResults.getAuditErrors()),
                    getMPComponentDocument(formResults, child.getDevelopmentProposal().getS2sOpportunity().getComponentType(), parentSchemaUrl),
                    Map.of(child.getDevelopmentProposal().getProposalNumber(), formResults.getAttachments())
            );
        } else {
            result = MultiProjectFormGenerationResult.invalid(Map.of(child.getDevelopmentProposal().getProposalNumber(), formResults.getAuditErrors()));
        }

        return result;
    }

    private String getMPComponentDocument(ComplexFormGeneratedResult formResults, String componentType, String parentSchemaUrl) {
        final ApplicationDocument applicationDocument = ApplicationDocument.Factory.newInstance();

        createSubApplicationGroup(applicationDocument, componentType, List.of(formResults));

        final XmlCursor cursor = applicationDocument.newCursor();

        cursor.toStartDoc();
        if (cursor.toFirstChild()) {
            final String defaultNameSpace = cursor.getName().getNamespaceURI();
            final String schemaLocation = defaultNameSpace + " " + parentSchemaUrl;
            cursor.setAttributeText(new QName("http://www.w3.org/2001/XMLSchema-instance", "schemaLocation"), schemaLocation);
        }

        return getXmlFromDocument(applicationDocument);
    }

    @Override
    public FormGenerationResult validateUserAttachedFormFile(S2sUserAttachedFormFileContract s2sUserAttachedFormFile,String formName) throws S2SException {
        if (s2sUserAttachedFormFile == null) {
            throw new IllegalArgumentException("s2sUserAttachedFormFile is null");
        }
        if (StringUtils.isBlank(formName)) {
            throw new IllegalArgumentException("formName is blank");
        }

        try {
            final XmlObject xmlObject = XmlObject.Factory.parse(s2sUserAttachedFormFile.getXmlFile());
            final ValidationResult result = s2SValidatorService.validateForm(xmlObject, formName);
            if (result.isValid()) {
                return FormGenerationResult.valid(result.getErrors(), s2sUserAttachedFormFile.getXmlFile(), Collections.emptyList());
            } else {
                return FormGenerationResult.invalid(result.getErrors());
            }
        } catch (RuntimeException|XmlException e) {
            throw new S2SException(e.getMessage(), e);
        }
    }

    public S2SFormGeneratorRetrievalService getS2SFormGeneratorRetrievalService() {
        return s2SFormGeneratorRetrievalService;
    }

	public void setS2SFormGeneratorRetrievalService(S2SFormGeneratorRetrievalService s2SFormGeneratorRetrievalService) {
		this.s2SFormGeneratorRetrievalService = s2SFormGeneratorRetrievalService;
	}

	public S2SValidatorService getS2SValidatorService() {
		return s2SValidatorService;
	}

	public void setS2SValidatorService(S2SValidatorService validatorService) {
		s2SValidatorService = validatorService;
	}

    public NarrativeService getNarrativeService() {
        return narrativeService;
    }

    public void setNarrativeService(NarrativeService narrativeService) {
        this.narrativeService = narrativeService;
    }

    public S2SDateTimeService getS2SDateTimeService() {
        return s2SDateTimeService;
    }

    public void setS2SDateTimeService(S2SDateTimeService s2SDateTimeService) {
        this.s2SDateTimeService = s2SDateTimeService;
    }

    public FormApplicationService getFormApplicationService() {
        return formApplicationService;
    }

    public void setFormApplicationService(FormApplicationService formApplicationService) {
        this.formApplicationService = formApplicationService;
    }

    private static class ComplexFormGeneratedResult {
        private ProposalDevelopmentDocumentContract developmentDocumentContract;
        private List<XmlObject> forms = new ArrayList<>();
        private List<AttachmentData> attachments = new ArrayList<>();
        private List<AuditError> auditErrors = new ArrayList<>();
        private boolean isValid = true;

        public ComplexFormGeneratedResult(ProposalDevelopmentDocumentContract proposalDevelopmentDocumentContract) {
            this.developmentDocumentContract = proposalDevelopmentDocumentContract;
        }

        public void addAuditErrors(List<AuditError> newErrors) {
            auditErrors.addAll(newErrors);
        }

        public void addAttachments(List<AttachmentData> newAttachments) {
            attachments.addAll(newAttachments);
        }

        public List<XmlObject> getForms() {
            return forms;
        }

        public void addForm(XmlObject form) {
            forms.add(form);
        }

        public boolean isValid() {
            return isValid;
        }

        public void setValid(boolean valid) {
            isValid = valid;
        }

        public ProposalDevelopmentDocumentContract getDevelopmentDocumentContract() {
            return developmentDocumentContract;
        }

        public List<AuditError> getAuditErrors() {
            return auditErrors;
        }

        public List<AttachmentData> getAttachments() {
            return attachments;
        }

        public String getComponentType() {
            return developmentDocumentContract.getDevelopmentProposal().getS2sOpportunity().getComponentType();
        }
    }

    public ResourceLoader getResourceLoader() {
        return resourceLoader;
    }

    public void setResourceLoader(ResourceLoader resourceLoader) {
        this.resourceLoader = resourceLoader;
    }
}
