/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.cd511V11.CD511Document;
import gov.grants.apply.forms.cd511V11.CD511Document.CD511;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.*;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonDto;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.Collections;
import java.util.List;

/**
 * 
 * This class is used to generate XML Document object for grants.gov CD511V1.1. This form is generated using XMLBean API's generated
 * by compiling CD511V1_1 schema.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@FormGenerator("CD511V1_1Generator")
public class CD511V1_1Generator extends S2SBaseFormGenerator<CD511Document> implements S2SFormGeneratorPdfFillable<CD511Document> {

    @Value("http://apply.grants.gov/forms/CD511-V1.1")
    private String namespace;

    @Value("CD511-V1.1")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/CD511-V1.1.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/CD511-V1.1.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    @Autowired
    @Qualifier("departmentalPersonService")
    private DepartmentalPersonService departmentalPersonService;

    private DepartmentalPersonDto aorInfo;

    /**
     * 
     * This method returns CD511Document object based on proposal development document which contains the CD511Document informations
     * OrganizationName,AwardNumber,ProjectName,ContactName,Title,Signature,SubmittedDate for a particular proposal
     * 
     * @return cd511Document(CD511Document){@link XmlObject} of type CD511Document.
     */
    private CD511Document getcd511Document() {
        CD511Document cd511Document = CD511Document.Factory.newInstance();
        CD511 cd511 = CD511.Factory.newInstance();
        cd511.setFormVersion(FormVersion.v1_1.getVersion());

        if (pdDoc.getDevelopmentProposal().getApplicantOrganization() != null) {
            cd511.setOrganizationName(StringUtils.substring(pdDoc.getDevelopmentProposal().getApplicantOrganization().getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
        }
        if (pdDoc.getDevelopmentProposal().getCurrentAwardNumber() != null && !pdDoc.getDevelopmentProposal().getCurrentAwardNumber().equals("")) {
            cd511.setAwardNumber(pdDoc.getDevelopmentProposal().getCurrentAwardNumber());
        }
        if (pdDoc.getDevelopmentProposal().getTitle() != null && !pdDoc.getDevelopmentProposal().getTitle().equals("")) {
            String title = StringUtils.left(pdDoc.getDevelopmentProposal().getTitle(), 60);
            cd511.setProjectName(title);
        }

        String title = "";

        cd511.setContactName(globLibV20Generator.getHumanNameDataType(aorInfo));
        if (aorInfo.getPrimaryTitle() != null && !aorInfo.getPrimaryTitle().equals("")) {
            title = StringUtils.substring(aorInfo.getPrimaryTitle(), 0, PRIMARY_TITLE_MAX_LENGTH);
        }

        cd511.setTitle(title);

        cd511.setSignature(aorInfo.getFullName());
        cd511.setSubmittedDate(aorInfo.getSubmittedDate());
        cd511Document.setCD511(cd511);
        return cd511Document;
    }

    /**
     * This method creates {@link XmlObject} of type {@link CD511Document} by populating data from the given
     * {@link ProposalDevelopmentDocumentContract}
     * 
     * @param proposalDevelopmentDocument for which the {@link XmlObject} needs to be created
     * @return {@link XmlObject} which is generated using the given {@link ProposalDevelopmentDocumentContract}
     */
    @Override
    public CD511Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        aorInfo = departmentalPersonService.getDepartmentalPerson(pdDoc);
        return getcd511Document();
    }

    public DepartmentalPersonService getDepartmentalPersonService() {
        return departmentalPersonService;
    }

    public void setDepartmentalPersonService(DepartmentalPersonService departmentalPersonService) {
        this.departmentalPersonService = departmentalPersonService;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(CD511Document form, List<AttachmentData> attachments) {
        return new Attachments(Collections.emptyMap(), attachments);
    }

    @Override
    public DocumentFactory<CD511Document> factory() {
        return CD511Document.Factory;
    }
}
