/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.nsfCoverPageV11.DegreeTypeDataType;
import gov.grants.apply.forms.nsfCoverPageV11.NSFCoverPageDocument;
import gov.grants.apply.forms.nsfCoverPageV11.NSFCoverPageDocument.NSFCoverPage;
import gov.grants.apply.forms.nsfCoverPageV11.NSFCoverPageDocument.NSFCoverPage.CoPIInfo;
import gov.grants.apply.forms.nsfCoverPageV11.NSFCoverPageDocument.NSFCoverPage.CoPIInfo.CoPI;
import gov.grants.apply.forms.nsfCoverPageV11.NSFCoverPageDocument.NSFCoverPage.NSFUnitConsideration;
import gov.grants.apply.forms.nsfCoverPageV11.NSFCoverPageDocument.NSFCoverPage.OtherInfo;
import gov.grants.apply.forms.nsfCoverPageV11.NSFCoverPageDocument.NSFCoverPage.PIInfo;
import gov.grants.apply.system.attachmentsV10.AttachmentGroupMin1Max100DataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.type.ProposalTypeContract;
import org.kuali.coeus.common.api.ynq.YnqConstant;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonDegreeContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonYnqContract;
import org.kuali.coeus.propdev.api.ynq.ProposalYnqContract;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.List;

/**
 * 
 * This class is used to generate XML Document object for grants.gov NSFCoverPageV1.1. This form is generated using XMLBean API's
 * generated by compiling NSFCoverPageV1.1 schema.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@FormGenerator("NSFCoverPageV1_1Generator")
public class NSFCoverPageV1_1Generator extends NSFCoverPageBaseGenerator<NSFCoverPageDocument> {

    @Value("http://apply.grants.gov/forms/NSF_CoverPage-V1.1")
    private String namespace;

    @Value("NSF_CoverPage-V1.1")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/NSF_CoverPage-V1.1.fo.xsl")
    private List<Resource> stylesheets;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    private static final String QUESTION_ID_ACCOMPLISHMENT_RENEWAL = "5";
    private static final String QUESTION_ID_ISCURRENT_PI = "19";
    private static final int PROGRAM_ANNOUNCEMENT_NUMBER_MAX_LENGTH = 40;
    private static final DegreeTypeDataType.Enum DEFAULT_DEGREE_TYPE = DegreeTypeDataType.UKNW_NO_DEGREE_INFORMATION_SPECIFIED;

    /**
     * 
     * This method returns NSFCoverPageDocument object based on proposal development document which contains the
     * NSFCoverPageDocument informations NSFUnitConsideration,FundingOpportunityNumber,PIInfo,CoPIInfo,OtherInfo,and
     * SingleCopyDocuments for a particular proposal
     * 
     * @return nsfCoverPageDocument {@link XmlObject} of type NSFCoverPageDocument.
     */
    private NSFCoverPageDocument getNSFCoverPage() {

        NSFCoverPageDocument nsfCoverPageDocument = NSFCoverPageDocument.Factory.newInstance();
        NSFCoverPage nsfCoverPage = NSFCoverPage.Factory.newInstance();
        nsfCoverPage.setFormVersion(FormVersion.v1_1.getVersion());
        if (pdDoc.getDevelopmentProposal().getProgramAnnouncementNumber() != null) {
            if (pdDoc.getDevelopmentProposal().getProgramAnnouncementNumber().length() > PROGRAM_ANNOUNCEMENT_NUMBER_MAX_LENGTH) {
                nsfCoverPage.setFundingOpportunityNumber(pdDoc.getDevelopmentProposal().getProgramAnnouncementNumber().substring(0, PROGRAM_ANNOUNCEMENT_NUMBER_MAX_LENGTH));
            }
            else {
                nsfCoverPage.setFundingOpportunityNumber(pdDoc.getDevelopmentProposal().getProgramAnnouncementNumber());
            }
        }
        nsfCoverPage.setNSFUnitConsideration(getNSFUnitConsideration());
        nsfCoverPage.setPIInfo(getPIInfo());
        nsfCoverPage.setCoPIInfo(getCoPI());
        nsfCoverPage.setOtherInfo(getOtherInfo());
        AttachmentGroupMin1Max100DataType attachmentGroup = AttachmentGroupMin1Max100DataType.Factory.newInstance();
        attachmentGroup.setAttachedFileArray(getAttachedFileDataTypes());
        nsfCoverPage.setSingleCopyDocuments(attachmentGroup);
        nsfCoverPageDocument.setNSFCoverPage(nsfCoverPage);
        return nsfCoverPageDocument;
    }

    /**
     * 
     * This method returns PIInfo informations such as DegreeType,DegreeYear,CurrentPI status, for the PI.
     * 
     * @return PIInfo object containing principal investigator Degree details.
     */
    private PIInfo getPIInfo() {
        PIInfo pInfo = PIInfo.Factory.newInstance();
        ProposalPersonContract PI = s2SProposalPersonService.getPrincipalInvestigator(pdDoc);
        if (PI != null) {
            for (ProposalPersonDegreeContract personDegree : PI.getProposalPersonDegrees()) {
                DegreeTypeDataType.Enum degreeType = DEFAULT_DEGREE_TYPE;
                if (personDegree.getDegreeType() != null && personDegree.getDegreeType().getCode() != null) {
                    StringBuilder degreeTypeDetail = new StringBuilder();
                    degreeTypeDetail.append(personDegree.getDegreeType().getCode());
                    degreeTypeDetail.append(": ");
                    degreeTypeDetail.append(personDegree.getDegreeType().getDescription());
                    degreeType = DegreeTypeDataType.Enum.forString(degreeTypeDetail.toString());
                    if(degreeType==null){
                        //Some degrees in nthe database are not available DegreeType. Therefor this extra check.
                        degreeType=DEFAULT_DEGREE_TYPE;
                    }
                }
                else {
                    degreeType = DEFAULT_DEGREE_TYPE;
                }
                pInfo.setDegreeType(degreeType);
                if (personDegree.getGraduationYear() != null) {
                    pInfo.setDegreeYear(getYearAsCalendar(personDegree.getGraduationYear()));
                }
            }
            pInfo.setIsCurrentPI(getYNQAnswer(QUESTION_ID_ISCURRENT_PI));
        }
        return pInfo;
    }

    /**
     * 
     * This method returns CoPIInfo informations such as Name,DegreeType,DegreeYear for the CoPI.
     * 
     * @return CoPIInfo object containing Co-principal investigator Degree details.
     */
    private CoPIInfo getCoPI() {

        CoPIInfo coPIInfo = CoPIInfo.Factory.newInstance();
        int count = 0;
        ProposalPersonContract coInvestigator = null;
        for (ProposalPersonContract proposalPerson : pdDoc.getDevelopmentProposal().getProposalPersons()) {
            if (proposalPerson.getProposalPersonRoleId() != null
                    && proposalPerson.getProposalPersonRoleId().equals(PI_C0_INVESTIGATOR)) {
                count++;
            }
        }
        CoPI[] coPIArray = new CoPI[count];
        count = 0;

        for (ProposalPersonContract proposalPerson : pdDoc.getDevelopmentProposal().getProposalPersons()) {
            if (proposalPerson.getProposalPersonRoleId() != null
                    && proposalPerson.getProposalPersonRoleId().equals(PI_C0_INVESTIGATOR)) {
                coInvestigator = proposalPerson;
                CoPI coPI = CoPI.Factory.newInstance();
                coPI.setName(globLibV20Generator.getHumanNameDataType(coInvestigator));
                for (ProposalPersonDegreeContract personDegree : coInvestigator.getProposalPersonDegrees()) {
                    DegreeTypeDataType.Enum degreeType = DEFAULT_DEGREE_TYPE;
                    if (personDegree!=null && personDegree.getDegreeType() != null && personDegree.getDegreeType().getCode() != null) {
                        StringBuilder degreeTypeDetail = new StringBuilder();
                        degreeTypeDetail.append(personDegree.getDegreeType().getCode());
                        degreeTypeDetail.append(": ");
                        degreeTypeDetail.append(personDegree.getDegreeType().getDescription());
                        degreeType = DegreeTypeDataType.Enum.forString(degreeTypeDetail.toString());
                        if(degreeType==null){
                            //Some degrees in the database are not available DegreeType. Therefor this extra check.
                            degreeType=DEFAULT_DEGREE_TYPE;
                        }
                    }
                    coPI.setDegreeType(degreeType);
                    if (personDegree.getGraduationYear() != null) {
                        coPI.setDegreeYear(getYearAsCalendar(personDegree.getGraduationYear()));
                    }
                }
                coPIArray[count] = coPI;
                count++;
            }
        }
        coPIInfo.setCoPIArray(coPIArray);
        return coPIInfo;
    }

    /**
     * 
     * This method returns Investigator status,DisclosureLobbyingActivities,ExploratoryResearch,HistoricPlaces,
     * HighResolutionGraphics and AccomplishmentRenewal information for the OtherInfo type.
     * 
     * @return OtherInfo object containing other informations about the principal investigator.
     */
    private OtherInfo getOtherInfo() {
        OtherInfo otherInfo = OtherInfo.Factory.newInstance();
        YesNoDataType.Enum yesNoDataType = getYNQAnswer(QUESTION_ID_BEGIN_INVESTIGATOR);
        if (yesNoDataType != null) {
            otherInfo.setIsBeginInvestigator(yesNoDataType);
        }
        yesNoDataType = getLobbyingAnswer();
        if (yesNoDataType != null) {
            otherInfo.setIsDisclosureLobbyingActivities(yesNoDataType);
        }
        yesNoDataType = getYNQAnswer(QUESTION_ID_EXPLORATORY_RESEARCH);
        if (yesNoDataType != null) {
            otherInfo.setIsExploratoryResearch(yesNoDataType);
        }
        yesNoDataType = getYNQAnswer(QUESTION_ID_HISTORIC_PLACES);
        if (yesNoDataType != null) {
            otherInfo.setIsHistoricPlaces(yesNoDataType);
        }

        ProposalTypeContract proposalType = pdDoc.getDevelopmentProposal()
                .getProposalType();
        if (proposalType != null && proposalType.getCode() != null) {
            otherInfo.setIsAccomplishmentRenewal(proposalType.getCode().equals(QUESTION_ID_ACCOMPLISHMENT_RENEWAL) ? YesNoDataType.Y_YES
                    : YesNoDataType.N_NO);
        }
        yesNoDataType = getYNQAnswer(QUESTION_ID_RESOLUTION_GRAPHICS);
        if (yesNoDataType != null) {
            otherInfo.setIsHighResolutionGraphics(yesNoDataType);
        }
        return otherInfo;
    }

    /**
     * 
     * This method YesNo data type YNQ answers based on the ProposalYnq QuestionId
     * 
     * @param questionId Proposal Ynq question id
     * @return answer (YesNoDataType.Enum) corresponding to the question id.
     */
    private YesNoDataType.Enum getYNQAnswer(String questionId) {

        YesNoDataType.Enum answer = null;
        for (ProposalYnqContract proposalYnq : pdDoc.getDevelopmentProposal().getProposalYnqs()) {
            if (proposalYnq.getYnq() != null && proposalYnq.getYnq().getQuestionId().equals(questionId)) {
                if (proposalYnq.getAnswer() != null) {
                    answer = (proposalYnq.getAnswer().equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES
                            : YesNoDataType.N_NO);
                }
            }
        }
        return answer;
    }

    /**
     * 
     * This method YesNo data type Lobbying answers based on the ProposalYnq QuestionId
     * 
     * @return answer (YesNoDataType.Enum) corresponding to Ynq question id.
     */
    private YesNoDataType.Enum getLobbyingAnswer() {

        YesNoDataType.Enum answer = YesNoDataType.N_NO;
        for (ProposalPersonContract proposalPerson : pdDoc.getDevelopmentProposal().getProposalPersons()) {
            if (proposalPerson.getProposalPersonRoleId() != null
                    && proposalPerson.getProposalPersonRoleId().equals(PRINCIPAL_INVESTIGATOR)
                    || PI_C0_INVESTIGATOR.equals(proposalPerson.getProposalPersonRoleId())) {
                for (ProposalPersonYnqContract personYnq : proposalPerson.getProposalPersonYnqs()) {
                    if (personYnq != null) {
                        if (personYnq.getQuestionId() != null && personYnq.getQuestionId().equals(PROPOSAL_YNQ_LOBBYING_ACTIVITIES)) {
                            if (personYnq.getAnswer() != null && personYnq.getAnswer().equals(YnqConstant.YES.code())) {
                                return YesNoDataType.Y_YES;
                            }
                        }
                    }
                }
            }
        }
        return answer;
    }

    /**
     * 
     * This method returns DivisionCode and ProgramCode information for the NSFUnitConsideration type.
     * 
     * @return NSFUnitConsideration object containing unit consideration informations like Division Code and Program code.
     */
    private NSFUnitConsideration getNSFUnitConsideration() {

        NSFUnitConsideration nsfConsideration = NSFUnitConsideration.Factory.newInstance();
        nsfConsideration.setDivisionCode(pdDoc.getDevelopmentProposal().getAgencyDivisionCode());
        nsfConsideration.setProgramCode(pdDoc.getDevelopmentProposal().getAgencyProgramCode());
        return nsfConsideration;
    }


    /**
     * This method creates {@link XmlObject} of type {@link NSFCoverPageDocument} by populating data from the given
     * {@link ProposalDevelopmentDocumentContract}
     * 
     * @param proposalDevelopmentDocument for which the {@link XmlObject} needs to be created
     * @return {@link XmlObject} which is generated using the given {@link ProposalDevelopmentDocumentContract}
     */
    @Override
    public NSFCoverPageDocument getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getNSFCoverPage();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public DocumentFactory<NSFCoverPageDocument> factory() {
        return NSFCoverPageDocument.Factory;
    }
}
