/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.rrKeyPerson30V30.PersonProfileDataType;
import gov.grants.apply.forms.rrKeyPerson30V30.PersonProfileDataType.Profile;
import gov.grants.apply.forms.rrKeyPerson30V30.PersonProfileDataType.Profile.OtherProjectRoleCategory;
import gov.grants.apply.forms.rrKeyPerson30V30.ProjectRoleDataType;
import gov.grants.apply.forms.rrKeyPerson30V30.RRKeyPerson30Document;
import gov.grants.apply.forms.rrKeyPerson30V30.RRKeyPerson30Document.RRKeyPerson30;
import gov.grants.apply.forms.rrKeyPerson30V30.RRKeyPerson30Document.RRKeyPerson30.AdditionalProfilesAttached;
import gov.grants.apply.forms.rrKeyPerson30V30.RRKeyPerson30Document.RRKeyPerson30.BioSketchsAttached;
import gov.grants.apply.forms.rrKeyPerson30V30.RRKeyPerson30Document.RRKeyPerson30.SupportsAttached;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.rolodex.RolodexContract;
import org.kuali.coeus.common.api.rolodex.RolodexService;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.core.DevelopmentProposalContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.util.FieldValueConstants;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

@FormGenerator("RRKeyPersonV3_0Generator")
public class RRKeyPersonV3_0Generator extends RRKeyPersonBase<RRKeyPerson30Document> implements S2SFormGeneratorPdfFillable<RRKeyPerson30Document> {

    private static final int DEPARTMENT_DIVISION_CHARACTER_LIMIT = 100;

	RolodexContract rolodex;

    @Value("http://apply.grants.gov/forms/RR_KeyPerson_3_0-V3.0")
    private String namespace;

    @Value("RR_KeyPerson_3_0-V3.0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/RR_KeyPerson_3_0-V3.0.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/RR_KeyPerson_3_0-V3.0.pdf")
    private Resource pdfForm;

    @Value("150")
    private int sortIndex;

    @Autowired
    @Qualifier("rolodexService")
    private RolodexService rolodexService;

    private RRKeyPerson30Document getRRKeyPerson() {
        RRKeyPerson30Document rrKeyPersonDocument = RRKeyPerson30Document.Factory.newInstance();
        RRKeyPerson30 rrKeyPerson = RRKeyPerson30.Factory.newInstance();
        rrKeyPerson.setFormVersion(FormVersion.v3_0.getVersion());
        rrKeyPerson.setPDPI(getPersonProfilePI());
        rrKeyPerson.setKeyPersonArray(getPersonProfileKeyPerson());
        saveKeyPersonAttachmentsToProposal();
        if (extraPersons.size() > 0) {
    		AttachedFileDataType attachedFileDataType;
            BioSketchsAttached bioSketchAttached = BioSketchsAttached.Factory.newInstance();
            List<NarrativeContract> narratives = (List<NarrativeContract>)pdDoc.getDevelopmentProposal().getNarratives();
            narratives.addAll(getAddedNarratives());
            for (NarrativeContract narrative : narratives) {
                if (narrative.getNarrativeType().getCode() != null) {
                    switch(Integer.parseInt(narrative.getNarrativeType().getCode())){
                        case(BIOSKETCH_DOC_TYPE):
                            attachedFileDataType = addAttachedFileType(narrative);
                        if (attachedFileDataType != null) {
                            bioSketchAttached.setBioSketchAttached(attachedFileDataType);
                        }
                        break;
                        case(CURRENTPENDING_DOC_TYPE):
                            attachedFileDataType = addAttachedFileType(narrative);
                            if (attachedFileDataType != null) {
                                SupportsAttached supportsAttached = SupportsAttached.Factory.newInstance();
                                supportsAttached.setSupportAttached(attachedFileDataType);
                                rrKeyPerson.setSupportsAttached(supportsAttached);
                            }
                        break;
                        case(PROFILE_TYPE):
                            attachedFileDataType = addAttachedFileType(narrative);
                            if(attachedFileDataType != null){
                                AdditionalProfilesAttached additionalProfilesAttached = AdditionalProfilesAttached.Factory.newInstance();
                                additionalProfilesAttached.setAdditionalProfileAttached(attachedFileDataType);
                                rrKeyPerson.setAdditionalProfilesAttached(additionalProfilesAttached);
                            }
                        break;
                    }
                }
            }
            rrKeyPerson.setBioSketchsAttached(bioSketchAttached);
        }

        rrKeyPersonDocument.setRRKeyPerson30(rrKeyPerson);
        return rrKeyPersonDocument;
    }

    private PersonProfileDataType getPersonProfilePI() {

        PersonProfileDataType personProfileDataType = PersonProfileDataType.Factory.newInstance();
        Profile profile = Profile.Factory.newInstance();
        ProposalPersonContract PI = s2SProposalPersonService.getPrincipalInvestigator(pdDoc);
        if (PI != null) {
            if (PI.getPersonId() != null) {
                pIPersonOrRolodexId = PI.getPersonId();
                rolodex = null;
            }
            else if (PI.getRolodexId() != null) {
                pIPersonOrRolodexId = PI.getRolodexId().toString();
                rolodex = rolodexService.getRolodex(Integer.valueOf(pIPersonOrRolodexId));
            }
            profile.setName(globLibV20Generator.getHumanNameDataType(PI));
            if (PI.getDirectoryTitle() != null) {
                if (PI.getDirectoryTitle().length() > DIRECTORY_TITLE_MAX_LENGTH) {
                    profile.setTitle(PI.getDirectoryTitle().substring(0, DIRECTORY_TITLE_MAX_LENGTH));
                }
                else {
                    profile.setTitle(PI.getDirectoryTitle());
                }
            }
            profile.setAddress(globLibV20Generator.getAddressDataTypeV3(PI));
            profile.setPhone(PI.getOfficePhone());
            if (StringUtils.isNotEmpty(PI.getFaxNumber())) {
                profile.setFax(PI.getFaxNumber());
            }
            profile.setEmail(PI.getEmailAddress());
            if (pdDoc.getDevelopmentProposal().getApplicantOrganization() != null) {
                profile.setOrganizationName(StringUtils.substring(pdDoc.getDevelopmentProposal().getApplicantOrganization().getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
            }
            setDepartmentNameToProfile(profile, PI);
            setDivisionNameToProfile(profile, PI);

            if (PI.getEraCommonsUserName() != null) {
                profile.setCredential(PI.getEraCommonsUserName());
            } else {
                if (getSponsorHierarchyService().isSponsorNihMultiplePi(pdDoc.getDevelopmentProposal().getSponsor().getSponsorCode())) {
                	getAuditErrors().add(new AuditError(AuditError.NO_FIELD_ERROR_KEY, ERROR_ERA_COMMON_USER_NAME + PI.getFullName(), AuditError.GG_LINK));
                }
            }
            profile.setProjectRole(ProjectRoleDataType.PD_PI);

            Profile.BioSketchsAttached personBioSketch = Profile.BioSketchsAttached.Factory
                    .newInstance();
            AttachedFileDataType bioSketchAttachment = getPersonnelAttachments(pdDoc, PI.getPersonId(), PI.getRolodexId(),
                    BIOSKETCH_TYPE);
            personBioSketch.setBioSketchAttached(bioSketchAttachment);
            profile.setBioSketchsAttached(personBioSketch);

            AttachedFileDataType supportAttachment = getPersonnelAttachments(pdDoc, PI.getPersonId(), PI.getRolodexId(),
                    CURRENT_PENDING_TYPE);
            if (supportAttachment != null) {
                Profile.SupportsAttached supportsAttached = Profile.SupportsAttached.Factory
                        .newInstance();
                supportsAttached.setSupportAttached(supportAttachment);
                profile.setSupportsAttached(supportsAttached);
            }
            personProfileDataType.setProfile(profile);
        }
        return personProfileDataType;
    }

    private void setDepartmentNameToProfile(Profile profile, ProposalPersonContract keyPerson) {
        if (keyPerson.getHomeUnit() != null && keyPerson.getPerson() != null && keyPerson.getPerson().getUnit() != null) {
            final String departmentName = keyPerson.getPerson().getUnit().getUnitName();
            profile.setDepartment(StringUtils.substring(departmentName, 0, DEPARTMENT_DIVISION_CHARACTER_LIMIT));
        } else {
            if (keyPerson.getRolodexId() != null && keyPerson.getDirectoryDepartment() != null) {
                profile.setDepartment(StringUtils.substring(keyPerson.getDirectoryDepartment(), 0, DEPARTMENT_DIVISION_CHARACTER_LIMIT));
            } else {
                DevelopmentProposalContract developmentProposal = pdDoc.getDevelopmentProposal();
                profile.setDepartment(StringUtils.substring(developmentProposal.getOwnedByUnit().getUnitName(), 0, DEPARTMENT_DIVISION_CHARACTER_LIMIT));
            }
        }
    }

    private void setDivisionNameToProfile(Profile profile, ProposalPersonContract keyPerson) {
        final String divisionName = getS2sDivisionService().getDivision(keyPerson);
        if (divisionName != null) {
            profile.setDivision(divisionName);
        }
    }

    private PersonProfileDataType[] getPersonProfileKeyPerson() {

        List<PersonProfileDataType> personProfileDataTypeList = new ArrayList<>();
        List<? extends ProposalPersonContract> keyPersons = pdDoc.getDevelopmentProposal().getProposalPersons();
        keyPersons.sort(new ProposalPersonComparator());
        List<ProposalPersonContract> nKeyPersons = s2SProposalPersonService.getNKeyPersons(keyPersons, MAX_KEY_PERSON_COUNT);
        extraPersons = keyPersons.stream()
                .filter(kp -> !nKeyPersons.contains(kp))
                .collect(Collectors.toList());

        if (nKeyPersons.size() > 0) {
            for (ProposalPersonContract keyPerson : nKeyPersons) {
                if (pIPersonOrRolodexId != null) {
                    // Don't add PI to keyperson list
                    if (keyPerson.getPersonId() != null && keyPerson.getPersonId().equals(pIPersonOrRolodexId)) {
                        continue;
                    }
                    else if ((keyPerson.getRolodexId() != null) && pIPersonOrRolodexId.equals(keyPerson.getRolodexId().toString())) {
                        continue;
                    }
                }
                if (keyPerson.getPersonId() != null) {
                    pIPersonOrRolodexId = keyPerson.getPersonId();
                    rolodex = null;
                }
                else if (keyPerson.getRolodexId() != null) {
                    pIPersonOrRolodexId = keyPerson.getRolodexId().toString();
                    rolodex = rolodexService.getRolodex(Integer.valueOf(pIPersonOrRolodexId));
                }
                Profile profileKeyPerson = Profile.Factory.newInstance();
                profileKeyPerson.setName(globLibV20Generator.getHumanNameDataType(keyPerson));
                if (keyPerson.getDirectoryTitle() != null) {
                    profileKeyPerson.setTitle(keyPerson.getDirectoryTitle());
                }
                profileKeyPerson.setAddress(globLibV20Generator.getAddressDataTypeV3(keyPerson));
                profileKeyPerson.setPhone(keyPerson.getOfficePhone());
                if (StringUtils.isNotEmpty(keyPerson.getFaxNumber())) {
                    profileKeyPerson.setFax(keyPerson.getFaxNumber());
                }
                profileKeyPerson.setEmail(keyPerson.getEmailAddress());
                if (pdDoc.getDevelopmentProposal().getApplicantOrganization() != null) {
                    profileKeyPerson.setOrganizationName(StringUtils.substring(pdDoc.getDevelopmentProposal().getApplicantOrganization().getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
                }
                setDepartmentNameToProfile(profileKeyPerson, keyPerson);
                setDivisionNameToProfile(profileKeyPerson, keyPerson);

                if (keyPerson.getEraCommonsUserName() != null) {
                    profileKeyPerson.setCredential(keyPerson.getEraCommonsUserName());
                } else {
                    if (getSponsorHierarchyService().isSponsorNihMultiplePi(pdDoc.getDevelopmentProposal().getSponsor().getSponsorCode())) {
                        if (keyPerson.isMultiplePi()) {
                            getAuditErrors().add(new AuditError(AuditError.NO_FIELD_ERROR_KEY, ERROR_ERA_COMMON_USER_NAME + keyPerson.getFullName(),
                                    AuditError.GG_LINK));
                        }
                    }
                }
                if (keyPerson.isMultiplePi() || keyPerson.isCoInvestigator()) {
                	if(getSponsorHierarchyService().isSponsorNihMultiplePi(pdDoc.getDevelopmentProposal().getSponsor().getSponsorCode())){
                	    if (keyPerson.isMultiplePi()) {
                	        profileKeyPerson.setProjectRole(ProjectRoleDataType.PD_PI);
                	    } else {
                	        profileKeyPerson.setProjectRole(ProjectRoleDataType.CO_PD_PI);
                	    }
                	}else{
                		profileKeyPerson.setProjectRole(ProjectRoleDataType.CO_PD_PI);
                	}
                } else {
                    profileKeyPerson.setProjectRole(ProjectRoleDataType.OTHER_SPECIFY);
                    OtherProjectRoleCategory otherProjectRole = OtherProjectRoleCategory.Factory.newInstance();
                    String otherRole;
                    if (keyPerson.getProjectRole() != null) {
                        if (keyPerson.getProjectRole().length() > ROLE_DESCRIPTION_MAX_LENGTH) {
                            otherRole = keyPerson.getProjectRole().substring(0, ROLE_DESCRIPTION_MAX_LENGTH);
                        }
                        else {
                            otherRole = keyPerson.getProjectRole();
                        }
                    }
                    else {
                        otherRole = FieldValueConstants.VALUE_UNKNOWN;
                    }
                    otherProjectRole.setStringValue(otherRole);
                    profileKeyPerson.setOtherProjectRoleCategory(otherProjectRole);
                }

                Profile.BioSketchsAttached personBioSketch = Profile.BioSketchsAttached.Factory
                        .newInstance();
                AttachedFileDataType bioSketchAttachment = getPersonnelAttachments(pdDoc, keyPerson.getPersonId(), keyPerson
                        .getRolodexId(), BIOSKETCH_TYPE);
                personBioSketch.setBioSketchAttached(bioSketchAttachment);
                profileKeyPerson.setBioSketchsAttached(personBioSketch);

                AttachedFileDataType supportAttachment = getPersonnelAttachments(pdDoc, keyPerson.getPersonId(), keyPerson
                        .getRolodexId(), CURRENT_PENDING_TYPE);
                if (supportAttachment != null) {
                    Profile.SupportsAttached supportsAttached = Profile.SupportsAttached.Factory
                            .newInstance();
                    supportsAttached.setSupportAttached(supportAttachment);
                    profileKeyPerson.setSupportsAttached(supportsAttached);
                }

                PersonProfileDataType personProfileDataTypeKeyperson = PersonProfileDataType.Factory.newInstance();
                personProfileDataTypeKeyperson.setProfile(profileKeyPerson);
                personProfileDataTypeList.add(personProfileDataTypeKeyperson);
            }
        }

        return personProfileDataTypeList.toArray(new PersonProfileDataType[0]);
    }

    @Override
    public RRKeyPerson30Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getRRKeyPerson();
    }

    public RolodexService getRolodexService() {
        return rolodexService;
    }

    public void setRolodexService(RolodexService rolodexService) {
        this.rolodexService = rolodexService;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(RRKeyPerson30Document form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {

            final AdditionalProfilesAttached additionalProfilesAttached = form.getRRKeyPerson30().getAdditionalProfilesAttached();
            if (additionalProfilesAttached != null && additionalProfilesAttached.getAdditionalProfileAttached() != null && a.getContentId().equals(additionalProfilesAttached.getAdditionalProfileAttached().getFileLocation().getHref())) {
                return entry("RR_KeyPerson_3_0_P1.footer.optionalFile4", a);
            }

            final BioSketchsAttached bioSketchsAttached = form.getRRKeyPerson30().getBioSketchsAttached();
            if (bioSketchsAttached != null && bioSketchsAttached.getBioSketchAttached() != null && a.getContentId().equals(bioSketchsAttached.getBioSketchAttached().getFileLocation().getHref())) {
                return entry("RR_KeyPerson_3_0_P1.footer.optionalFile5", a);
            }

            final SupportsAttached supportsAttached = form.getRRKeyPerson30().getSupportsAttached();
            if (supportsAttached != null && supportsAttached.getSupportAttached() != null && a.getContentId().equals(supportsAttached.getSupportAttached().getFileLocation().getHref())) {
                return entry("RR_KeyPerson_3_0_P1.footer.optionalFile6", a);
            }

            final PersonProfileDataType pdpi = form.getRRKeyPerson30().getPDPI();
            if (pdpi != null && pdpi.getProfile() != null) {
                final PersonProfileDataType.Profile.BioSketchsAttached pdpiBioSketchsAttached = pdpi.getProfile().getBioSketchsAttached();
                if (pdpiBioSketchsAttached != null && pdpiBioSketchsAttached.getBioSketchAttached() != null && a.getContentId().equals(pdpiBioSketchsAttached.getBioSketchAttached().getFileLocation().getHref())) {
                    return entry("RR_KeyPerson_3_0_P1.PDPI.mandatoryFile0", a);
                }

                final PersonProfileDataType.Profile.SupportsAttached pdpiSupportsAttached = pdpi.getProfile().getSupportsAttached();
                if (pdpiSupportsAttached != null && pdpiSupportsAttached.getSupportAttached() != null && a.getContentId().equals(pdpiSupportsAttached.getSupportAttached().getFileLocation().getHref())) {
                    return entry("RR_KeyPerson_3_0_P1.PDPI.optionalFile0", a);
                }
            }

            final List<PersonProfileDataType> keyPersons = form.getRRKeyPerson30().getKeyPersonList();
            if (keyPersons != null) {
                for (int i = 0; i < keyPersons.size(); i++) {
                    final PersonProfileDataType keyPerson = keyPersons.get(i);
                    if (keyPerson != null && keyPerson.getProfile() != null) {
                        final PersonProfileDataType.Profile.BioSketchsAttached keyPersonBioSketchsAttached = keyPerson.getProfile().getBioSketchsAttached();
                        if (keyPersonBioSketchsAttached != null && keyPersonBioSketchsAttached.getBioSketchAttached() != null && a.getContentId().equals(keyPersonBioSketchsAttached.getBioSketchAttached().getFileLocation().getHref())) {
                            return entry("RR_KeyPerson_3_0_P1.KeyPerson.mandatoryFile" + i, a);
                        }

                        final PersonProfileDataType.Profile.SupportsAttached keyPersonSupportsAttached = keyPerson.getProfile().getSupportsAttached();
                        if (keyPersonSupportsAttached != null && keyPersonSupportsAttached.getSupportAttached() != null && a.getContentId().equals(keyPersonSupportsAttached.getSupportAttached().getFileLocation().getHref())) {
                            return entry("RR_KeyPerson_3_0_P1.KeyPerson.optionalFile" + i, a);
                        }
                    }
                }
            }
            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<RRKeyPerson30Document> factory() {
        return RRKeyPerson30Document.Factory;
    }
}
