/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.kuali.coeus.common.api.person.KcPersonContract;
import org.kuali.coeus.common.api.unit.UnitContract;
import org.kuali.coeus.common.budget.api.modular.ModularBudgetService;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerHeaderContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.s2s.S2SConfigurationService;
import org.kuali.coeus.propdev.api.s2s.S2sOpportunityContract;
import org.kuali.coeus.s2sgen.api.core.ConfigurationConstants;
import org.kuali.coeus.s2sgen.impl.datetime.S2SDateTimeService;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonDto;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonService;
import org.kuali.coeus.s2sgen.impl.person.S2sDivisionService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public abstract class RRSF424BaseGenerator<T extends XmlObject> extends CommonSF424BaseGenerator<T> {

    private static final Logger LOG = LogManager.getLogger(RRSF424BaseGenerator.class);

    protected static final String PRINCIPAL_INVESTIGATOR = "PI";
    protected static final int PRE_APPLICATION = 6;
    protected static final int ADDITIONAL_CONGRESSIONAL_DESTRICT = 59;
    private static final String CONTACT_TYPE_O = "O";
    protected static final String CONTACT_TYPE_I = "I";
    protected static final String STATE_REVIEW_YES = "Y";
    protected static final String STATE_REVIEW_NO = "N";
    protected static final String VALUE_YES = "Yes";
    protected static final int PRIMARY_TITLE_MAX_LENGTH = 45;
    protected static final int DIRECTORY_TITLE_MAX_LENGTH = 45;
    protected static final int DEPARTMENT_NAME_MAX_LENGTH = 30;
    protected static final int ANSWER_EXPLANATION_MAX_LENGTH = 20;
    protected static final int SFLLL_OTHEREXPLANATORY = 86;
    protected static final Integer ANSWER_128 = 128;
    protected static final Integer ANSWER_111 = 111;
    protected static final String NOT_ANSWERED = "No";
    private static final String SUBMISSION_TYPE_CODE = "submissionTypeCode";
    private static final String SUBMISSION_TYPE_DESCRIPTION = "submissionTypeDescription";
    protected static final String KEY_REVISION_CODE = "revisionCode";
    protected static final String KEY_REVISION_OTHER_DESCRIPTION = "revisionOtherDescription";
    private static final int FEDERAL_ID_MAX_LENGTH = 30;

    @Autowired
    @Qualifier("s2SDateTimeService")
    protected S2SDateTimeService s2SDateTimeService;

    @Autowired
    @Qualifier("departmentalPersonService")
    protected DepartmentalPersonService departmentalPersonService;

    @Autowired
    @Qualifier("s2sDivisionService")
    protected S2sDivisionService s2sDivisionService;

    @Autowired
    @Qualifier("s2SConfigurationService")
    protected S2SConfigurationService s2SConfigurationService;

    @Autowired
    @Qualifier("modularBudgetService")
    protected ModularBudgetService modularBudgetService;

    protected String getOtherAgencySubmissionExplanation() {
        Long answerId = getAnswerId(ANSWER_111, getAnswerHeaders());
        String submissionExplanation = null;
        if (getQuestionAnswerService().isAnswerDescriptionRetrievalSupported(answerId)) {
            submissionExplanation = getQuestionAnswerService().getAnswerDescription(answerId);
            submissionExplanation  = submissionExplanation != null ? submissionExplanation.substring(5) : null;
        } else {
            LOG.warn("answer description retrieval not supported for answer id: " + answerId);
        }

        if (submissionExplanation == null) {
            submissionExplanation = "Unknown";
        }

        if (submissionExplanation.length() > ANSWER_EXPLANATION_MAX_LENGTH) {
            return submissionExplanation.substring(0, ANSWER_EXPLANATION_MAX_LENGTH);
        } else {
            return submissionExplanation;
        }
    }

    protected String getContactType() {
        String contactType = s2SConfigurationService.getValueAsString(ConfigurationConstants.PROPOSAL_CONTACT_TYPE);
        if (contactType == null || contactType.length() == 0) {
            contactType = CONTACT_TYPE_O;
        }
        return contactType;
    }

    protected String getFederalId() {
        final String federalId = pdDoc.getDevelopmentProposal().getSponsorProposalNumber();

        return StringUtils.substring(federalId, 0, FEDERAL_ID_MAX_LENGTH);
    }

    protected DepartmentalPersonDto getContactPerson(
            ProposalDevelopmentDocumentContract pdDoc) {
        return departmentalPersonService.getContactPerson(pdDoc);
    }

    /**
     * This method creates and returns Map of submission details like submission type, description and Revision code
     *
     * @param pdDoc Proposal Development Document.
     * @return Map&lt;String, String&gt; Map of submission details.
     */
    protected Map<String, String> getSubmissionType(ProposalDevelopmentDocumentContract pdDoc) {
        Map<String, String> submissionInfo = new HashMap<>();
        S2sOpportunityContract opportunity = pdDoc.getDevelopmentProposal().getS2sOpportunity();
        if (opportunity != null) {
            if (opportunity.getS2sSubmissionType() != null) {
                String submissionTypeCode = opportunity.getS2sSubmissionType().getCode();
                String submissionTypeDescription = opportunity.getS2sSubmissionType().getDescription();
                submissionInfo.put(SUBMISSION_TYPE_CODE, submissionTypeCode);
                submissionInfo.put(SUBMISSION_TYPE_DESCRIPTION, submissionTypeDescription);
            }

            if (opportunity.getS2sRevisionType() != null) {
                String revisionCode = opportunity.getS2sRevisionType().getCode();
                submissionInfo.put(KEY_REVISION_CODE, revisionCode);
            }

            if (opportunity.getRevisionOtherDescription() != null) {
                submissionInfo.put(KEY_REVISION_OTHER_DESCRIPTION, opportunity.getRevisionOtherDescription());
            }
        }
        return submissionInfo;
    }

    protected String getDepartmentName(KcPersonContract kcPerson) {
        return getDepartmentName(kcPerson.getUnit());
    }

    protected String getDepartmentName(UnitContract unit) {
        return unit != null ? StringUtils.substring(unit.getUnitName(), 0, DEPARTMENT_NAME_MAX_LENGTH) : "";
    }

    protected String getDepartmentName(DepartmentalPersonDto departmentalPerson) {
        return departmentalPerson != null ? StringUtils.substring(departmentalPerson.getDirDept(), 0, DEPARTMENT_NAME_MAX_LENGTH) : "";
    }

    protected abstract List<? extends AnswerHeaderContract> getAnswerHeaders();

    public S2SDateTimeService getS2SDateTimeService() {
        return s2SDateTimeService;
    }

    public void setS2SDateTimeService(S2SDateTimeService s2SDateTimeService) {
        this.s2SDateTimeService = s2SDateTimeService;
    }

    public DepartmentalPersonService getDepartmentalPersonService() {
        return departmentalPersonService;
    }

    public void setDepartmentalPersonService(DepartmentalPersonService departmentalPersonService) {
        this.departmentalPersonService = departmentalPersonService;
    }

    public S2sDivisionService getS2sDivisionService() {
        return s2sDivisionService;
    }

    public void setS2sDivisionService(S2sDivisionService s2sDivisionService) {
        this.s2sDivisionService = s2sDivisionService;
    }

    public S2SConfigurationService getS2SConfigurationService() {
        return s2SConfigurationService;
    }

    public void setS2SConfigurationService(S2SConfigurationService s2SConfigurationService) {
        this.s2SConfigurationService = s2SConfigurationService;
    }
}
