/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.sf424AV10.BudgetCategoriesDocument.BudgetCategories;
import gov.grants.apply.forms.sf424AV10.BudgetFirstQuarterAmountsDocument.BudgetFirstQuarterAmounts;
import gov.grants.apply.forms.sf424AV10.BudgetFirstYearAmountsDocument.BudgetFirstYearAmounts;
import gov.grants.apply.forms.sf424AV10.BudgetForecastedCashNeedsDocument.BudgetForecastedCashNeeds;
import gov.grants.apply.forms.sf424AV10.BudgetFourthQuarterAmountsDocument.BudgetFourthQuarterAmounts;
import gov.grants.apply.forms.sf424AV10.BudgetInformationDocument;
import gov.grants.apply.forms.sf424AV10.BudgetInformationType;
import gov.grants.apply.forms.sf424AV10.BudgetSecondQuarterAmountsDocument.BudgetSecondQuarterAmounts;
import gov.grants.apply.forms.sf424AV10.BudgetSummaryDocument.BudgetSummary;
import gov.grants.apply.forms.sf424AV10.BudgetThirdQuarterAmountsDocument.BudgetThirdQuarterAmounts;
import gov.grants.apply.forms.sf424AV10.CategorySetDocument.CategorySet;
import gov.grants.apply.forms.sf424AV10.CategoryTotalsDocument.CategoryTotals;
import gov.grants.apply.forms.sf424AV10.FederalFundsNeededDocument.FederalFundsNeeded;
import gov.grants.apply.forms.sf424AV10.FundsLineItemDocument.FundsLineItem;
import gov.grants.apply.forms.sf424AV10.FundsTotalsDocument.FundsTotals;
import gov.grants.apply.forms.sf424AV10.NonFederalResourcesDocument.NonFederalResources;
import gov.grants.apply.forms.sf424AV10.OtherInformationDocument;
import gov.grants.apply.forms.sf424AV10.ResourceLineItemDocument.ResourceLineItem;
import gov.grants.apply.forms.sf424AV10.ResourceTotalsDocument.ResourceTotals;
import gov.grants.apply.forms.sf424AV10.SummaryLineItemDocument.SummaryLineItem;
import gov.grants.apply.forms.sf424AV10.SummaryTotalsDocument.SummaryTotals;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.budget.api.core.BudgetContract;
import org.kuali.coeus.common.budget.api.core.category.BudgetCategoryMapContract;
import org.kuali.coeus.common.budget.api.core.category.BudgetCategoryMappingContract;
import org.kuali.coeus.common.budget.api.income.BudgetProjectIncomeContract;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemCalculatedAmountContract;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetLineItemContract;
import org.kuali.coeus.common.budget.api.nonpersonnel.BudgetRateAndBaseContract;
import org.kuali.coeus.common.budget.api.period.BudgetPeriodContract;
import org.kuali.coeus.common.budget.api.rate.RateClassType;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.s2s.S2sOpportunityCfdaContract;
import org.kuali.coeus.s2sgen.api.budget.S2SBudgetDto;
import org.kuali.coeus.s2sgen.api.budget.S2SBudgetInfoService;
import org.kuali.coeus.s2sgen.api.core.ConfigurationConstants;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.budget.BudgetPeriodNum;
import org.kuali.coeus.s2sgen.impl.budget.S2SBudgetCategoryMapService;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.sys.api.model.ScaleTwoDecimal;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

@FormGenerator("SF424AV1_0Generator")
public class SF424AV1_0Generator extends SF424BaseGenerator<BudgetInformationDocument> implements S2SFormGeneratorPdfFillable<BudgetInformationDocument> {

    private static final int MAX_LENGTH = 50;
    private static final String RATE_TYPE = "Type=";
    private static final String IDC_BASE = " Base=";
    private static final String IDC_EXP = " IDCExp=";
    private static final String ELLIPSES = "...";
    private static final int MIN_DISPLAY_CHARS = 4;

    private BudgetContract budget = null;

    @Value("http://apply.grants.gov/forms/SF424A-V1.0")
    private String namespace;

    @Value("SF424A-V1.0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/SF424A-V1.0.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/SF424A-V1.0.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    @Autowired
    @Qualifier("s2SBudgetCategoryMapService")
    protected S2SBudgetCategoryMapService s2SBudgetCategoryMapService;

    @Autowired
    @Qualifier("s2SBudgetInfoService")
    protected S2SBudgetInfoService s2sBudgetInfoService;

    private BudgetInformationDocument getSF424A() {
        BudgetInformationDocument budgetInformationDocument = BudgetInformationDocument.Factory.newInstance();

        budget = getS2SCommonBudgetService().getBudget(pdDoc.getDevelopmentProposal());

        BudgetInformationType SF424A = BudgetInformationType.Factory.newInstance();
        SF424A.setCoreSchemaVersion(CORE_SCHEMA_VERSION_1_0);
        SF424A.setFormVersionIdentifier(FormVersion.v1_0.getVersion());
        // value is hard coded
        SF424A.setProgramType(NON_CONSTRUCTION);

        if (budget != null) {
            BudgetSummary budgetSummary = getBudgetSummary();
            BudgetCategories budgetCategories = getBudgetCategories();
            NonFederalResources nonFederalResources = getNonFederalResources();
            BudgetForecastedCashNeeds budgetForecastedCashNeeds = getBudgetForecastedCashNeeds();
            FederalFundsNeeded federalFundsNeeded = getFederalFundsNeeded();

            SF424A.setBudgetSummary(budgetSummary);
            SF424A.setBudgetCategories(budgetCategories);
            SF424A.setNonFederalResources(nonFederalResources);
            SF424A.setBudgetForecastedCashNeeds(budgetForecastedCashNeeds);
            SF424A.setFederalFundsNeeded(federalFundsNeeded);
            SF424A.setOtherInformation(getOtherInformation());
        }

        budgetInformationDocument.setBudgetInformation(SF424A);
        return budgetInformationDocument;
    }

    protected OtherInformationDocument.OtherInformation getOtherInformation() {
        OtherInformationDocument.OtherInformation otherInformation = OtherInformationDocument.OtherInformation.Factory.newInstance();
        otherInformation.setOtherIndirectChargesExplanation(getFormattedIndirectChargeExplanation());
        return otherInformation;
    }

    private String getFormattedIndirectChargeExplanation() {
        ScaleTwoDecimal indirectCostBase = getIndirectBaseCost();
        int totalLength = RATE_TYPE.length() + budget.getRateClass().getDescription().length() + IDC_BASE.length() + indirectCostBase.toString().length() + IDC_EXP.length() + budget.getTotalIndirectCost().toString().length();

        String rateClass = budget.getRateClass().getDescription();

        if (totalLength > MAX_LENGTH) {
            int rateClassLength = budget.getRateClass().getDescription().length();
            int overflowLength = totalLength - MAX_LENGTH;
            int availableLength = rateClassLength - overflowLength;
            if (availableLength <= 0) {
                rateClass = "";
            } else if (availableLength <= MIN_DISPLAY_CHARS + ELLIPSES.length()) {
                rateClass = budget.getRateClass().getDescription().substring(0, availableLength);
            } else {
                rateClass = budget.getRateClass().getDescription().substring(0, availableLength - ELLIPSES.length()) + ELLIPSES;
            }
        }
        String rateSection = rateClass.isBlank() ? "" : RATE_TYPE + rateClass;
        String explanation = rateSection + IDC_BASE + indirectCostBase + IDC_EXP + budget.getTotalIndirectCost();

        return explanation.length() > MAX_LENGTH ? explanation.substring(0, MAX_LENGTH) : explanation;
    }

    protected ScaleTwoDecimal getIndirectBaseCost() {
        ScaleTwoDecimal indirectCostBase = ScaleTwoDecimal.ZERO;
        for (BudgetPeriodContract period : budget.getBudgetPeriods()) {
            for (BudgetLineItemContract lineItem : period.getBudgetLineItems()) {
                for (BudgetRateAndBaseContract rateAndBase : lineItem.getBudgetRateAndBaseList()) {
                    if (rateAndBase.getRateClass().getRateClassType().getCode().equalsIgnoreCase(RateClassType.OVERHEAD.getRateClassType())) {
                        indirectCostBase = indirectCostBase.add(rateAndBase.getBaseCost());
                    }
                }
            }
        }
        return indirectCostBase;
    }

    private BudgetCategories getBudgetCategories() {
        ScaleTwoDecimal constructionCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal contractualCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal equipmentCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal personnelCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal suppliesCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal otherCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal travelCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal programIncome = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal calculatedCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal laSalaries = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal labAllocation = ScaleTwoDecimal.ZERO;

        ScaleTwoDecimal constructionCostCostShare = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal contractualCostCostShare = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal equipmentCostCostShare = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal personnelCostCostShare = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal suppliesCostCostShare = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal otherCostCostShare = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal travelCostCostShare = ScaleTwoDecimal.ZERO;

        ScaleTwoDecimal calculatedCostCostShare = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal laSalariesCostShare = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal labAllocationCostShare = ScaleTwoDecimal.ZERO;

        BudgetCategories budgetCategories = BudgetCategories.Factory.newInstance();
        CategoryTotals categoryTotals = CategoryTotals.Factory.newInstance();
        if (budget == null) {
            return budgetCategories;
        }

        final boolean separateFedNonFed = getS2SConfigurationService().getValuesFromCommaSeparatedParam(ConfigurationConstants.S2SBUDGET_SF424A_SECTIONB_SEPARATE_FED_NONFED_SPONSORS).contains(pdDoc.getDevelopmentProposal().getSponsor().getSponsorCode());

        CategorySet[] categorySetArray = separateFedNonFed ? new CategorySet[2] : new CategorySet[1];
        CategorySet categorySet = CategorySet.Factory.newInstance();
        CategorySet categorySetCostShare = CategorySet.Factory.newInstance();
        if (pdDoc.getDevelopmentProposal().getS2sOpportunity() != null && pdDoc.getDevelopmentProposal().getS2sOpportunity().getS2sSubmissionType() != null) {
            final String opportunityTitle = StringUtils.substring(pdDoc.getDevelopmentProposal().getS2sOpportunity().getOpportunityTitle(), 0, PROGRAM_ANNOUNCEMENT_TITLE_LENGTH);
            categorySet.setActivityTitle(opportunityTitle);
            categorySetCostShare.setActivityTitle(opportunityTitle);
        }

        final List<? extends BudgetCategoryMapContract> budgetCategoryMapList = s2SBudgetCategoryMapService.getBudgetCategoryMapList(
                new ArrayList<>(), new ArrayList<>());


        for (BudgetPeriodContract budgetPeriod : budget.getBudgetPeriods()) {
            for (BudgetLineItemContract budgetLineItem : budgetPeriod.getBudgetLineItems()) {
                for (BudgetCategoryMapContract budgetCategoryMap : budgetCategoryMapList) {
                    for (BudgetCategoryMappingContract budgetCategoryMapping : budgetCategoryMap.getBudgetCategoryMappings()) {
                        if (budgetLineItem.getBudgetCategory().getCode().equals(budgetCategoryMapping.getBudgetCategoryCode())) {
                            if (budgetCategoryMap.getTargetCategoryCode().equals(TARGET_CATEGORY_CODE_CONSTRUCTION)) {
                                constructionCost = constructionCost.add(budgetLineItem.getLineItemCost());
                                if (budgetLineItem.getSubmitCostSharingFlag()) {
                                    constructionCostCostShare = constructionCostCostShare.add(budgetLineItem.getCostSharingAmount());
                                }
                            } else if (budgetCategoryMap.getTargetCategoryCode().equals(TARGET_CATEGORY_CODE_CONTRACTUAL)) {
                                contractualCost = contractualCost.add(budgetLineItem.getLineItemCost());
                                if (budgetLineItem.getSubmitCostSharingFlag()) {
                                    contractualCostCostShare = contractualCostCostShare.add(budgetLineItem.getCostSharingAmount());
                                }
                            } else if (budgetCategoryMap.getTargetCategoryCode().equals(TARGET_CATEGORY_CODE_EQUIPMENT)) {
                                equipmentCost = equipmentCost.add(budgetLineItem.getLineItemCost());
                                if (budgetLineItem.getSubmitCostSharingFlag()) {
                                    equipmentCostCostShare = equipmentCostCostShare.add(budgetLineItem.getCostSharingAmount());
                                }
                            } else if (budgetCategoryMap.getCategoryType().equals(TARGET_CATEGORY_TYPE_CODE_PERSONNEL)) {
                                personnelCost = personnelCost.add(budgetLineItem.getLineItemCost());
                                if (budgetLineItem.getSubmitCostSharingFlag()) {
                                    personnelCostCostShare = personnelCostCostShare.add(budgetLineItem.getCostSharingAmount());
                                }
                            } else if (budgetCategoryMap.getTargetCategoryCode().equals(TARGET_CATEGORY_CODE_SUPPLIES)) {
                                suppliesCost = suppliesCost.add(budgetLineItem.getLineItemCost());
                                if (budgetLineItem.getSubmitCostSharingFlag()) {
                                    suppliesCostCostShare = suppliesCostCostShare.add(budgetLineItem.getCostSharingAmount());
                                }
                            } else if (budgetCategoryMap.getTargetCategoryCode().equals(TARGET_CATEGORY_CODE_TRAVEL)
                                    || budgetCategoryMap.getTargetCategoryCode().equals(TARGET_CATEGORY_CODE_FOREIGN_TRAVEL)) {
                                travelCost = travelCost.add(budgetLineItem.getLineItemCost());
                                if (budgetLineItem.getSubmitCostSharingFlag()) {
                                    travelCostCostShare = travelCostCostShare.add(budgetLineItem.getCostSharingAmount());
                                }
                            } else {
                                otherCost = otherCost.add(budgetLineItem.getLineItemCost());
                                if (budgetLineItem.getSubmitCostSharingFlag()) {
                                    otherCostCostShare = otherCostCostShare.add(budgetLineItem.getCostSharingAmount());
                                }
                            }
                        }
                    }
                }

                for (BudgetLineItemCalculatedAmountContract budgetLineItemCalculatedAmount : budgetLineItem.getBudgetLineItemCalculatedAmounts()) {
                    if (budgetLineItemCalculatedAmount.getRateClass().getRateClassType().getCode().equals(RATE_CLASS_TYPE_EMPLOYEE_BENEFITS)
                            || budgetLineItemCalculatedAmount.getRateClass().getRateClassType().getCode().equals(RATE_CLASS_TYPE_VACATION)) {
                        calculatedCost = calculatedCost.add(budgetLineItemCalculatedAmount.getCalculatedCost());
                        if (budgetLineItem.getSubmitCostSharingFlag()) {
                            calculatedCostCostShare = calculatedCostCostShare.add(budgetLineItemCalculatedAmount.getCalculatedCostSharing());
                        }
                    }
                    if (budgetLineItemCalculatedAmount.getRateClass().getRateClassType().getCode().equals(RATE_CLASS_TYPE_LA_SALARIES)) {
                        laSalaries = laSalaries.add(budgetLineItemCalculatedAmount.getCalculatedCost());
                        if (budgetLineItem.getSubmitCostSharingFlag()) {
                            laSalariesCostShare = laSalariesCostShare.add(budgetLineItemCalculatedAmount.getCalculatedCostSharing());
                        }
                    }
                    if (budgetLineItemCalculatedAmount.getRateClass().getRateClassType().getCode().equals(RATE_CLASS_TYPE_LAB_ALLOCATION)) {
                        labAllocation = labAllocation.add(budgetLineItemCalculatedAmount.getCalculatedCost());
                        if (budgetLineItem.getSubmitCostSharingFlag()) {
                            labAllocationCostShare = labAllocationCostShare.add(budgetLineItemCalculatedAmount.getCalculatedCostSharing());
                        }
                    }
                }
            }
        }

        for (BudgetProjectIncomeContract budgetProjectIncome : budget.getBudgetProjectIncomes()) {
            programIncome = programIncome.add(new ScaleTwoDecimal(budgetProjectIncome.getProjectIncome().bigDecimalValue()));
        }

        BigDecimal otherCostWithLA = labAllocation.bigDecimalValue().add(otherCost.bigDecimalValue());
        BigDecimal otherCostWithLACostShare = labAllocationCostShare.bigDecimalValue().add(otherCostCostShare.bigDecimalValue());

        BigDecimal salariesWithLA = laSalaries.bigDecimalValue().add(personnelCost.bigDecimalValue());
        BigDecimal salariesWithLACostShare = laSalariesCostShare.bigDecimalValue().add(personnelCostCostShare.bigDecimalValue());

        final S2SBudgetDto budgetSummarydto = s2sBudgetInfoService.getBudgetInfo(pdDoc);

        categorySet.setBudgetConstructionRequestedAmount(constructionCost.bigDecimalValue());
        categorySetCostShare.setBudgetConstructionRequestedAmount(constructionCostCostShare.bigDecimalValue());
        categoryTotals.setBudgetConstructionRequestedAmount(categorySet.getBudgetConstructionRequestedAmount().add(categorySetCostShare.getBudgetConstructionRequestedAmount()));


        categorySet.setBudgetContractualRequestedAmount(contractualCost.bigDecimalValue());
        categorySetCostShare.setBudgetContractualRequestedAmount(contractualCostCostShare.bigDecimalValue());
        categoryTotals.setBudgetContractualRequestedAmount(categorySet.getBudgetContractualRequestedAmount().add(categorySetCostShare.getBudgetContractualRequestedAmount()));


        categorySet.setBudgetEquipmentRequestedAmount(equipmentCost.bigDecimalValue());
        categorySetCostShare.setBudgetEquipmentRequestedAmount(equipmentCostCostShare.bigDecimalValue());
        categoryTotals.setBudgetEquipmentRequestedAmount(categorySet.getBudgetEquipmentRequestedAmount().add(categorySetCostShare.getBudgetEquipmentRequestedAmount()));


        categorySet.setBudgetFringeBenefitsRequestedAmount(calculatedCost.bigDecimalValue());
        categorySetCostShare.setBudgetFringeBenefitsRequestedAmount(calculatedCostCostShare.bigDecimalValue());
        categoryTotals.setBudgetFringeBenefitsRequestedAmount(
                categorySet.getBudgetFringeBenefitsRequestedAmount().add(categorySetCostShare.getBudgetFringeBenefitsRequestedAmount()));

        categorySet.setBudgetIndirectChargesAmount(budgetSummarydto.getCumTotalIndirectCosts().bigDecimalValue());
        categorySetCostShare.setBudgetIndirectChargesAmount(budgetSummarydto.getCumTotalIndirectCostSharing().bigDecimalValue());
        categoryTotals.setBudgetIndirectChargesAmount(categorySet.getBudgetIndirectChargesAmount().add(categorySetCostShare.getBudgetIndirectChargesAmount()));

        categorySet.setBudgetOtherRequestedAmount(otherCostWithLA);
        categorySetCostShare.setBudgetOtherRequestedAmount(otherCostWithLACostShare);
        categoryTotals.setBudgetOtherRequestedAmount(categorySet.getBudgetOtherRequestedAmount().add(categorySetCostShare.getBudgetOtherRequestedAmount()));

        categorySet.setBudgetPersonnelRequestedAmount(salariesWithLA);
        categorySetCostShare.setBudgetPersonnelRequestedAmount(salariesWithLACostShare);
        categoryTotals.setBudgetPersonnelRequestedAmount(categorySet.getBudgetPersonnelRequestedAmount().
                add(categorySetCostShare.getBudgetPersonnelRequestedAmount()));

        categorySet.setBudgetSuppliesRequestedAmount(suppliesCost.bigDecimalValue());
        categorySetCostShare.setBudgetSuppliesRequestedAmount(suppliesCostCostShare.bigDecimalValue());
        categoryTotals.setBudgetSuppliesRequestedAmount(categorySet.getBudgetSuppliesRequestedAmount().add(categorySetCostShare.getBudgetSuppliesRequestedAmount()));

        // costshare total
        categorySetCostShare.setBudgetTotalAmount(
                constructionCostCostShare.bigDecimalValue().add(contractualCostCostShare.bigDecimalValue()).
                        add(equipmentCostCostShare.bigDecimalValue()).add(personnelCostCostShare.bigDecimalValue()).
                        add(suppliesCostCostShare.bigDecimalValue()).add(travelCostCostShare.bigDecimalValue()).
                        add(otherCostCostShare.bigDecimalValue().add(calculatedCostCostShare.bigDecimalValue()).
                                add(laSalariesCostShare.bigDecimalValue().add(labAllocationCostShare.bigDecimalValue()))).
                        add(categorySetCostShare.getBudgetIndirectChargesAmount()));

        // budget without cost share total
        categorySet.setBudgetTotalAmount(
                constructionCost.bigDecimalValue().add(contractualCost.bigDecimalValue().
                                add(equipmentCost.bigDecimalValue().add(personnelCost.bigDecimalValue().
                                        add(suppliesCost.bigDecimalValue().add(travelCost.bigDecimalValue()).
                                                add(otherCost.bigDecimalValue().add(calculatedCost.bigDecimalValue().
                                                        add(laSalaries.bigDecimalValue().add(labAllocation.bigDecimalValue())))))))).
                        add(categorySet.getBudgetIndirectChargesAmount()));

        // costshare + budget total
        categoryTotals.setBudgetTotalAmount(categorySet.getBudgetTotalAmount().add(categorySetCostShare.getBudgetTotalAmount()));

        // direct cost totals
        categorySet.setBudgetTotalDirectChargesAmount(
                constructionCost.bigDecimalValue().add(contractualCost.bigDecimalValue().
                        add(equipmentCost.bigDecimalValue().add(personnelCost.bigDecimalValue().
                                add(suppliesCost.bigDecimalValue().add(travelCost.bigDecimalValue()).
                                        add(otherCost.bigDecimalValue().add(calculatedCost.bigDecimalValue().
                                                add(laSalaries.bigDecimalValue().add(labAllocation.bigDecimalValue())))))))));
        // direct cost cost share totals
        categorySetCostShare.setBudgetTotalDirectChargesAmount(constructionCostCostShare.bigDecimalValue().add(contractualCostCostShare.bigDecimalValue()).
                add(equipmentCostCostShare.bigDecimalValue()).add(personnelCostCostShare.bigDecimalValue()).
                add(suppliesCostCostShare.bigDecimalValue()).add(travelCostCostShare.bigDecimalValue()).
                add(otherCostCostShare.bigDecimalValue().add(calculatedCostCostShare.bigDecimalValue()).
                        add(laSalariesCostShare.bigDecimalValue().add(labAllocationCostShare.bigDecimalValue()))));
        // total for direct cost
        categoryTotals.setBudgetTotalDirectChargesAmount(categorySet.getBudgetTotalDirectChargesAmount().add(categorySetCostShare.getBudgetTotalDirectChargesAmount()));

        categorySet.setBudgetTravelRequestedAmount(travelCost.bigDecimalValue());
        categorySetCostShare.setBudgetTravelRequestedAmount(travelCostCostShare.bigDecimalValue());
        categoryTotals.setBudgetTravelRequestedAmount(categorySet.getBudgetTravelRequestedAmount().add(categorySetCostShare.getBudgetTravelRequestedAmount()));

        categorySet.setProgramIncomeAmount(programIncome.bigDecimalValue());
        categorySetCostShare.setProgramIncomeAmount(BigDecimal.ZERO);
        categoryTotals.setProgramIncomeAmount(categorySet.getProgramIncomeAmount().add(categorySetCostShare.getProgramIncomeAmount()));

        if (separateFedNonFed) {
            categorySetArray[0] = categorySet;
            categorySetArray[1] = categorySetCostShare;
        } else {
            categorySet.setBudgetConstructionRequestedAmount(categorySet.getBudgetConstructionRequestedAmount().add(categorySetCostShare.getBudgetConstructionRequestedAmount()));
            categorySet.setBudgetContractualRequestedAmount(categorySet.getBudgetContractualRequestedAmount().add(categorySetCostShare.getBudgetContractualRequestedAmount()));
            categorySet.setBudgetEquipmentRequestedAmount(categorySet.getBudgetEquipmentRequestedAmount().add(categorySetCostShare.getBudgetEquipmentRequestedAmount()));
            categorySet.setBudgetFringeBenefitsRequestedAmount(categorySet.getBudgetFringeBenefitsRequestedAmount().add(categorySetCostShare.getBudgetFringeBenefitsRequestedAmount()));
            categorySet.setBudgetIndirectChargesAmount(categorySet.getBudgetIndirectChargesAmount().add(categorySetCostShare.getBudgetIndirectChargesAmount()));
            categorySet.setBudgetOtherRequestedAmount(categorySet.getBudgetOtherRequestedAmount().add(categorySetCostShare.getBudgetOtherRequestedAmount()));
            categorySet.setBudgetPersonnelRequestedAmount(categorySet.getBudgetPersonnelRequestedAmount().add(categorySetCostShare.getBudgetPersonnelRequestedAmount()));
            categorySet.setBudgetSuppliesRequestedAmount(categorySet.getBudgetSuppliesRequestedAmount().add(categorySetCostShare.getBudgetSuppliesRequestedAmount()));
            categorySet.setBudgetTotalAmount(categorySet.getBudgetTotalAmount().add(categorySetCostShare.getBudgetTotalAmount()));
            categorySet.setBudgetTotalDirectChargesAmount(categorySet.getBudgetTotalDirectChargesAmount().add(categorySetCostShare.getBudgetTotalDirectChargesAmount()));
            categorySet.setBudgetTravelRequestedAmount(categorySet.getBudgetTravelRequestedAmount().add(categorySetCostShare.getBudgetTravelRequestedAmount()));
            categorySet.setProgramIncomeAmount(categorySet.getProgramIncomeAmount().add(categorySetCostShare.getProgramIncomeAmount()));

            categorySetArray[0] = categorySet;
        }

        budgetCategories.setCategorySetArray(categorySetArray);
        budgetCategories.setCategoryTotals(categoryTotals);
        return budgetCategories;
    }

    private BudgetSummary getBudgetSummary() {
        BudgetSummary budgetSummary = BudgetSummary.Factory.newInstance();
        SummaryLineItem[] summaryLineItemArray = new SummaryLineItem[1];
        SummaryLineItem summaryLineItem = SummaryLineItem.Factory.newInstance();
        boolean hasBudgetLineItem = false;

        if (pdDoc.getDevelopmentProposal().getS2sOpportunity() != null && pdDoc.getDevelopmentProposal().getS2sOpportunity().getS2sSubmissionType() != null) {
            final String opportunityTitle = StringUtils.substring(pdDoc.getDevelopmentProposal().getS2sOpportunity().getOpportunityTitle(), 0, PROGRAM_ANNOUNCEMENT_TITLE_LENGTH);
            summaryLineItem.setActivityTitle(opportunityTitle);
            pdDoc.getDevelopmentProposal().getS2sOpportunity().getS2sOpportunityCfdas().stream().map(S2sOpportunityCfdaContract::getCfdaNumber).findFirst().ifPresent(summaryLineItem::setCFDANumber);
        }
        if (budget != null) {
            ScaleTwoDecimal fedNonFedCost = budget.getTotalCost();
            ScaleTwoDecimal costSharingAmount = ScaleTwoDecimal.ZERO;

            for (BudgetPeriodContract budgetPeriod : budget.getBudgetPeriods()) {
                for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
                    hasBudgetLineItem = true;
                    if (budget.getSubmitCostSharingFlag() && lineItem.getSubmitCostSharingFlag()) {
                        costSharingAmount = costSharingAmount.add(lineItem.getCostSharingAmount());
                        List<? extends BudgetLineItemCalculatedAmountContract> calculatedAmounts = lineItem.getBudgetLineItemCalculatedAmounts();
                        for (BudgetLineItemCalculatedAmountContract budgetLineItemCalculatedAmount : calculatedAmounts) {
                            costSharingAmount = costSharingAmount.add(budgetLineItemCalculatedAmount.getCalculatedCostSharing());
                        }

                    }
                }
            }
            if (!hasBudgetLineItem && budget.getSubmitCostSharingFlag()) {
                costSharingAmount = budget.getCostSharingAmount();
            }
            fedNonFedCost = fedNonFedCost.add(costSharingAmount);
            summaryLineItem.setBudgetFederalNewOrRevisedAmount(budget.getTotalCost().bigDecimalValue());
            summaryLineItem.setBudgetNonFederalNewOrRevisedAmount(costSharingAmount.bigDecimalValue());
            summaryLineItem.setBudgetTotalNewOrRevisedAmount(fedNonFedCost.bigDecimalValue());
            summaryLineItemArray[0] = summaryLineItem;
            budgetSummary.setSummaryLineItemArray(summaryLineItemArray);

            SummaryTotals summaryTotals = SummaryTotals.Factory.newInstance();
            summaryTotals.setBudgetFederalNewOrRevisedAmount(budget.getTotalCost().bigDecimalValue());
            summaryTotals.setBudgetNonFederalNewOrRevisedAmount(costSharingAmount.bigDecimalValue());
            summaryTotals.setBudgetTotalNewOrRevisedAmount(fedNonFedCost.bigDecimalValue());
            budgetSummary.setSummaryTotals(summaryTotals);
        }

        return budgetSummary;
    }

    private NonFederalResources getNonFederalResources() {
        NonFederalResources nonFederalResources = NonFederalResources.Factory.newInstance();
        ResourceLineItem[] resourceLineItemArray = new ResourceLineItem[1];
        ResourceLineItem resourceLineItem = ResourceLineItem.Factory.newInstance();
        boolean hasBudegetLineItem = false;
        if (pdDoc.getDevelopmentProposal().getS2sOpportunity() != null && pdDoc.getDevelopmentProposal().getS2sOpportunity().getS2sSubmissionType() != null) {
            final String opportunityTitle = StringUtils.substring(pdDoc.getDevelopmentProposal().getS2sOpportunity().getOpportunityTitle(), 0, PROGRAM_ANNOUNCEMENT_TITLE_LENGTH);
            resourceLineItem.setActivityTitle(opportunityTitle);
        }
        if (budget != null) {
            ScaleTwoDecimal fedNonFedCost = ScaleTwoDecimal.ZERO;
            for (BudgetPeriodContract budgetPeriod : budget.getBudgetPeriods()) {
                for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
                    hasBudegetLineItem = true;
                    if (budget.getSubmitCostSharingFlag() && lineItem.getSubmitCostSharingFlag()) {
                        fedNonFedCost = fedNonFedCost.add(lineItem.getCostSharingAmount());
                        for (BudgetLineItemCalculatedAmountContract budgetLineItemCalculatedAmount : lineItem.getBudgetLineItemCalculatedAmounts()) {
                            fedNonFedCost = fedNonFedCost.add(budgetLineItemCalculatedAmount.getCalculatedCostSharing());
                        }
                    }
                }
            }
            if (!hasBudegetLineItem && budget.getSubmitCostSharingFlag()) {
                fedNonFedCost = fedNonFedCost.add(budget.getCostSharingAmount());
            }
            resourceLineItem.setBudgetApplicantContributionAmount(fedNonFedCost.bigDecimalValue());
            resourceLineItem.setBudgetTotalContributionAmount(fedNonFedCost.bigDecimalValue());
            resourceLineItemArray[0] = resourceLineItem;
            nonFederalResources.setResourceLineItemArray(resourceLineItemArray);

            ResourceTotals resourceTotals = ResourceTotals.Factory.newInstance();
            resourceTotals.setBudgetApplicantContributionAmount(fedNonFedCost.bigDecimalValue());
            resourceTotals.setBudgetTotalContributionAmount(fedNonFedCost.bigDecimalValue());
            nonFederalResources.setResourceTotals(resourceTotals);
        }

        return nonFederalResources;
    }

    private BudgetForecastedCashNeeds getBudgetForecastedCashNeeds() {
        ScaleTwoDecimal totalFedCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal costSharing = ScaleTwoDecimal.ZERO;
        BigDecimal totalEstimation = ScaleTwoDecimal.ZERO.bigDecimalValue();
        BigDecimal costShareEstimation = ScaleTwoDecimal.ZERO.bigDecimalValue();
        BigDecimal totalFedEstimation = ScaleTwoDecimal.ZERO.bigDecimalValue();
        BudgetForecastedCashNeeds budgetForecastedCashNeeds = BudgetForecastedCashNeeds.Factory.newInstance();
        if (budget != null) {
            BudgetFirstYearAmounts budgetFirstYearAmounts = BudgetFirstYearAmounts.Factory.newInstance();
            BudgetFirstQuarterAmounts budgetFirstQuarterAmounts = BudgetFirstQuarterAmounts.Factory.newInstance();
            BudgetSecondQuarterAmounts budgetSecondQuarterAmounts = BudgetSecondQuarterAmounts.Factory.newInstance();
            BudgetThirdQuarterAmounts budgetThirdQuarterAmounts = BudgetThirdQuarterAmounts.Factory.newInstance();
            BudgetFourthQuarterAmounts budgetFourthQuarterAmounts = BudgetFourthQuarterAmounts.Factory.newInstance();
            for (BudgetPeriodContract budgetPeriod : budget.getBudgetPeriods()) {
                for (BudgetLineItemContract lineItem : budgetPeriod.getBudgetLineItems()) {
                    if (budget.getSubmitCostSharingFlag() && lineItem.getSubmitCostSharingFlag()) {
                        if (budgetPeriod.getBudgetPeriod() == BudgetPeriodNum.P1.getNum()) {
                            costSharing = costSharing.add(lineItem.getCostSharingAmount());
                            for (BudgetLineItemCalculatedAmountContract budgetLineItemCalculatedAmount : lineItem.getBudgetLineItemCalculatedAmounts()) {
                                costSharing = costSharing.add(budgetLineItemCalculatedAmount.getCalculatedCostSharing());
                            }
                        }
                    }
                }
                if (budgetPeriod.getBudgetPeriod() == BudgetPeriodNum.P1.getNum()) {
                    totalFedCost = budgetPeriod.getTotalCost();
                    totalFedEstimation = totalFedCost.bigDecimalValue().divide(new ScaleTwoDecimal(4).bigDecimalValue(), RoundingMode.HALF_UP);
                    costShareEstimation = costSharing.bigDecimalValue().divide(new ScaleTwoDecimal(4).bigDecimalValue(), RoundingMode.HALF_UP);
                    totalEstimation = totalFedEstimation.add(costShareEstimation);
                }
            }

            budgetFirstYearAmounts.setBudgetFederalForecastedAmount(totalFedCost.bigDecimalValue());
            budgetFirstYearAmounts.setBudgetNonFederalForecastedAmount(costSharing.bigDecimalValue());
            budgetFirstYearAmounts.setBudgetTotalForecastedAmount(costSharing.add(totalFedCost).bigDecimalValue());

            budgetForecastedCashNeeds.setBudgetFirstYearAmounts(budgetFirstYearAmounts);

            budgetFirstQuarterAmounts.setBudgetFederalForecastedAmount(totalFedEstimation);
            budgetFirstQuarterAmounts.setBudgetNonFederalForecastedAmount(costShareEstimation);
            budgetFirstQuarterAmounts.setBudgetTotalForecastedAmount(totalEstimation);

            budgetForecastedCashNeeds.setBudgetFirstQuarterAmounts(budgetFirstQuarterAmounts);

            budgetSecondQuarterAmounts.setBudgetFederalForecastedAmount(totalFedEstimation);
            budgetSecondQuarterAmounts.setBudgetNonFederalForecastedAmount(costShareEstimation);
            budgetSecondQuarterAmounts.setBudgetTotalForecastedAmount(totalEstimation);

            budgetForecastedCashNeeds.setBudgetSecondQuarterAmounts(budgetSecondQuarterAmounts);

            budgetThirdQuarterAmounts.setBudgetFederalForecastedAmount(totalFedEstimation);
            budgetThirdQuarterAmounts.setBudgetNonFederalForecastedAmount(costShareEstimation);
            budgetThirdQuarterAmounts.setBudgetTotalForecastedAmount(totalEstimation);

            budgetForecastedCashNeeds.setBudgetThirdQuarterAmounts(budgetThirdQuarterAmounts);

            budgetFourthQuarterAmounts.setBudgetFederalForecastedAmount(totalFedEstimation);
            budgetFourthQuarterAmounts.setBudgetNonFederalForecastedAmount(costShareEstimation);
            budgetFourthQuarterAmounts.setBudgetTotalForecastedAmount(totalEstimation);

            budgetForecastedCashNeeds.setBudgetFourthQuarterAmounts(budgetFourthQuarterAmounts);
        }
        return budgetForecastedCashNeeds;
    }

    private FederalFundsNeeded getFederalFundsNeeded() {

        ScaleTwoDecimal firstYearNetCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal secondYearNetCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal thirdYearNetCost = ScaleTwoDecimal.ZERO;
        ScaleTwoDecimal fourthYearNetCost = ScaleTwoDecimal.ZERO;

        FederalFundsNeeded federalFundsNeeded = FederalFundsNeeded.Factory.newInstance();
        if (budget == null) {
            return federalFundsNeeded;
        }

        FundsTotals fundsTotals = FundsTotals.Factory.newInstance();

        FundsLineItem[] fundsLineItemArray = new FundsLineItem[1];
        FundsLineItem fundsLineItem = FundsLineItem.Factory.newInstance();

        if (pdDoc.getDevelopmentProposal().getS2sOpportunity() != null && pdDoc.getDevelopmentProposal().getS2sOpportunity().getS2sSubmissionType() != null) {
            final String opportunityTitle = StringUtils.substring(pdDoc.getDevelopmentProposal().getS2sOpportunity().getOpportunityTitle(), 0, PROGRAM_ANNOUNCEMENT_TITLE_LENGTH);
            fundsLineItem.setActivityTitle(opportunityTitle);
        }

        for (BudgetPeriodContract budgetPeriod : budget.getBudgetPeriods()) {
            if (budgetPeriod.getBudgetPeriod() == BudgetPeriodNum.P2.getNum()) {
                firstYearNetCost = firstYearNetCost.add(budgetPeriod.getTotalCost());
            }
            if (budgetPeriod.getBudgetPeriod() == BudgetPeriodNum.P3.getNum()) {
                secondYearNetCost = secondYearNetCost.add(budgetPeriod.getTotalCost());
            }
            if (budgetPeriod.getBudgetPeriod() == BudgetPeriodNum.P4.getNum()) {
                thirdYearNetCost = thirdYearNetCost.add(budgetPeriod.getTotalCost());
            }
            if (budgetPeriod.getBudgetPeriod() == BudgetPeriodNum.P5.getNum()) {
                fourthYearNetCost = fourthYearNetCost.add(budgetPeriod.getTotalCost());
            }
        }
        fundsLineItem.setBudgetFirstYearAmount(firstYearNetCost.bigDecimalValue());
        fundsTotals.setBudgetFirstYearAmount(firstYearNetCost.bigDecimalValue());
        fundsLineItem.setBudgetSecondYearAmount(secondYearNetCost.bigDecimalValue());
        fundsTotals.setBudgetSecondYearAmount(secondYearNetCost.bigDecimalValue());
        fundsLineItem.setBudgetThirdYearAmount(thirdYearNetCost.bigDecimalValue());
        fundsTotals.setBudgetThirdYearAmount(thirdYearNetCost.bigDecimalValue());
        fundsLineItem.setBudgetFourthYearAmount(fourthYearNetCost.bigDecimalValue());
        fundsTotals.setBudgetFourthYearAmount(fourthYearNetCost.bigDecimalValue());

        fundsLineItemArray[0] = fundsLineItem;
        federalFundsNeeded.setFundsLineItemArray(fundsLineItemArray);
        federalFundsNeeded.setFundsTotals(fundsTotals);
        return federalFundsNeeded;
    }

    @Override
    public BudgetInformationDocument getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getSF424A();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    public S2SBudgetCategoryMapService getS2SBudgetCategoryMapService() {
        return s2SBudgetCategoryMapService;
    }

    public void setS2SBudgetCategoryMapService(S2SBudgetCategoryMapService s2SBudgetCategoryMapService) {
        this.s2SBudgetCategoryMapService = s2SBudgetCategoryMapService;
    }

    public S2SBudgetInfoService getS2sBudgetInfoService() {
        return s2sBudgetInfoService;
    }

    public void setS2sBudgetInfoService(S2SBudgetInfoService s2sBudgetInfoService) {
        this.s2sBudgetInfoService = s2sBudgetInfoService;
    }

    @Override
    public Attachments getMappedAttachments(BudgetInformationDocument form, List<AttachmentData> attachments) {
        return new Attachments(Collections.emptyMap(), attachments);
    }

    @Override
    public DocumentFactory<BudgetInformationDocument> factory() {
        return BudgetInformationDocument.Factory;
    }
}
