/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.sflll20V20.AwardeeDataType;
import gov.grants.apply.forms.sflll20V20.LobbyingActivitiesDisclosure20Document;
import gov.grants.apply.forms.sflll20V20.LobbyingActivitiesDisclosure20Document.LobbyingActivitiesDisclosure20;
import gov.grants.apply.forms.sflll20V20.LobbyingActivitiesDisclosure20Document.LobbyingActivitiesDisclosure20.*;
import gov.grants.apply.forms.sflll20V20.LobbyingActivitiesDisclosure20Document.LobbyingActivitiesDisclosure20.IndividualsPerformingServices.Individual;
import gov.grants.apply.forms.sflll20V20.ReportDataType;
import gov.grants.apply.forms.sflll20V20.ReportEntityDataType;
import gov.grants.apply.system.globalLibraryV20.HumanNameDataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.org.OrganizationContract;
import org.kuali.coeus.common.api.rolodex.RolodexContract;
import org.kuali.coeus.common.api.rolodex.RolodexService;
import org.kuali.coeus.common.api.sponsor.SponsorContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.location.ProposalSiteContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonDto;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.Collections;
import java.util.List;

@FormGenerator("SFLLL2_0V2_0Generator")
public class SFLLL2_0V2_0Generator extends SFLLLBaseGenerator<LobbyingActivitiesDisclosure20Document> implements S2SFormGeneratorPdfFillable<LobbyingActivitiesDisclosure20Document> {

    private DepartmentalPersonDto aorInfo;

    @Value("http://apply.grants.gov/forms/SFLLL_2_0-V2.0")
    private String namespace;

    @Value("SFLLL2_0V2_0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/SFLLL_2_0-V2.0.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/SFLLL_2_0-V2.0.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    @Autowired
    @Qualifier("rolodexService")
    private RolodexService rolodexService;

    private LobbyingActivitiesDisclosure20Document getLobbyingActivitiesDisclosure() {

        LobbyingActivitiesDisclosure20Document lobbyinADDocument = LobbyingActivitiesDisclosure20Document.Factory.newInstance();
        LobbyingActivitiesDisclosure20 lobbyingDisclosure = LobbyingActivitiesDisclosure20.Factory.newInstance();
        lobbyingDisclosure.setFormVersion(FormVersion.v2_0.getVersion());
        lobbyingDisclosure.setFederalActionType(FederalActionType.GRANT);
        lobbyingDisclosure.setFederalActionStatus(FederalActionStatus.BID_OFFER);
        lobbyingDisclosure.setReportType(ReportDataType.INITIAL_FILING);
        lobbyingDisclosure.setReportEntity(getReportEntity());
        lobbyingDisclosure.setFederalAgencyDepartment("");
        SponsorContract sponsor = pdDoc.getDevelopmentProposal().getSponsor();
        if (sponsor != null) {
            if (sponsor.getSponsorName() != null) {
                lobbyingDisclosure.setFederalAgencyDepartment(StringUtils.substring(sponsor.getSponsorName(), 0, SPONSOR_NAME_MAX_LENGTH));
            }
        } else {
            String primeSponsorCode = pdDoc.getDevelopmentProposal().getPrimeSponsor().getSponsorCode();
            if (primeSponsorCode != null) {
                SponsorContract primeSponsor = pdDoc.getDevelopmentProposal().getPrimeSponsor();
                if (primeSponsor.getSponsorName() != null) {
                    lobbyingDisclosure.setFederalAgencyDepartment(StringUtils.substring(primeSponsor.getSponsorName(), 0, SPONSOR_NAME_MAX_LENGTH));
                }
            }
        }
        lobbyingDisclosure.setFederalProgramName(getFedProgramName());
        lobbyingDisclosure.setLobbyingRegistrant(getLobbyingRegistrant());
        lobbyingDisclosure.setIndividualsPerformingServices(getIndividualsPerformingServices());
        lobbyingDisclosure.setSignatureBlock(getSignatureBlock());
        lobbyinADDocument.setLobbyingActivitiesDisclosure20(lobbyingDisclosure);
        return lobbyinADDocument;
    }

    private SignatureBlock getSignatureBlock() {

        SignatureBlock signatureBlock = SignatureBlock.Factory.newInstance();
        signatureBlock.setName(globLibV20Generator.getHumanNameDataType(aorInfo));
        if (aorInfo.getPrimaryTitle() != null) {
            signatureBlock.setTitle(StringUtils.substring(aorInfo.getPrimaryTitle(), 0, PRIMARY_TITLE_MAX_LENGTH));
        }

        if (aorInfo.getOfficePhone() != null) {
            signatureBlock.setTelephone(StringUtils.substring(aorInfo.getOfficePhone(), 0, PHONE_MAX_LENGTH));
        }

        signatureBlock.setSignature(aorInfo.getFullName());

        signatureBlock.setSignedDate(aorInfo.getSubmittedDate());
        return signatureBlock;
    }

    private LobbyingRegistrant getLobbyingRegistrant() {

        final LobbyingRegistrant lobbyingRegistrant = LobbyingRegistrant.Factory.newInstance();
        final OrganizationContract organization = pdDoc.getDevelopmentProposal().getApplicantOrganization().getOrganization();
        final RolodexContract rolodex = organization != null && organization.getLobbyingRegistrant() != null ? rolodexService.getRolodex(organization.getLobbyingRegistrant()) : null;

        if (rolodex != null) {
            final HumanNameDataType individualName = HumanNameDataType.Factory.newInstance();
            final LobbyingRegistrant.Address address = LobbyingRegistrant.Address.Factory.newInstance();
            if (rolodex.getFirstName() != null) {
                individualName.setFirstName(StringUtils.substring(rolodex.getFirstName(), 0, FIRST_NAME_MAX_LENGTH));
            }

            if (rolodex.getMiddleName() != null) {
                individualName.setMiddleName(StringUtils.substring(rolodex.getMiddleName(), 0, MIDDLE_NAME_MAX_LENGTH));
            }

            if (rolodex.getLastName() != null) {
                individualName.setLastName(StringUtils.substring(rolodex.getLastName(), 0, LAST_NAME_MAX_LENGTH));
            }

            if (rolodex.getAddressLine1() != null) {
                address.setStreet1(StringUtils.substring(rolodex.getAddressLine1(), 0, ADDRESS_LINE1_MAX_LENGTH));
            }
            if (rolodex.getAddressLine2() != null) {
                address.setStreet2(StringUtils.substring(rolodex.getAddressLine2(), 0, ADDRESS_LINE2_MAX_LENGTH));
            }
            if (rolodex.getCity() != null) {
                address.setCity(StringUtils.substring(rolodex.getCity(), 0, CITY_MAX_LENGTH));
            }
            globLibV20Generator.getStateCodeDataTypeV3(rolodex.getCountryCode(), rolodex.getState())
                    .ifPresent(address::setState);

            address.setZipPostalCode(rolodex.getPostalCode());

            lobbyingRegistrant.setIndividualName(individualName);
            lobbyingRegistrant.setAddress(address);
        } else {
            final HumanNameDataType naName = HumanNameDataType.Factory.newInstance();
            naName.setFirstName(NOT_APPLICABLE);
            naName.setLastName(NOT_APPLICABLE);

            lobbyingRegistrant.setIndividualName(naName);
        }

        return lobbyingRegistrant;
    }

    private IndividualsPerformingServices getIndividualsPerformingServices() {

        IndividualsPerformingServices individualServices = IndividualsPerformingServices.Factory.newInstance();
        Individual individual = Individual.Factory.newInstance();

        final OrganizationContract organization = pdDoc.getDevelopmentProposal().getApplicantOrganization().getOrganization();
        final RolodexContract rolodex = organization != null && organization.getLobbyingIndividual() != null ? rolodexService.getRolodex(organization.getLobbyingIndividual()) : null;

        if (rolodex != null) {
            HumanNameDataType name = HumanNameDataType.Factory.newInstance();
            final Individual.Address address = Individual.Address.Factory.newInstance();

            if (rolodex.getFirstName() != null) {
                name.setFirstName(StringUtils.substring(rolodex.getFirstName(), 0, FIRST_NAME_MAX_LENGTH));
            }

            if (rolodex.getMiddleName() != null) {
                name.setMiddleName(StringUtils.substring(rolodex.getMiddleName(), 0, MIDDLE_NAME_MAX_LENGTH));
            }

            if (rolodex.getLastName() != null) {
                name.setLastName(StringUtils.substring(rolodex.getLastName(), 0, LAST_NAME_MAX_LENGTH));
            }

            if (rolodex.getAddressLine1() != null) {
                address.setStreet1(StringUtils.substring(rolodex.getAddressLine1(), 0, ADDRESS_LINE1_MAX_LENGTH));
            }
            if (rolodex.getAddressLine2() != null) {
                address.setStreet2(StringUtils.substring(rolodex.getAddressLine2(), 0, ADDRESS_LINE2_MAX_LENGTH));
            }
            if (rolodex.getCity() != null) {
                address.setCity(StringUtils.substring(rolodex.getCity(), 0, CITY_MAX_LENGTH));
            }
            globLibV20Generator.getStateCodeDataTypeV3(rolodex.getCountryCode(), rolodex.getState())
                    .ifPresent(address::setState);

            address.setZipPostalCode(rolodex.getPostalCode());

            individual.setName(name);
            individual.setAddress(address);
        } else {
            final HumanNameDataType naName = HumanNameDataType.Factory.newInstance();

            naName.setFirstName(NOT_APPLICABLE);
            naName.setLastName(NOT_APPLICABLE);

            individual.setName(naName);
        }

        final Individual[] individualArray = new Individual[1];
        individualArray[0] = individual;
        individualServices.setIndividualArray(individualArray);
        return individualServices;
    }

    private FederalProgramName getFedProgramName() {

        FederalProgramName federalProgramName = FederalProgramName.Factory.newInstance();
        if (pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle() != null) {
            federalProgramName.setFederalProgramName(StringUtils.substring(pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle(), 0, PROGRAM_ANNOUNCEMENT_TITLE_MAX_LENGTH));
        }

        final String cfdaNumber = getFirstCfdaNumber();
        if (cfdaNumber != null) {
            federalProgramName.setCFDANumber(cfdaNumber);
        }
        return federalProgramName;
    }

    private ReportEntity getReportEntity() {

        ReportEntity reportEntity = ReportEntity.Factory.newInstance();
        reportEntity.setReportEntityType(ReportEntityDataType.PRIME);
        reportEntity.setReportEntityIsPrime(YesNoDataType.Y_YES);

        AwardeeDataType awardeeDataType = AwardeeDataType.Factory.newInstance();
        awardeeDataType.setEntityType(ReportEntityDataType.PRIME);

        ProposalSiteContract proposalSite = pdDoc.getDevelopmentProposal().getApplicantOrganization();
        OrganizationContract organization = proposalSite.getOrganization();

        if (organization != null) {
            if (proposalSite.getLocationName() != null) {
                awardeeDataType.setOrganizationName(StringUtils.substring(proposalSite.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
            }
            if (organization.getCongressionalDistrict() != null) {
                awardeeDataType.setCongressionalDistrict(StringUtils.substring(organization.getCongressionalDistrict(), 0, CONGRESSIONAL_DISTRICT_MAX_LENGTH));
            }
        }

        AwardeeDataType.Address address = AwardeeDataType.Address.Factory.newInstance();

        if (proposalSite != null) {
            if (proposalSite.getAddressLine1() != null) {
                address.setStreet1(StringUtils.substring(proposalSite.getAddressLine1(), 0, ADDRESS_LINE1_MAX_LENGTH));
            }
            if (proposalSite.getAddressLine2() != null) {
                address.setStreet2(StringUtils.substring(proposalSite.getAddressLine2(), 0, ADDRESS_LINE2_MAX_LENGTH));
            }
            if (proposalSite.getCity() != null) {
                address.setCity(StringUtils.substring(proposalSite.getCity(), 0, CITY_MAX_LENGTH));
            }
            globLibV20Generator.getStateCodeDataTypeV3(proposalSite.getCountryCode(), proposalSite.getState())
                    .ifPresent(address::setState);

            address.setZipPostalCode(proposalSite.getPostalCode());
        }
        awardeeDataType.setAddress(address);
        reportEntity.setReportingEntity(awardeeDataType);
        return reportEntity;
    }

    @Override
    public LobbyingActivitiesDisclosure20Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        aorInfo = departmentalPersonService.getDepartmentalPerson(pdDoc);
        return getLobbyingActivitiesDisclosure();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    public RolodexService getRolodexService() {
        return rolodexService;
    }

    public void setRolodexService(RolodexService rolodexService) {
        this.rolodexService = rolodexService;
    }

    @Override
    public Attachments getMappedAttachments(LobbyingActivitiesDisclosure20Document form, List<AttachmentData> attachments) {
        return new Attachments(Collections.emptyMap(), attachments);
    }

    @Override
    public DocumentFactory<LobbyingActivitiesDisclosure20Document> factory() {
        return LobbyingActivitiesDisclosure20Document.Factory;
    }
}
