/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.coeus.s2sgen.impl.generate;

import gov.grants.apply.system.globalV10.HashValueDocument;
import gov.grants.apply.system.header20V20.Header20Document;
import gov.grants.apply.system.headerV10.GrantSubmissionHeaderDocument;
import gov.grants.apply.system.metaGrantApplication.GrantApplicationDocument;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlObject;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.s2s.S2sOpportunityContract;
import org.kuali.coeus.s2sgen.api.core.InfrastructureConstants;
import org.kuali.coeus.s2sgen.api.core.S2SException;
import org.kuali.coeus.s2sgen.api.hash.GrantApplicationHashService;
import org.kuali.coeus.s2sgen.impl.datetime.S2SDateTimeService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import javax.xml.namespace.QName;
import java.util.Calendar;

@Component("formApplicationService")
public class FormApplicationServiceImpl implements FormApplicationService {

    @Autowired
    @Qualifier("s2SDateTimeService")
    private S2SDateTimeService s2SDateTimeService;

    @Autowired
    @Qualifier("grantApplicationHashService")
    private GrantApplicationHashService grantApplicationHashService;

    @Autowired
    @Qualifier("s2SFormGeneratorRetrievalService")
    private S2SFormGeneratorRetrievalService s2SFormGeneratorRetrievalService;

    @Override
    public String getGrantApplicationDocument(ProposalDevelopmentDocumentContract pdDoc, GrantApplicationDocument.GrantApplication.Forms forms) throws S2SException {
        final GrantApplicationDocument grantApplicationDocument = GrantApplicationDocument.Factory.newInstance();
        final GrantApplicationDocument.GrantApplication grantApplication = GrantApplicationDocument.GrantApplication.Factory.newInstance();
        grantApplication.setForms(forms);

        final S2sOpportunityContract s2sOpportunity = pdDoc.getDevelopmentProposal().getS2sOpportunity();
        if (StringUtils.isNotBlank(s2sOpportunity.getPackageId())) {
            grantApplication.setHeader20(createHeaderVersion2(pdDoc));
        } else {
            grantApplication.setGrantSubmissionHeader(createHeaderVersion1(pdDoc, getGrantApplicationHashService().computeGrantFormsHash(getXmlFromDocument(grantApplication))));
        }

        grantApplicationDocument.setGrantApplication(grantApplication);

        final XmlCursor cursor = grantApplicationDocument.newCursor();

        cursor.toStartDoc();
        if (cursor.toFirstChild()){
            final String defaultNameSpace = cursor.getName().getNamespaceURI();
            final String schemaLocation = defaultNameSpace + " " + s2sOpportunity.getSchemaUrl();
            cursor.setAttributeText(new QName("http://www.w3.org/2001/XMLSchema-instance","schemaLocation"), schemaLocation);
        }

        return getXmlFromDocument(grantApplicationDocument);
    }

    /**
     * @deprecated Remove when header version1 is no longer supported
     */
    @Deprecated
    private GrantSubmissionHeaderDocument.GrantSubmissionHeader createHeaderVersion1(ProposalDevelopmentDocumentContract pdDoc, String hashVal) {
        final GrantSubmissionHeaderDocument.GrantSubmissionHeader grantSubmissionHeader = GrantSubmissionHeaderDocument.GrantSubmissionHeader.Factory.newInstance();
        grantSubmissionHeader.setOpportunityTitle(pdDoc.getDevelopmentProposal().getProgramAnnouncementTitle());
        grantSubmissionHeader.setAgencyName(pdDoc.getDevelopmentProposal().getSponsor().getSponsorName());
        pdDoc.getDevelopmentProposal().getS2sOpportunity().getS2sOpportunityCfdas().stream().findFirst().ifPresent(cfda -> {
            grantSubmissionHeader.setCFDANumber(cfda.getCfdaNumber());

            final String activityTitle = cfda.getCfdaDescription();
            if (StringUtils.isNotEmpty(activityTitle)) {
                grantSubmissionHeader.setActivityTitle(activityTitle);
            }
        });

        final S2sOpportunityContract s2sOpportunity = pdDoc.getDevelopmentProposal().getS2sOpportunity();
        if (s2sOpportunity.getCompetitionId() != null) {
            grantSubmissionHeader.setCompetitionID(s2sOpportunity.getCompetitionId());
        }
        grantSubmissionHeader.setOpportunityID(s2sOpportunity.getOpportunityId());
        grantSubmissionHeader.setSchemaVersion(FormVersion.v1_0.getVersion());
        grantSubmissionHeader.setSubmissionTitle(s2sOpportunity.getProposalNumber());

        // set closing date unless null
        final Calendar closingDate = s2sOpportunity.getClosingDate();
        if (closingDate != null) {
            grantSubmissionHeader.setClosingDate(closingDate);
        }

        // set opening date unless null
        final Calendar openingDate = s2sOpportunity.getOpeningDate();
        if (openingDate != null) {
            grantSubmissionHeader.setOpeningDate(openingDate);
        }

        final HashValueDocument.HashValue hashValue = HashValueDocument.HashValue.Factory.newInstance();
        hashValue.setHashAlgorithm(InfrastructureConstants.HASH_ALGORITHM);
        hashValue.setStringValue(hashVal);
        grantSubmissionHeader.setHashValue(hashValue);

        return grantSubmissionHeader;
    }

    private Header20Document.Header20 createHeaderVersion2(ProposalDevelopmentDocumentContract pdDoc) {
        final Header20Document.Header20 header20 = Header20Document.Header20.Factory.newInstance();

        header20.setApplicationFilingName(pdDoc.getDevelopmentProposal().getTitle());
        header20.setPackageID(pdDoc.getDevelopmentProposal().getS2sOpportunity().getPackageId());
        header20.setSchemaVersion(FormVersion.v2_0.getVersion());

        return header20;
    }

    private String getXmlFromDocument(XmlObject grantApplicationDocument) {
        final String applicationXmlText = grantApplicationDocument.xmlText(getS2SFormGeneratorRetrievalService().getXmlOptionsPrefixes());
        return getS2SDateTimeService().removeTimezoneFactor(applicationXmlText);
    }

    public S2SDateTimeService getS2SDateTimeService() {
        return s2SDateTimeService;
    }

    public void setS2SDateTimeService(S2SDateTimeService s2SDateTimeService) {
        this.s2SDateTimeService = s2SDateTimeService;
    }

    public GrantApplicationHashService getGrantApplicationHashService() {
        return grantApplicationHashService;
    }

    public void setGrantApplicationHashService(GrantApplicationHashService grantApplicationHashService) {
        this.grantApplicationHashService = grantApplicationHashService;
    }

    public S2SFormGeneratorRetrievalService getS2SFormGeneratorRetrievalService() {
        return s2SFormGeneratorRetrievalService;
    }

    public void setS2SFormGeneratorRetrievalService(S2SFormGeneratorRetrievalService s2SFormGeneratorRetrievalService) {
        this.s2SFormGeneratorRetrievalService = s2SFormGeneratorRetrievalService;
    }
}
