/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.edSF424Supplement30V30.EDSF424Supplement30Document;
import gov.grants.apply.forms.edSF424Supplement30V30.EDSF424Supplement30Document.EDSF424Supplement30;
import gov.grants.apply.forms.edSF424Supplement30V30.EDSF424Supplement30ProjectDirectorDataType;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.org.OrganizationContract;
import org.kuali.coeus.common.api.ynq.YnqConstant;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerContract;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerHeaderContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.s2s.S2SConfigurationService;
import org.kuali.coeus.propdev.api.specialreview.ProposalSpecialReviewContract;
import org.kuali.coeus.propdev.api.specialreview.ProposalSpecialReviewExemptionContract;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

/**
 * This class is used to generate XML Document object for grants.gov
 * EDSF424SupplementV3.0 This form is generated using XMLBean API's generated
 * by compiling EDSF424SupplementV3.0 schema.
 */
@FormGenerator("EDSF424SupplementV3_0Generator")
public class EDSF424SupplementV3_0Generator extends
        EDSF424SupplementBaseGenerator<EDSF424Supplement30Document> implements S2SFormGeneratorPdfFillable<EDSF424Supplement30Document> {

    protected static final int IS_NOVICE_APPLICANT_QUESTION = -10147;
    protected static final int NIA_PREFERENCE_POINTS_QUESTION = -10148;
    protected static final int QUALIFIED_OPPORTUNITY_ZONE_QUESTION = -10149;

    @Value("http://apply.grants.gov/forms/ED_SF424_Supplement_3_0-V3.0")
    private String namespace;

    @Value("ED_SF424_Supplement_3_0-V3.0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/ED_SF424_Supplement-V3.0.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/ED_SF424_Supplement_3_0-V3.0.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    @Autowired
    @Qualifier("s2SConfigurationService")
    private S2SConfigurationService s2SConfigurationService;

    protected static final String MISSING_PERCENTAGE_EFFORT_ERROR = "The Project Director's (Principal Investigator) Percentage Effort is required.";

    private EDSF424Supplement30Document getEDSF424Supplement() {
        EDSF424Supplement30Document edsf424SupplementDocument = EDSF424Supplement30Document.Factory.newInstance();
        EDSF424Supplement30 edsf424Supplement = EDSF424Supplement30.Factory.newInstance();

        edsf424Supplement.setFormVersion(FormVersion.v3_0.getVersion());
        ProposalPersonContract pi = getS2SProposalPersonService().getPrincipalInvestigator(pdDoc);
        edsf424Supplement.setProjectDirector(getProjectDirector(pi));
        setQuestionnaireAnswers(edsf424Supplement);
        setHumanSubjectsInformation(edsf424Supplement);
        setNarratives(edsf424Supplement);
        edsf424SupplementDocument.setEDSF424Supplement30(edsf424Supplement);
        return edsf424SupplementDocument;
    }

    private EDSF424Supplement30ProjectDirectorDataType getProjectDirector(ProposalPersonContract proposalPerson) {
        EDSF424Supplement30ProjectDirectorDataType projectDirector = EDSF424Supplement30ProjectDirectorDataType.Factory.newInstance();

        if (proposalPerson != null) {
            projectDirector.setName(globLibV20Generator.getHumanNameDataType(proposalPerson));

            if (StringUtils.isNotEmpty(proposalPerson.getOfficePhone())) {
                projectDirector.setPhone(proposalPerson.getOfficePhone());
            }

            if (StringUtils.isNotEmpty(proposalPerson.getEmailAddress())) {
                projectDirector.setEmail(proposalPerson.getEmailAddress());
            }

            if (StringUtils.isNotEmpty(proposalPerson.getFaxNumber())) {
                projectDirector.setFax(proposalPerson.getFaxNumber());
            }

            projectDirector.setAddress(globLibV20Generator.getAddressDataTypeV3(proposalPerson));

            if (proposalPerson.getPercentageEffort() != null) {
                projectDirector.setProjectDirectorLevelOfEffort(proposalPerson.getPercentageEffort().intValue());
            } else {
                getAuditErrors().add(new AuditError(AuditError.NO_FIELD_ERROR_KEY, MISSING_PERCENTAGE_EFFORT_ERROR, AuditError.GG_LINK));
            }
        }

        return projectDirector;
    }

    private void setHumanSubjectsInformation(EDSF424Supplement30 edsf424Supplement) {
        edsf424Supplement.setIsHumanResearch(YesNoDataType.N_NO);
        boolean areSpecialReviewsPresent = Boolean.FALSE;
        for (ProposalSpecialReviewContract specialReview : pdDoc.getDevelopmentProposal().getPropSpecialReviews()) {
            if (specialReview.getSpecialReviewType() != null && specialReview.getSpecialReviewType().getCode().equals(SPECIAL_REVIEW_CODE)) {
                edsf424Supplement.setIsHumanResearch(YesNoDataType.Y_YES);
                if (specialReview.getApprovalType() != null && specialReview.getApprovalType().getCode().equals(APPROVAL_TYPE_CODE)) {
                    edsf424Supplement.setIsHumanResearchExempt(YesNoDataType.Y_YES);
                    areSpecialReviewsPresent = Boolean.TRUE;

                    if (specialReview.getSpecialReviewExemptions() != null && specialReview.getSpecialReviewExemptions().size() > 0) {
                        for (ProposalSpecialReviewExemptionContract exemption : specialReview.getSpecialReviewExemptions()) {
                            switch (exemption.getExemptionType().getCode()) {
                                case "1":
                                    edsf424Supplement.setExemptionNumber1(YesNoDataType.Y_YES);
                                    break;
                                case "2":
                                    edsf424Supplement.setExemptionNumber2(YesNoDataType.Y_YES);
                                    break;
                                case "3":
                                    edsf424Supplement.setExemptionNumber3(YesNoDataType.Y_YES);
                                    break;
                                case "4":
                                    edsf424Supplement.setExemptionNumber4(YesNoDataType.Y_YES);
                                    break;
                                case "5":
                                    edsf424Supplement.setExemptionNumber5(YesNoDataType.Y_YES);
                                    break;
                                case "6":
                                    edsf424Supplement.setExemptionNumber6(YesNoDataType.Y_YES);
                                    break;
                                case "7":
                                    edsf424Supplement.setExemptionNumber7(YesNoDataType.Y_YES);
                                    break;
                                case "8":
                                    edsf424Supplement.setExemptionNumber8(YesNoDataType.Y_YES);
                                    break;
                            }
                        }
                    }
                }
            }
        }
        if (!areSpecialReviewsPresent) {
            populateAssuranceInformation(edsf424Supplement);
        }
    }

    protected void populateAssuranceInformation(EDSF424Supplement30 edsf424Supplement) {
        final OrganizationContract organization = pdDoc.getDevelopmentProposal().getApplicantOrganization().getOrganization();
        if (organization != null) {
            edsf424Supplement.setIsHumanResearchExempt(YesNoDataType.N_NO);
            if (organization.getHumanSubAssurance() != null) {
                edsf424Supplement.setAssuranceNumber(organization.getHumanSubAssurance());
            }
        }
    }

    private void setQuestionnaireAnswers(EDSF424Supplement30 edsf424Supplement) {
        final List<? extends AnswerHeaderContract> answerHeaders = getPropDevQuestionAnswerService().getQuestionnaireAnswerHeaders(pdDoc.getDevelopmentProposal().getProposalNumber(), getNamespace(), getFormName());

        /*
         * We set this to null to trigger validation error in case the field isn't filled
         * appropriately later on. If left unset, this will cause a cascading issue that would make
         * the user think the issue is on the following fields and not in the Is Novice Applicant one.
         */
        edsf424Supplement.setIsNoviceApplicant(null);

        answerHeaders
                .stream()
                .filter(Objects::nonNull)
                .flatMap(answerHeader -> answerHeader.getAnswers().stream())
                .filter(answer -> answer.getAnswer() != null)
                .forEach(answer -> {
                    switch (answer.getQuestionSeqId()) {
                        case IS_NOVICE_APPLICANT_QUESTION:
                            setIsNoviceApplicant(edsf424Supplement, answer.getAnswer());
                            break;
                        case NIA_PREFERENCE_POINTS_QUESTION:
                            edsf424Supplement.setNIAPreferencePoints(Integer.valueOf(answer.getAnswer()));
                            break;
                        case QUALIFIED_OPPORTUNITY_ZONE_QUESTION:
                            setQualifiedOpportunityZone(edsf424Supplement, answer);
                            break;
                    }
                });
    }

    private void setIsNoviceApplicant(EDSF424Supplement30 edsf424Supplement, String answer) {
        if (YnqConstant.YES.code().equals(answer)) {
            edsf424Supplement.setIsNoviceApplicant(YesNoDataType.Y_YES);
        } else if (YnqConstant.NO.code().equals(answer)) {
            edsf424Supplement.setIsNoviceApplicant(YesNoDataType.N_NO);
        }
    }

    private void setQualifiedOpportunityZone(EDSF424Supplement30 edsf424Supplement, AnswerContract answerContract) {
        final String answer = answerContract.getAnswer();

        switch (answerContract.getAnswerNumber()) {
            case 1:
                edsf424Supplement.setQualifiedOpportunityZone1(answer);
                break;
            case 2:
                edsf424Supplement.setQualifiedOpportunityZone2(answer);
                break;
            case 3:
                edsf424Supplement.setQualifiedOpportunityZone3(answer);
                break;
            case 4:
                edsf424Supplement.setQualifiedOpportunityZone4(answer);
                break;
            case 5:
                edsf424Supplement.setQualifiedOpportunityZone5(answer);
                break;
            case 6:
                edsf424Supplement.setQualifiedOpportunityZone6(answer);
                break;
            case 7:
                edsf424Supplement.setQualifiedOpportunityZone7(answer);
                break;
            case 8:
                edsf424Supplement.setQualifiedOpportunityZone8(answer);
                break;
            case 9:
                edsf424Supplement.setQualifiedOpportunityZone9(answer);
                break;
            case 10:
                edsf424Supplement.setQualifiedOpportunityZone10(answer);
                break;
            case 11:
                edsf424Supplement.setQualifiedOpportunityZone11(answer);
                break;
            case 12:
                edsf424Supplement.setQualifiedOpportunityZone12(answer);
                break;
        }
    }

    protected void setNarratives(EDSF424Supplement30 edsf424Supplement) {
        final Optional<? extends NarrativeContract> edsf424SupplementNarrativeOptional = pdDoc.getDevelopmentProposal().getNarratives().stream()
                .filter(narrative -> narrative.getNarrativeType().getCode() != null && Integer.parseInt(narrative.getNarrativeType().getCode()) == NARRATIVE_TYPE_ED_SF424_SUPPLIMENT)
                .findFirst();

        if (edsf424SupplementNarrativeOptional.isPresent()) {
            final AttachedFileDataType attachment = addAttachedFileType(edsf424SupplementNarrativeOptional.get());

            if (attachment != null) {
                edsf424Supplement.setAttachment(attachment);
            }
        }
    }

    /**
     * This method creates {@link XmlObject} of type
     * {@link EDSF424Supplement30Document} by populating data from the given
     * {@link ProposalDevelopmentDocumentContract}
     *
     * @param proposalDevelopmentDocument for which the {@link XmlObject} needs to be created
     * @return {@link XmlObject} which is generated using the given
     * {@link ProposalDevelopmentDocumentContract}
     */
    @Override
    public EDSF424Supplement30Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        pdDoc = proposalDevelopmentDocument;
        return getEDSF424Supplement();
    }

    public S2SConfigurationService getS2SConfigurationService() {
        return s2SConfigurationService;
    }

    public void setS2SConfigurationService(S2SConfigurationService s2SConfigurationService) {
        this.s2SConfigurationService = s2SConfigurationService;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(EDSF424Supplement30Document form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {
            if (form.getEDSF424Supplement30().getAttachment() != null && a.getContentId().equals(form.getEDSF424Supplement30().getAttachment().getFileLocation().getHref())) {
                return entry("ED_SF424_Supplement_3_0_P2.optionalFile0", a);
            }

            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<EDSF424Supplement30Document> factory() {
        return EDSF424Supplement30Document.Factory;
    }
}
