/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.nsfCoverPage16V16.NSFCoverPage16Document;
import gov.grants.apply.forms.nsfCoverPage16V16.NSFCoverPage16Document.NSFCoverPage16;
import gov.grants.apply.forms.nsfCoverPage16V16.NSFCoverPage16Document.NSFCoverPage16.FundingMechanism;
import gov.grants.apply.forms.nsfCoverPage16V16.NSFCoverPage16Document.NSFCoverPage16.NSFUnitConsideration;
import gov.grants.apply.forms.nsfCoverPage16V16.NSFCoverPage16Document.NSFCoverPage16.OtherInfo;
import gov.grants.apply.forms.nsfCoverPage16V16.NSFCoverPage16Document.NSFCoverPage16.PIInfo;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.attachmentsV10.AttachmentGroupMin1Max100DataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.ynq.YnqConstant;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

@FormGenerator("NSFCoverPageV1_6Generator")
public class NSFCoverPageV1_6Generator extends NSFCoverPageBaseGenerator<NSFCoverPage16Document> implements S2SFormGeneratorPdfFillable<NSFCoverPage16Document> {

    private static final int MENTORING_PLAN = 147;
    private static final int DATA_MANAGEMENT_PLAN = 146;
    private static final int LOBBYING_ACTIVITIES_QUESTION = 11;

    @Value("http://apply.grants.gov/forms/NSF_CoverPage_1_6-V1.6")
    private String namespace;

    @Value("NSF_CoverPage_1_6-V1.6")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/NSF_CoverPage-V1.6.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/NSF_CoverPage_1_6-V1.6.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    private NSFCoverPage16Document getNSFCoverPage16() {
        NSFCoverPage16Document nsfCoverPage16Document = NSFCoverPage16Document.Factory.newInstance();
        NSFCoverPage16 nsfCoverPage16 = NSFCoverPage16.Factory.newInstance();
        nsfCoverPage16.setFormVersion(FormVersion.v1_6.getVersion());
        setFundingOpportunityNumber(nsfCoverPage16);

        if (pdDoc.getDevelopmentProposal().getS2sOpportunity() != null && pdDoc.getDevelopmentProposal().getS2sOpportunity().getClosingDate() != null) {
            nsfCoverPage16.setDueDate(pdDoc.getDevelopmentProposal().getS2sOpportunity().getClosingDate());
        }
        nsfCoverPage16.setNSFUnitConsideration(getNSFUnitConsideration());
        setOtherInfo(nsfCoverPage16);
        AttachmentGroupMin1Max100DataType attachmentGroup = AttachmentGroupMin1Max100DataType.Factory.newInstance();
        attachmentGroup.setAttachedFileArray(getAttachedFileDataTypes());
        if (attachmentGroup.getAttachedFileArray().length > 0) {
            nsfCoverPage16.setSingleCopyDocuments(attachmentGroup);
        }
        final AttachedFileDataType dataManagementPlan = getAttachedNarrativeFile(DATA_MANAGEMENT_PLAN);
        if (dataManagementPlan != null) {
            nsfCoverPage16.setDataManagementPlan(dataManagementPlan);
        }

        final AttachedFileDataType mentoringPlan = getAttachedNarrativeFile(MENTORING_PLAN);
        if (mentoringPlan != null) {
            nsfCoverPage16.setMentoringPlan(mentoringPlan);
        }
        nsfCoverPage16Document.setNSFCoverPage16(nsfCoverPage16);
        return nsfCoverPage16Document;
    }


    private void setFundingOpportunityNumber(NSFCoverPage16 nsfCoverPage16) {
        nsfCoverPage16.setFundingOpportunityNumber(StringUtils.substring(pdDoc.getDevelopmentProposal().getProgramAnnouncementNumber(), 0, PROGRAM_ANNOUNCEMENT_NUMBER_MAX_LENGTH));
    }

    private void setOtherInfo(NSFCoverPage16 nsfCoverPage16) {
        OtherInfo otherInfo = OtherInfo.Factory.newInstance();
        PIInfo pInfo = PIInfo.Factory.newInstance();
        for (AnswerContract questionnaireAnswer : getPropDevQuestionAnswerService().getQuestionnaireAnswers(pdDoc.getDevelopmentProposal().getProposalNumber(), getNamespace(), getFormName())) {
            String answer = questionnaireAnswer.getAnswer();
            int questionId = questionnaireAnswer.getQuestionNumber();

            if (answer != null) {
                switch (questionId) {
                    case QUESTION_CURRENT_PI:
                        pInfo.setIsCurrentPI(answer.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES : YesNoDataType.N_NO);
                        break;
                    case QUESTION_BEGIN_INVESTIGATOR:
                        otherInfo.setIsBeginInvestigator(answer.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES : YesNoDataType.N_NO);
                        break;
                    case QUESTION_ACCOMPLISHMENT_RENEWAL:
                        otherInfo.setIsAccomplishmentRenewal(answer.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES : YesNoDataType.N_NO);
                        break;
                    case FUNDING_MECHANISM_QUESTION:
                        setFundingMechanism(nsfCoverPage16, answer);
                        break;
                    case LOBBYING_ACTIVITIES_QUESTION:
                        otherInfo.setIsDisclosureLobbyingActivities(answer.equals(YnqConstant.YES.code()) ? YesNoDataType.Y_YES : YesNoDataType.N_NO);
                        break;
                    default:
                        break;
                }
            }
        }
        nsfCoverPage16.setPIInfo(pInfo);
        nsfCoverPage16.setOtherInfo(otherInfo);
    }

    private void setFundingMechanism(NSFCoverPage16 nsfCoverPage16, String answer) {
        FundingMechanism.Enum fundingMechanism = FundingMechanism.RAPID;
        if (StringUtils.equalsIgnoreCase(answer, FundingMechanism.CONFERENCE.toString())) {
            fundingMechanism = FundingMechanism.CONFERENCE;
        } else if (StringUtils.equalsIgnoreCase(answer, FundingMechanism.EAGER.toString())) {
            fundingMechanism = FundingMechanism.EAGER;
        } else if (StringUtils.equalsIgnoreCase(answer, FundingMechanism.EQUIPMENT.toString())) {
            fundingMechanism = FundingMechanism.EQUIPMENT;
        } else if (StringUtils.equalsIgnoreCase(answer, FundingMechanism.FACILITY_CENTER.toString())) {
            fundingMechanism = FundingMechanism.FACILITY_CENTER;
        } else if (StringUtils.equalsIgnoreCase(answer, FundingMechanism.FELLOWSHIP.toString())) {
            fundingMechanism = FundingMechanism.FELLOWSHIP;
        } else if (StringUtils.equalsIgnoreCase(answer, FundingMechanism.IDEAS_LAB.toString())) {
            fundingMechanism = FundingMechanism.IDEAS_LAB;
        } else if (StringUtils.equalsIgnoreCase(answer, FundingMechanism.INTERNATIONAL_TRAVEL.toString())) {
            fundingMechanism = FundingMechanism.INTERNATIONAL_TRAVEL;
        } else if (StringUtils.equalsIgnoreCase(answer, FundingMechanism.RESEARCH_OTHER_THAN_RAPID_OR_EAGER.toString())) {
            fundingMechanism = FundingMechanism.RESEARCH_OTHER_THAN_RAPID_OR_EAGER;
        }
        nsfCoverPage16.setFundingMechanism(fundingMechanism);
    }

    private NSFUnitConsideration getNSFUnitConsideration() {
        NSFUnitConsideration nsfConsideration = NSFUnitConsideration.Factory.newInstance();
        nsfConsideration.setDivisionCode(pdDoc.getDevelopmentProposal().getAgencyDivisionCode());
        nsfConsideration.setProgramCode(pdDoc.getDevelopmentProposal().getAgencyProgramCode());
        return nsfConsideration;
    }

    @Override
    public NSFCoverPage16Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getNSFCoverPage16();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(NSFCoverPage16Document form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {
            if (form.getNSFCoverPage16().getSingleCopyDocuments() != null && form.getNSFCoverPage16().getSingleCopyDocuments().getAttachedFileList() != null) {
                final List<AttachedFileDataType> attachedFileList = form.getNSFCoverPage16().getSingleCopyDocuments().getAttachedFileList();
                for (int i = 0; i < attachedFileList.size(); i++) {
                    final AttachedFileDataType attachedFile = attachedFileList.get(i);
                    if (a.getContentId().equals(attachedFile.getFileLocation().getHref())){
                        return entry("NSF_CoverPage_1_6_P1.optionalFile1_" + i, a);
                    }
                }
            }

            final AttachedFileDataType dataManagementPlanAttachedFile = form.getNSFCoverPage16().getDataManagementPlan();
            if (dataManagementPlanAttachedFile != null && a.getContentId().equals(dataManagementPlanAttachedFile.getFileLocation().getHref())) {
                return entry("NSF_CoverPage_1_6_P1.optionalFile2", a);
            }

            final AttachedFileDataType mentoringPlanAttachedFile = form.getNSFCoverPage16().getMentoringPlan();
            if (mentoringPlanAttachedFile != null && a.getContentId().equals(mentoringPlanAttachedFile.getFileLocation().getHref())) {
                return entry("NSF_CoverPage_1_6_P1.optionalFile3", a);
            }

            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<NSFCoverPage16Document> factory() {
        return NSFCoverPage16Document.Factory;
    }
}
