/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.performanceSite20V20.PerformanceSite20Document;
import gov.grants.apply.forms.performanceSite20V20.PerformanceSite20Document.PerformanceSite20;
import gov.grants.apply.forms.performanceSite20V20.SiteLocationDataType;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.org.OrganizationContract;
import org.kuali.coeus.common.api.org.OrganizationYnqContract;
import org.kuali.coeus.common.api.rolodex.RolodexService;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.location.CongressionalDistrictContract;
import org.kuali.coeus.propdev.api.location.ProposalSiteContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.validate.S2SErrorHandlerService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 
 * This class is used to generate XML Document object for grants.gov
 * PerformanceSiteV2.0. This form is generated using XMLBean API's generated by
 * compiling PerformanceSiteV2.0 schema.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@FormGenerator("PerformanceSiteV2_0Generator")
public class PerformanceSiteV2_0Generator extends PerformanceSiteBaseGenerator<PerformanceSite20Document> implements S2SFormGeneratorPdfFillable<PerformanceSite20Document> {

	private static final String QUESTION_ID_FOR_INDIVIDUAL_YNQ = "30";
    private static final int PROPOSAL_SITE_APPLICANT_ORGANIZATION = 1;
    private static final int PERFORMING_ORG_LOCATION_TYPE_CODE = 2;
    private static final int OTHER_ORG_LOCATION_TYPE_CODE = 3;
    private static final int PERFORMANCE_SITE_LOCATION_TYPE_CODE = 4;

    private final Map<Integer, Integer> locationTypeCodePriorityMap = Map.of(PROPOSAL_SITE_APPLICANT_ORGANIZATION, 0, PERFORMING_ORG_LOCATION_TYPE_CODE, 1, PERFORMANCE_SITE_LOCATION_TYPE_CODE, 2, OTHER_ORG_LOCATION_TYPE_CODE, 3);

    @Value("http://apply.grants.gov/forms/PerformanceSite_2_0-V2.0")
    private String namespace;

    @Value("PerformanceSite_2_0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/PerformanceSite-V2.0.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/PerformanceSite_2_0-V2.0.pdf")
    private Resource pdfForm;

    @Value("130")
    private int sortIndex;

    @Autowired
    @Qualifier("rolodexService")
    private RolodexService rolodexService;

    @Autowired
    @Qualifier("s2SErrorHandlerService")
    private S2SErrorHandlerService s2SErrorHandlerService;

	private PerformanceSite20Document getPerformanceSite() {
		PerformanceSite20Document performanceSite14Document = PerformanceSite20Document.Factory
				.newInstance();
		PerformanceSite20 performanceSite14 = PerformanceSite20.Factory
				.newInstance();
		performanceSite14.setFormVersion(FormVersion.v2_0.getVersion());
		setOtherSites(performanceSite14);
		AttachedFileDataType attachedFile = getAttachedFileDataType("40");
		if(attachedFile!=null){
		    performanceSite14.setAttachedFile(attachedFile);
		}
		performanceSite14Document.setPerformanceSite20(performanceSite14);
		return performanceSite14Document;
	}


	private void setSiteLocationDataType(
			SiteLocationDataType siteLocationDataType, OrganizationContract organization) {
		if (organization.getOrganizationYnqs() != null
				&& !organization.getOrganizationYnqs().isEmpty()) {
			for (OrganizationYnqContract organizationYnq : organization
					.getOrganizationYnqs()) {
				if (organizationYnq.getQuestionId().equals(
						QUESTION_ID_FOR_INDIVIDUAL_YNQ)) {
					YesNoDataType.Enum answer = organizationYnq.getAnswer()
							.equals("Y") ? YesNoDataType.Y_YES
							: YesNoDataType.N_NO;
					siteLocationDataType.setIndividual(answer);
				}
			}
		}
	}

    private void setOtherSites(PerformanceSite20  performanceSite) {
	    List<ProposalSiteContract> proposalSites = pdDoc.getDevelopmentProposal().getProposalSites().stream()
                .sorted((o1, o2) -> locationTypeCodePriorityMap.get(o1.getLocationTypeCode()).compareTo(locationTypeCodePriorityMap.get(o2.getLocationTypeCode())))
                .collect(Collectors.toList());

        if (proposalSites != null) {
            OrganizationContract organization = null;
            SiteLocationDataType siteLocationDataType = null;
            for (ProposalSiteContract proposalSite : proposalSites) {
                organization = proposalSite.getOrganization();
                switch(proposalSite.getLocationTypeCode()){
                    case(PERFORMING_ORG_LOCATION_TYPE_CODE):
                        siteLocationDataType = performanceSite.addNewPrimarySite();
                        setSiteLocationDataType(siteLocationDataType, organization);
                        siteLocationDataType.setOrganizationName(StringUtils.substring(proposalSite.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
                        break;
                    case(OTHER_ORG_LOCATION_TYPE_CODE):
                    case(PERFORMANCE_SITE_LOCATION_TYPE_CODE):
                        siteLocationDataType = performanceSite.addNewOtherSite();
                        siteLocationDataType.setOrganizationName(StringUtils.substring(proposalSite.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
                        break;
                }
                if(siteLocationDataType!=null){
                    siteLocationDataType.setAddress(globLibV20Generator.getAddressDataType(proposalSite));
                    if (organization!=null && organization.getDunsNumber() != null) {
                        siteLocationDataType.setDUNSNumber(organization.getDunsNumber());
                    }
                    String congressionalDistrict = getCongressionalDistrict(proposalSite);
                    if (congressionalDistrict == null) {
                        getAuditErrors().add(s2SErrorHandlerService.getError(CONGRESSIONAL_DISTRICT, getFormName()));
                        siteLocationDataType.setCongressionalDistrictProgramProject(null);
                    }else{
                        siteLocationDataType.setCongressionalDistrictProgramProject(congressionalDistrict);
                    }
                }
            }
        }
    }

    private String getCongressionalDistrict(ProposalSiteContract proposalSite) {
        String congDistrictProject = null;
        for (CongressionalDistrictContract congDistrict : proposalSite.getCongressionalDistricts()) {
            congDistrictProject = congDistrict.getCongressionalDistrict();
            if (congDistrictProject != null && congDistrictProject.length() > CONGRESSIONAL_DISTRICT_MAX_LENGTH) {
                congDistrictProject = congDistrictProject.substring(0, CONGRESSIONAL_DISTRICT_MAX_LENGTH);
            }
        }
        return congDistrictProject;
    }

	/**
	 * This method creates {@link XmlObject} of type
	 * {@link PerformanceSite20Document} by populating data from the given
	 * {@link ProposalDevelopmentDocumentContract}
	 * 
	 * @param proposalDevelopmentDocument
	 *            for which the {@link XmlObject} needs to be created
	 * @return {@link XmlObject} which is generated using the given
	 *         {@link ProposalDevelopmentDocumentContract}
	 */
	@Override
    public PerformanceSite20Document getFormObject(
			ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
		this.pdDoc = proposalDevelopmentDocument;
		return getPerformanceSite();
	}

    public RolodexService getRolodexService() {
        return rolodexService;
    }

    public void setRolodexService(RolodexService rolodexService) {
        this.rolodexService = rolodexService;
    }

    public S2SErrorHandlerService getS2SErrorHandlerService() {
        return s2SErrorHandlerService;
    }

    public void setS2SErrorHandlerService(S2SErrorHandlerService s2SErrorHandlerService) {
        this.s2SErrorHandlerService = s2SErrorHandlerService;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(PerformanceSite20Document form, List<AttachmentData> attachments) {
        return new Attachments(Collections.emptyMap(), attachments);
    }

    @Override
    public DocumentFactory<PerformanceSite20Document> factory() {
        return PerformanceSite20Document.Factory;
    }
}
