/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;


import gov.grants.apply.forms.sf424BV11.AssuranceType;
import gov.grants.apply.forms.sf424BV11.AssurancesDocument;
import gov.grants.apply.forms.sf424BV11.AuthorizedRepresentativeDocument.AuthorizedRepresentative;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.location.ProposalSiteContract;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.person.DepartmentalPersonDto;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.Collections;
import java.util.List;


/**
 * This Class is used to generate XML object for grants.gov SF424BV1.1. This form is generated using XMLBean classes and is based on
 * SF424BV1.1 schema.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@FormGenerator("SF424BV1_1Generator")
public class SF424BV1_1Generator extends SF424BaseGenerator<AssurancesDocument> implements S2SFormGeneratorPdfFillable<AssurancesDocument> {

    private static final String CORE_SCHEMA_VERSION_1_1 = "1.1";

    @Value("http://apply.grants.gov/forms/SF424B-V1.1")
    private String namespace;

    @Value("SF424B-V1.1")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/SF424B-V1.1.fo.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/SF424B-V1.1.pdf")
    private Resource pdfForm;

    @Value(DEFAULT_SORT_INDEX)
    private int sortIndex;

    /**
     * 
     * This method returns AssurancesDocument object based on proposal development document which contains all the information for a
     * particular proposal
     * 
     * @return assuranceDocument {@link XmlObject} of type AssurancesDocument.
     */
    private AssurancesDocument getAssurance() {

        AssurancesDocument assuranceDocument = AssurancesDocument.Factory.newInstance();
        assuranceDocument.setAssurances(getAssuranceType());
        return assuranceDocument;
    }

    /**
     * 
     * This method gets AssuranceType details based on proposal development document.AssuranceType type includes details like
     * ProgramType,AuthorizedRepresentative,ApplicantOrganizationName.
     * 
     * @return AssuranceType object containing information about ApplicantOrganizationName,ProgramType,AuthorizedRepresentative
     */
    private AssuranceType getAssuranceType() {

        AssuranceType assuranceType = AssuranceType.Factory.newInstance();
        assuranceType.setFormVersionIdentifier(FormVersion.v1_1.getVersion());
        assuranceType.setProgramType(PROGRAM_TYPE);
        assuranceType.setCoreSchemaVersion(CORE_SCHEMA_VERSION_1_1);
        DepartmentalPersonDto aorInfo = departmentalPersonService.getDepartmentalPerson(pdDoc);
        assuranceType.setAuthorizedRepresentative(getAuthorizedRepresentative(aorInfo));
        assuranceType.setSubmittedDate(aorInfo.getSubmittedDate());
        ProposalSiteContract proposalSiteContract = pdDoc.getDevelopmentProposal().getApplicantOrganization();
        assuranceType.setApplicantOrganizationName(StringUtils.substring(proposalSiteContract.getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));

        return assuranceType;
    }

    /**
     * 
     * This method gets AuthorizedRepresentative details RepresentativeTitle based on ProposalDevelopmentDocumentContract.
     * 
     * @return AuthorizedRepresentative authorized representative title.
     */
    private AuthorizedRepresentative getAuthorizedRepresentative(DepartmentalPersonDto aorInfo) {

        AuthorizedRepresentative authorizedRepresentative = AuthorizedRepresentative.Factory.newInstance();
        if (aorInfo.getPrimaryTitle() != null) {
            authorizedRepresentative.setRepresentativeTitle(StringUtils.substring(aorInfo.getPrimaryTitle(), 0, PRIMARY_TITLE_MAX_LENGTH));
        }
        authorizedRepresentative.setRepresentativeName(aorInfo.getFullName());
        return authorizedRepresentative;
    }

    /**
     * This method creates {@link XmlObject} of type {@link AssurancesDocument} by populating data from the given
     * {@link ProposalDevelopmentDocumentContract}
     * 
     * @param proposalDevelopmentDocument for which the {@link XmlObject} needs to be created
     * @return {@link XmlObject} which is generated using the given {@link ProposalDevelopmentDocumentContract}
     */
    @Override
    public AssurancesDocument getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getAssurance();
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(AssurancesDocument form, List<AttachmentData> attachments) {
        return new Attachments(Collections.emptyMap(), attachments);
    }

    @Override
    public DocumentFactory<AssurancesDocument> factory() {
        return AssurancesDocument.Factory;
    }
}
