/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.util;

import java.util.AbstractMap;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collector;
import java.util.stream.Collectors;

/**
 * Yet another collection utility.  Contains methods that are not found in the apache util classes or the
 * jdk util classes.
 */
public final class CollectionUtils {

    /** private ctor. */
    private CollectionUtils() {
        throw new UnsupportedOperationException("do not call");
    }

    /**
     * Creates an entry from a key and value.
     * @param key the key. Can be null.
     * @param value the value.  Can be null.
     * @param <K> the key type
     * @param <V> the value type
     * @return a Map.Entry
     */
    public static <K, V> Map.Entry<K, V> entry(K key, V value) {
        return new AbstractMap.SimpleEntry<K, V>(key, value);
    }


    /**
     * Convenience method to a return a Collector that converts an Map.Entry to a Map which type is provided as a mapSupplier.
     * @param <K> the key type
     * @param <U> the value type
     * @param <M> the map type
     * @param mapSupplier supplies a new instance of a Map.  cannot be null.
     * @return A Collector from Map.Entry to Map as defined by the mapSupplier
     */
    public static <K, U, M extends Map<K, U>> Collector<Map.Entry<K, U>, ?, M> entriesToMap(Supplier<M> mapSupplier) {
        if (mapSupplier == null) {
            throw new IllegalArgumentException("mapSupplier cannot be null");
        }

        return Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (u,v) -> { throw new IllegalStateException(String.format("Duplicate key %s", u)); }, mapSupplier);
    }

    /**
     * Convenience method to a return a Collector that converts an Map.Entry to a Map.
     * @param <K> the key type
     * @param <U> the value type
     * @return A Collector from Map.Entry to Map
     */
    public static <K, U> Collector<Map.Entry<K, U>, ?, Map<K, U>> entriesToMap() {
        return Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue);
    }

    /**
     * Convenience method to a return a Collector that converts an Map.Entry to a Map.  If a duplicate key is detected, the value is replaced rather
     * throwing an exception as is the default behavior.
     *
     * @param <K> the key type
     * @param <U> the value type
     * @return A Collector from Map.Entry to Map
     */
    public static <K, U> Collector<Map.Entry<K, U>, ?, Map<K, U>> entriesToMapWithReplacing() {
        return Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (v1, v2) -> v2, HashMap::new);
    }
    
    /**
     * Convenience method to a return a Collector that converts an Map.Entry to a Map.
     * @param <K> the key type
     * @param <U> the value type
     * @return A Collector from Map.Entry to Map
     */
    public static <K, U> Collector<Map.Entry<K, U>, ?, Map<K, U>> nullSafeEntriesToMap() {
        return nullSafeToMap(Map.Entry::getKey, Map.Entry::getValue);
    }
    
    /**
     * Null-safe replacement for Collectors.toMap. Allows null keys and values. 
     * If a duplicate key is found, throws IllegalStateException
     * @param keyMapper function to get the key from the items
     * @param valueMapper function to get the value from the items
     * @return A Collector from List to Map
     */
    public static <T, K, U> Collector<T, ?, Map<K, U>> nullSafeToMap(Function<? super T, ? extends K> keyMapper,
            Function<? super T, ? extends U> valueMapper) {
    	return Collectors.collectingAndThen(
	        Collectors.toList(),
	        list -> {
	            Map<K, U> result = new HashMap<>();
	            for (T item : list) {
	                K key = keyMapper.apply(item);
	                if (result.putIfAbsent(key, valueMapper.apply(item)) != null) {
	                    throw new IllegalStateException(String.format("Duplicate key %s", key));
	                }
	            }
	            return result;
	        }
    	);
    }
}
