/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.validate;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlError;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlOptions;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.impl.util.SafeXmlUtils;
import org.kuali.coeus.s2sgen.impl.util.XmlBeansUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;
import org.w3c.dom.Node;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import javax.xml.XMLConstants;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 *
 * This class validates a XML document passed as XMLObject
 *
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@Component("s2SValidatorService")
public class S2SValidatorServiceImpl implements S2SValidatorService {

    private static final Logger LOG = LogManager.getLogger(S2SValidatorServiceImpl.class);

    private static final Cache<String, Schema> SCHEMA_CACHE = CacheBuilder.newBuilder()
            .maximumSize(1000)
            .expireAfterAccess(30, TimeUnit.MINUTES)
            .build();

    @Autowired
    @Qualifier("s2SErrorHandlerService")
    private S2SErrorHandlerService s2SErrorHandlerService;

    //we should use this catalog file as a added enhancement to avoid remote fetching of resources
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/schema/catalog.xml")
    private Resource catalog;

    @Value("#{{'http', 'https'}}")
    private Set<String> approvedSchemaProtocols;

    @Value("#{{'https://apply07.grants.gov/', 'https://trainingapply.grants.gov/', 'http://at07apply.grants.gov/', ''}}")
    private Set<String> approvedSchemaPrefixes;

    /**
     * This method receives an XMLObject and validates it against its schema and returns the validation result. It also receives a
     * list in which upon validation failure, populates it with XPaths of the error nodes.
     *
     * @param formObject XML document as {@link XmlObject}
     * @return validation result true if valid false otherwise.
     */
    @Override
    public ValidationResult validateForm(XmlObject formObject, String formName) {

        final List<String> formErrors = new ArrayList<>();
        final boolean valid = validateXml(formObject, formErrors);

        final ValidationResult result = new ValidationResult();
        result.setValid(valid);
        result.setErrors(formErrors
                .stream()
                .filter(error -> notTopFormNodeWhenChildrenExist(error, formErrors))
                .map(validationError -> s2SErrorHandlerService.getError(GRANTS_GOV_PREFIX + validationError, formName))
                .collect(Collectors.toList()));

        return result;
    }

    // Filter out /Form if any children like /Form/field have errors.
    private boolean notTopFormNodeWhenChildrenExist(String error, List<String> formErrors) {
        return formErrors.stream().noneMatch(other -> other.startsWith(error + "/"));
    }

    @Override
    public ValidationResult validateApplication(String applicationXml, Resource oppSchemaResource) {
        final ValidationResult result = new ValidationResult();
        result.setValid(true);
        result.setErrors(Collections.emptyList());

        if (oppSchemaResource == null) {
            result.setValid(false);
            addAuditError(result, new AuditError(AuditError.NO_FIELD_ERROR_KEY, "Opportunity Schema URL is blank or does not exist", AuditError.GG_LINK));
        }

        if (StringUtils.isBlank(applicationXml)) {
            result.setValid(false);
            addAuditError(result, new AuditError(AuditError.NO_FIELD_ERROR_KEY, "Application XML is blank", AuditError.GG_LINK));
        }

        if (!result.isValid()) {
            return result;
        }

        Schema schema = null;
        try {
            schema = SCHEMA_CACHE.get(oppSchemaResource.getURL().toString(), () -> {
                final Schema fetchedSchema = fetchSchema(oppSchemaResource);
                if (fetchedSchema == null) {
                    throw new IllegalStateException("Schema is null");
                }
                return fetchedSchema;
            });
        } catch (RuntimeException|IOException|ExecutionException e) {
            LOG.warn("Unable to fetch a valid schema " + oppSchemaResource.getFilename(), e);
            addAuditError(result, new AuditError(AuditError.NO_FIELD_ERROR_KEY, "Unable to retrieve opportunity schema from grants.gov.  Application validation did not occur.", AuditError.GG_LINK, AuditError.Level.WARNING));
        }

        if (schema != null) {
            try(ByteArrayInputStream stream = new ByteArrayInputStream(applicationXml.getBytes(StandardCharsets.UTF_8)))  {
                final Source xmlFile = new StreamSource(stream);
                final Validator validator = SafeXmlUtils.safeValidator(schema);

                validator.setErrorHandler(new ErrorHandler() {
                    @Override
                    public void warning(SAXParseException e) {
                        addAuditError(result, new AuditError(AuditError.NO_FIELD_ERROR_KEY, e.getMessage(), AuditError.GG_LINK, AuditError.Level.WARNING));
                    }

                    @Override
                    public void error(SAXParseException e) {
                        addError(result, e);
                    }

                    @Override
                    public void fatalError(SAXParseException e) {
                        addError(result, e);
                    }
                });
                validator.validate(xmlFile);
            } catch (SAXException|IOException e) {
                addError(result, e);
            }
        }

        return result;
    }

    private void addError(ValidationResult result, Exception e) {
        result.setValid(false);
        addAuditError(result, new AuditError(AuditError.NO_FIELD_ERROR_KEY, e.getMessage(), AuditError.GG_LINK));
    }

    private void addAuditError(ValidationResult result, AuditError error) {
        result.setErrors(Stream.concat(Stream.of(error), result.getErrors().stream()).collect(Collectors.toList()));
    }

    private Schema fetchSchema(Resource oppSchemaResource) throws IOException, SAXException {
        final String schemaUrl = oppSchemaResource.getURL().toString();

        if (approvedSchemaPrefixes.stream().noneMatch(schemaUrl::startsWith)) {
            throw new IllegalArgumentException("Unsafe schema location " + schemaUrl);
        }

        Schema schema = null;
        if (StringUtils.isNotBlank(schemaUrl)) {
            final SchemaFactory schemaFactory = SafeXmlUtils.safeSchemaFactory();
            schemaFactory.setProperty(XMLConstants.ACCESS_EXTERNAL_SCHEMA, String.join(",", approvedSchemaProtocols));
            //adding random UUID as an attempt to understand why sometimes we get a connection reset.  Is it some kind of blacklist system?
            schema = schemaFactory.newSchema(new URL(schemaUrl + "?" + UUID.randomUUID()));
        }
        return schema;
    }

    /**
     *
     * This method receives an XMLObject and validates it against its schema and returns the validation result. It also receives a
     * list in which upon validation failure, populates it with XPaths of the error nodes
     *
     * @param formObject XML document as {@link XmlObject}
     * @param errors {@link List} of XPaths of the error nodes.
     * @return validation result true if valid false otherwise.
     */
    protected boolean validateXml(XmlObject formObject, List<String> errors) {
        final XmlOptions validationOptions = XmlBeansUtils.getXmlOptionsPrefixes();
        final Collection<XmlError> validationErrors = new ArrayList<>();
        validationOptions.setErrorListener(validationErrors);

        final boolean isValid = StringUtils.isEmpty(formObject.schemaType().getFullJavaName()) || formObject.validate(validationOptions);

        if (!isValid) {
            if (LOG.isInfoEnabled()) {
                LOG.info("Errors occurred during validation of XML from form generator");
            }

            for (XmlError error : validationErrors) {
                    final Node node = error.getCursorLocation().getDomNode();
                    final String xpath = getXPath(node);
                    errors.add(xpath);
                if (LOG.isInfoEnabled()) {
                    LOG.info("Error: " + error + (xpath != null ? " " + xpath : ""));
                }
            }
        }
        return isValid;
    }

    /**
     *
     * This method receives a node, fetches its name, and recurses up to its parent node, until it reaches the document node, thus
     * creating the XPath of the node passed and returns it as String
     *
     * @param node for which Document node has to found.
     * @return String which represents XPath of the node
     */
    protected String getXPath(Node node) {
        if (node==null || node.getNodeType() == Node.DOCUMENT_NODE) {
            return "";
        }
        else {
            return getXPath(node.getParentNode()) + "/" + node.getNodeName();
        }
    }

    public S2SErrorHandlerService getS2SErrorHandlerService() {
        return s2SErrorHandlerService;
    }

    public void setS2SErrorHandlerService(S2SErrorHandlerService s2SErrorHandlerService) {
        this.s2SErrorHandlerService = s2SErrorHandlerService;
    }

    public Resource getCatalog() {
        return catalog;
    }

    public void setCatalog(Resource catalog) {
        this.catalog = catalog;
    }

    public Set<String> getApprovedSchemaProtocols() {
        return approvedSchemaProtocols;
    }

    public void setApprovedSchemaProtocols(Set<String> approvedSchemaProtocols) {
        this.approvedSchemaProtocols = approvedSchemaProtocols;
    }

    public Set<String> getApprovedSchemaPrefixes() {
        return approvedSchemaPrefixes;
    }

    public void setApprovedSchemaPrefixes(Set<String> approvedSchemaPrefixes) {
        this.approvedSchemaPrefixes = approvedSchemaPrefixes;
    }
}
