/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2016 The Kuali Foundation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.bc.document.service.impl;

import org.apache.ojb.broker.PersistenceBrokerException;
import org.kuali.kfs.module.bc.BCConstants;
import org.kuali.kfs.module.bc.BCKeyConstants;
import org.kuali.kfs.module.bc.businessobject.BudgetConstructionAccountSummary;
import org.kuali.kfs.module.bc.businessobject.BudgetConstructionOrgAccountSummaryReport;
import org.kuali.kfs.module.bc.businessobject.BudgetConstructionOrgAccountSummaryReportTotal;
import org.kuali.kfs.module.bc.document.dataaccess.BudgetConstructionAccountSummaryReportDao;
import org.kuali.kfs.module.bc.document.service.BudgetConstructionAccountSummaryReportService;
import org.kuali.kfs.module.bc.document.service.BudgetConstructionReportsServiceHelper;
import org.kuali.kfs.module.bc.report.BudgetConstructionReportHelper;
import org.kuali.kfs.module.bc.util.BudgetConstructionUtils;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.rice.core.api.config.property.ConfigurationService;
import org.kuali.rice.core.api.util.type.KualiInteger;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * Service implementation of BudgetConstructionAccountSummaryReportService.
 */
@Transactional
public class BudgetConstructionAccountSummaryReportServiceImpl implements BudgetConstructionAccountSummaryReportService {

    protected BudgetConstructionAccountSummaryReportDao budgetConstructionAccountSummaryReportDao;
    protected ConfigurationService kualiConfigurationService;
    protected BudgetConstructionReportsServiceHelper budgetConstructionReportsServiceHelper;
    protected boolean trExist = false;

    /**
     * @see org.kuali.kfs.module.bc.document.service.BudgetReportsControlListService#updateRepotsAccountSummaryTable(java.lang.String)
     */
    @Override
    public void updateReportsAccountSummaryTable(String principalName) {
        String expenditureINList = BudgetConstructionUtils.getExpenditureINList();
        String revenueINList = BudgetConstructionUtils.getRevenueINList();
        budgetConstructionAccountSummaryReportDao.cleanReportsAccountSummaryTable(principalName);
        budgetConstructionAccountSummaryReportDao.updateReportsAccountSummaryTable(principalName, revenueINList, expenditureINList);
    }

    /**
     * @see org.kuali.kfs.module.bc.document.service.BudgetReportsControlListService#updateRepotsAccountSummaryTableWithConsolidation(java.lang.String)
     */
    @Override
    public void updateReportsAccountSummaryTableWithConsolidation(String principalName) {
        String expenditureINList = BudgetConstructionUtils.getExpenditureINList();
        String revenueINList = BudgetConstructionUtils.getRevenueINList();
        budgetConstructionAccountSummaryReportDao.cleanReportsAccountSummaryTable(principalName);
        budgetConstructionAccountSummaryReportDao.updateReportsAccountSummaryTableWithConsolidation(principalName, revenueINList, expenditureINList);
    }

    /**
     * @see org.kuali.kfs.module.bc.document.service.BudgetConstructionAccountSummaryReportService#updateReportsAccountSummaryTable(java.lang.String,
     * boolean)
     */
    @Override
    public void updateReportsAccountSummaryTable(String principalName, boolean consolidated) {
        if (consolidated) {
            updateReportsAccountSummaryTableWithConsolidation(principalName);
        } else {
            updateReportsAccountSummaryTable(principalName);
        }
    }

    /**
     * sets budgetConstructionAccountSummaryReportDao
     *
     * @param budgetConstructionAccountSummaryReportDao
     */
    public void setBudgetConstructionAccountSummaryReportDao(BudgetConstructionAccountSummaryReportDao budgetConstructionAccountSummaryReportDao) {
        this.budgetConstructionAccountSummaryReportDao = budgetConstructionAccountSummaryReportDao;
    }

    /**
     * @see org.kuali.kfs.module.bc.document.service.BudgetConstructionAccountSummaryReportService#buildReports(java.lang.Integer,
     * java.util.Collection)
     */
    @Override
    public Collection<BudgetConstructionOrgAccountSummaryReport> buildReports(Integer universityFiscalYear, String principalName, boolean consolidated) {
        Collection<BudgetConstructionOrgAccountSummaryReport> reportSet = new ArrayList();
        Collection<BudgetConstructionAccountSummary> accountSummaryList = budgetConstructionReportsServiceHelper.getDataForBuildingReports(BudgetConstructionAccountSummary.class, principalName, buildOrderByList());

        // Making a list with same organizationChartOfAccountsCode, organizationCode, chartOfAccountsCode, subFundGroupCode
        List totalList = BudgetConstructionReportHelper.deleteDuplicated((List) accountSummaryList, fieldsForTotal());

        // Calculate Total Section
        List<BudgetConstructionOrgAccountSummaryReportTotal> orgAccountSummaryReportTotalList = calculateTotal((List) accountSummaryList, totalList);

        // builds report
        for (BudgetConstructionAccountSummary accountSummaryEntry : accountSummaryList) {
            BudgetConstructionOrgAccountSummaryReport orgAccountSummaryReportEntry = new BudgetConstructionOrgAccountSummaryReport();
            buildReportsHeader(universityFiscalYear, orgAccountSummaryReportEntry, accountSummaryEntry, consolidated);
            buildReportsBody(orgAccountSummaryReportEntry, accountSummaryEntry);
            buildReportsTotal(orgAccountSummaryReportEntry, accountSummaryEntry, orgAccountSummaryReportTotalList);
            reportSet.add(orgAccountSummaryReportEntry);
        }

        return reportSet;
    }

    /**
     * builds report Header
     *
     * @param BudgetConstructionAccountSummary bcas
     */
    protected void buildReportsHeader(Integer universityFiscalYear, BudgetConstructionOrgAccountSummaryReport orgAccountSummaryReportEntry, BudgetConstructionAccountSummary accountSummary, boolean consolidated) {
        String orgChartDesc = accountSummary.getOrganizationChartOfAccounts().getFinChartOfAccountDescription();
        String chartDesc = accountSummary.getChartOfAccounts().getFinChartOfAccountDescription();
        String orgName = accountSummary.getOrganization().getOrganizationName();
        String reportChartDesc = accountSummary.getChartOfAccounts().getReportsToChartOfAccounts().getFinChartOfAccountDescription();
        String subFundGroupName = accountSummary.getFundGroup().getName();
        String subFundGroupDes = accountSummary.getSubFundGroup().getSubFundGroupDescription();
        Integer prevFiscalyear = universityFiscalYear - 1;
        orgAccountSummaryReportEntry.setFiscalYear(prevFiscalyear.toString() + "-" + universityFiscalYear.toString().substring(2, 4));
        orgAccountSummaryReportEntry.setOrgChartOfAccountsCode(accountSummary.getOrganizationChartOfAccountsCode());

        if (orgChartDesc == null) {
            orgAccountSummaryReportEntry.setOrgChartOfAccountDescription(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.ERROR_REPORT_GETTING_CHART_DESCRIPTION));
        } else {
            orgAccountSummaryReportEntry.setOrgChartOfAccountDescription(orgChartDesc);
        }

        orgAccountSummaryReportEntry.setOrganizationCode(accountSummary.getOrganizationCode());
        if (orgName == null) {
            orgAccountSummaryReportEntry.setOrganizationName(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.ERROR_REPORT_GETTING_ORGANIZATION_NAME));
        } else {
            orgAccountSummaryReportEntry.setOrganizationName(orgName);
        }

        orgAccountSummaryReportEntry.setChartOfAccountsCode(accountSummary.getChartOfAccountsCode());
        if (chartDesc == null) {
            orgAccountSummaryReportEntry.setChartOfAccountDescription(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.ERROR_REPORT_GETTING_CHART_DESCRIPTION));
        } else {
            orgAccountSummaryReportEntry.setChartOfAccountDescription(chartDesc);
        }

        orgAccountSummaryReportEntry.setFundGroupCode(accountSummary.getFundGroupCode());
        if (subFundGroupName == null) {
            orgAccountSummaryReportEntry.setFundGroupName(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.ERROR_REPORT_GETTING_FUNDGROUP_NAME));
        } else {
            orgAccountSummaryReportEntry.setFundGroupName(subFundGroupName);
        }

        orgAccountSummaryReportEntry.setSubFundGroupCode(accountSummary.getSubFundGroupCode());
        if (subFundGroupName == null) {
            orgAccountSummaryReportEntry.setSubFundGroupDescription(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.ERROR_REPORT_GETTING_SUBFUNDGROUP_DESCRIPTION));
        } else {
            orgAccountSummaryReportEntry.setSubFundGroupDescription(subFundGroupDes);
        }

        Integer prevPrevFiscalyear = prevFiscalyear - 1;
        orgAccountSummaryReportEntry.setBaseFy(prevPrevFiscalyear.toString() + "-" + prevFiscalyear.toString().substring(2, 4));
        orgAccountSummaryReportEntry.setReqFy(prevFiscalyear.toString() + "-" + universityFiscalYear.toString().substring(2, 4));
        orgAccountSummaryReportEntry.setHeader1(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.MSG_REPORT_HEADER_ACCOUNT_SUB));
        orgAccountSummaryReportEntry.setHeader2(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.MSG_REPORT_HEADER_ACCOUNT_SUB_NAME));
        orgAccountSummaryReportEntry.setHeader3(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.MSG_REPORT_HEADER_BASE_AMOUNT));
        orgAccountSummaryReportEntry.setHeader4(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.MSG_REPORT_HEADER_REQ_AMOUNT));
        orgAccountSummaryReportEntry.setHeader5(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.MSG_REPORT_HEADER_CHANGE));
        orgAccountSummaryReportEntry.setHeader6(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.MSG_REPORT_HEADER_CHANGE));
        if (consolidated) {
            orgAccountSummaryReportEntry.setConsHdr(BCConstants.Report.CONSOLIIDATED);
        } else {
            orgAccountSummaryReportEntry.setConsHdr(BCConstants.Report.BLANK);
        }
    }

    /**
     * builds report body
     *
     * @param BudgetConstructionAccountSummary bcas
     */
    protected void buildReportsBody(BudgetConstructionOrgAccountSummaryReport orgAccountSummaryReportEntry, BudgetConstructionAccountSummary accountSummary) {
        orgAccountSummaryReportEntry.setAccountNumber(accountSummary.getAccountNumber());
        orgAccountSummaryReportEntry.setSubAccountNumber(accountSummary.getSubAccountNumber());

        if (accountSummary.getSubAccountNumber().equals(KFSConstants.getDashSubAccountNumber())) {
            if (accountSummary.getAccount().getAccountName() == null) {
                orgAccountSummaryReportEntry.setAccountNameAndSubAccountName(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.ERROR_REPORT_GETTING_ACCOUNT_DESCRIPTION));
            } else {
                orgAccountSummaryReportEntry.setAccountNameAndSubAccountName(accountSummary.getAccount().getAccountName());
            }
        } else {
            try {
                if (accountSummary.getSubAccount().getSubAccountName() == null) {
                    orgAccountSummaryReportEntry.setAccountNameAndSubAccountName(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.ERROR_REPORT_GETTING_SUB_ACCOUNT_DESCRIPTION));
                } else {
                    orgAccountSummaryReportEntry.setAccountNameAndSubAccountName(accountSummary.getSubAccount().getSubAccountName());
                }
            } catch (PersistenceBrokerException e) {
                orgAccountSummaryReportEntry.setAccountNameAndSubAccountName(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.ERROR_REPORT_GETTING_SUB_ACCOUNT_DESCRIPTION));
            }
        }

        // build income expense description
        if (accountSummary.getIncomeExpenseCode().equals(BCConstants.Report.INCOME_EXP_TYPE_A)) {
            orgAccountSummaryReportEntry.setIncExpDesc(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.MSG_REPORT_INCOME_EXP_DESC_REVENUE));
        } else if (accountSummary.getIncomeExpenseCode().equals(BCConstants.Report.INCOME_EXP_TYPE_E)) {
            orgAccountSummaryReportEntry.setIncExpDesc(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.MSG_REPORT_INCOME_EXP_DESC_EXP_GROSS));
        } else if (accountSummary.getIncomeExpenseCode().equals(BCConstants.Report.INCOME_EXP_TYPE_T)) {
            trExist = true;
            orgAccountSummaryReportEntry.setIncExpDesc(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.MSG_REPORT_INCOME_EXP_DESC_TRNFR_IN));
        } else {
            if (trExist) {
                trExist = false;
                orgAccountSummaryReportEntry.setIncExpDesc(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.MSG_REPORT_INCOME_EXP_DESC_EXP_NET_TRNFR));
            } else {
                orgAccountSummaryReportEntry.setIncExpDesc(kualiConfigurationService.getPropertyValueAsString(BCKeyConstants.MSG_REPORT_INCOME_EXP_DESC_EXPENDITURE));
            }
        }

        KualiInteger beginingBalanceLineAmt = KualiInteger.ZERO;
        KualiInteger accountLineAnnualBalAmt = KualiInteger.ZERO;
        if (accountSummary.getFinancialBeginningBalanceLineAmount() != null) {
            beginingBalanceLineAmt = accountSummary.getFinancialBeginningBalanceLineAmount();
        }

        if (accountSummary.getFinancialBeginningBalanceLineAmount() != null) {
            accountLineAnnualBalAmt = accountSummary.getAccountLineAnnualBalanceAmount();
        }

        orgAccountSummaryReportEntry.setBaseAmount(beginingBalanceLineAmt);
        orgAccountSummaryReportEntry.setReqAmount(accountLineAnnualBalAmt);
        orgAccountSummaryReportEntry.setAmountChange(accountLineAnnualBalAmt.subtract(beginingBalanceLineAmt));

        orgAccountSummaryReportEntry.setPercentChange(BudgetConstructionReportHelper.calculatePercent(orgAccountSummaryReportEntry.getAmountChange(), beginingBalanceLineAmt));

    }

    /**
     * builds report total
     *
     * @param BudgetConstructionAccountSummary bcas
     * @param List                             reportTotalList
     */
    protected void buildReportsTotal(BudgetConstructionOrgAccountSummaryReport orgAccountSummaryReportEntry, BudgetConstructionAccountSummary accountSummary, List reportTotalList) {
        Iterator totalListIter = reportTotalList.iterator();
        while (totalListIter.hasNext()) {
            BudgetConstructionOrgAccountSummaryReportTotal bcasTotalEntry = (BudgetConstructionOrgAccountSummaryReportTotal) totalListIter.next();
            if (BudgetConstructionReportHelper.isSameEntry(accountSummary, bcasTotalEntry.getBcas(), fieldsForTotal())) {
                BigDecimal percentChange = BigDecimal.ZERO;
                orgAccountSummaryReportEntry.setTotalRevenueBaseAmount(bcasTotalEntry.getTotalRevenueBaseAmount());
                orgAccountSummaryReportEntry.setTotalGrossBaseAmount(bcasTotalEntry.getTotalGrossBaseAmount());
                orgAccountSummaryReportEntry.setTotalTransferInBaseAmount(bcasTotalEntry.getTotalTransferInBaseAmount());
                orgAccountSummaryReportEntry.setTotalNetTransferBaseAmount(bcasTotalEntry.getTotalNetTransferBaseAmount());

                orgAccountSummaryReportEntry.setTotalRevenueReqAmount(bcasTotalEntry.getTotalRevenueReqAmount());
                orgAccountSummaryReportEntry.setTotalGrossReqAmount(bcasTotalEntry.getTotalGrossReqAmount());
                orgAccountSummaryReportEntry.setTotalTransferInReqAmount(bcasTotalEntry.getTotalTransferInReqAmount());
                orgAccountSummaryReportEntry.setTotalNetTransferReqAmount(bcasTotalEntry.getTotalNetTransferReqAmount());

                orgAccountSummaryReportEntry.setTotalRevenueAmountChange(orgAccountSummaryReportEntry.getTotalRevenueReqAmount().subtract(orgAccountSummaryReportEntry.getTotalRevenueBaseAmount()));
                percentChange = BudgetConstructionReportHelper.calculatePercent(orgAccountSummaryReportEntry.getTotalRevenueAmountChange(), orgAccountSummaryReportEntry.getTotalRevenueBaseAmount());
                orgAccountSummaryReportEntry.setTotalRevenuePercentChange(percentChange);

                orgAccountSummaryReportEntry.setTotalGrossAmountChange(orgAccountSummaryReportEntry.getTotalGrossReqAmount().subtract(orgAccountSummaryReportEntry.getTotalGrossBaseAmount()));
                percentChange = BudgetConstructionReportHelper.calculatePercent(orgAccountSummaryReportEntry.getTotalGrossAmountChange(), orgAccountSummaryReportEntry.getTotalGrossBaseAmount());
                orgAccountSummaryReportEntry.setTotalGrossPercentChange(percentChange);

                orgAccountSummaryReportEntry.setTotalTransferAmountChange(orgAccountSummaryReportEntry.getTotalTransferInReqAmount().subtract(orgAccountSummaryReportEntry.getTotalTransferInBaseAmount()));
                percentChange = BudgetConstructionReportHelper.calculatePercent(orgAccountSummaryReportEntry.getTotalTransferAmountChange(), orgAccountSummaryReportEntry.getTotalTransferInBaseAmount());
                orgAccountSummaryReportEntry.setTotalTransferInPercentChange(percentChange);

                orgAccountSummaryReportEntry.setTotalNetTransferAmountChange(orgAccountSummaryReportEntry.getTotalNetTransferReqAmount().subtract(orgAccountSummaryReportEntry.getTotalNetTransferBaseAmount()));
                percentChange = BudgetConstructionReportHelper.calculatePercent(orgAccountSummaryReportEntry.getTotalNetTransferAmountChange(), orgAccountSummaryReportEntry.getTotalNetTransferBaseAmount());
                orgAccountSummaryReportEntry.setTotalNetTransferPercentChange(percentChange);

                orgAccountSummaryReportEntry.setRevExpDifferenceBaseAmount(orgAccountSummaryReportEntry.getTotalRevenueBaseAmount().subtract(orgAccountSummaryReportEntry.getTotalNetTransferBaseAmount()));
                orgAccountSummaryReportEntry.setRevExpDifferenceReqAmount(orgAccountSummaryReportEntry.getTotalRevenueReqAmount().subtract(orgAccountSummaryReportEntry.getTotalNetTransferReqAmount()));
                orgAccountSummaryReportEntry.setRevExpDifferenceAmountChange(orgAccountSummaryReportEntry.getRevExpDifferenceReqAmount().subtract(orgAccountSummaryReportEntry.getRevExpDifferenceBaseAmount()));
                percentChange = BudgetConstructionReportHelper.calculatePercent(orgAccountSummaryReportEntry.getRevExpDifferenceAmountChange(), orgAccountSummaryReportEntry.getRevExpDifferenceBaseAmount());
                orgAccountSummaryReportEntry.setRevExpDifferencePercentChange(percentChange);
            }
        }
    }

    /**
     * Calculates total part of report
     *
     * @param List bcasList
     * @param List simpleList
     */
    protected List calculateTotal(List bcasList, List simpleList) {

        KualiInteger totalRevenueBaseAmount = KualiInteger.ZERO;
        KualiInteger totalGrossBaseAmount = KualiInteger.ZERO;
        KualiInteger totalTransferInBaseAmount = KualiInteger.ZERO;
        KualiInteger totalNetTransferBaseAmount = KualiInteger.ZERO;
        KualiInteger totalRevenueReqAmount = KualiInteger.ZERO;
        KualiInteger totalGrossReqAmount = KualiInteger.ZERO;
        KualiInteger totalTransferInReqAmount = KualiInteger.ZERO;
        KualiInteger totalNetTransferReqAmount = KualiInteger.ZERO;

        List returnList = new ArrayList();
        Iterator simpleListIterator = simpleList.iterator();
        boolean prev = false;
        while (simpleListIterator.hasNext()) {
            BudgetConstructionAccountSummary simpleBcasEntry = (BudgetConstructionAccountSummary) simpleListIterator.next();
            Iterator bcasListIterator = bcasList.iterator();
            while (bcasListIterator.hasNext()) {
                BudgetConstructionAccountSummary bcasListEntry = (BudgetConstructionAccountSummary) bcasListIterator.next();
                if (BudgetConstructionReportHelper.isSameEntry(simpleBcasEntry, bcasListEntry, fieldsForTotal())) {
                    if (bcasListEntry.getIncomeExpenseCode().equals(BCConstants.Report.INCOME_EXP_TYPE_A)) {
                        prev = false;
                        totalRevenueBaseAmount = totalRevenueBaseAmount.add(bcasListEntry.getFinancialBeginningBalanceLineAmount());
                        totalRevenueReqAmount = totalRevenueReqAmount.add(bcasListEntry.getAccountLineAnnualBalanceAmount());
                    } else if (bcasListEntry.getIncomeExpenseCode().equals(BCConstants.Report.INCOME_EXP_TYPE_E)) {
                        prev = false;
                        totalGrossBaseAmount = totalGrossBaseAmount.add(bcasListEntry.getFinancialBeginningBalanceLineAmount());
                        totalGrossReqAmount = totalGrossReqAmount.add(bcasListEntry.getAccountLineAnnualBalanceAmount());

                    } else if (bcasListEntry.getIncomeExpenseCode().equals(BCConstants.Report.INCOME_EXP_TYPE_T)) {
                        prev = true;
                        totalTransferInBaseAmount = totalTransferInBaseAmount.add(bcasListEntry.getFinancialBeginningBalanceLineAmount());
                        totalTransferInReqAmount = totalTransferInReqAmount.add(bcasListEntry.getAccountLineAnnualBalanceAmount());
                    } else if (bcasListEntry.getIncomeExpenseCode().equals(BCConstants.Report.INCOME_EXP_TYPE_X)) {
                        totalNetTransferBaseAmount = totalNetTransferBaseAmount.add(bcasListEntry.getFinancialBeginningBalanceLineAmount());
                        totalNetTransferReqAmount = totalNetTransferReqAmount.add(bcasListEntry.getAccountLineAnnualBalanceAmount());
                        if (!prev) {
                            prev = false;
                            totalGrossBaseAmount = totalGrossBaseAmount.add(bcasListEntry.getFinancialBeginningBalanceLineAmount());
                            totalGrossReqAmount = totalGrossReqAmount.add(bcasListEntry.getAccountLineAnnualBalanceAmount());
                        }
                    }
                }
            }
            BudgetConstructionOrgAccountSummaryReportTotal bcoasrTotal = new BudgetConstructionOrgAccountSummaryReportTotal();
            bcoasrTotal.setBcas(simpleBcasEntry);
            bcoasrTotal.setTotalGrossBaseAmount(totalGrossBaseAmount);
            bcoasrTotal.setTotalGrossReqAmount(totalGrossReqAmount);
            bcoasrTotal.setTotalNetTransferBaseAmount(totalNetTransferBaseAmount);
            bcoasrTotal.setTotalNetTransferReqAmount(totalNetTransferReqAmount);
            bcoasrTotal.setTotalRevenueBaseAmount(totalRevenueBaseAmount);
            bcoasrTotal.setTotalRevenueReqAmount(totalRevenueReqAmount);
            bcoasrTotal.setTotalTransferInBaseAmount(totalTransferInBaseAmount);
            bcoasrTotal.setTotalTransferInReqAmount(totalTransferInReqAmount);
            returnList.add(bcoasrTotal);
            totalGrossBaseAmount = KualiInteger.ZERO;
            totalGrossReqAmount = KualiInteger.ZERO;
            totalNetTransferBaseAmount = KualiInteger.ZERO;
            totalNetTransferReqAmount = KualiInteger.ZERO;
            totalRevenueBaseAmount = KualiInteger.ZERO;
            totalRevenueReqAmount = KualiInteger.ZERO;
            totalTransferInBaseAmount = KualiInteger.ZERO;
            totalTransferInReqAmount = KualiInteger.ZERO;
        }

        return returnList;
    }

    protected List<String> fieldsForTotal() {

        List<String> fieldList = new ArrayList();
        fieldList.add(KFSPropertyConstants.ORGANIZATION_CHART_OF_ACCOUNTS_CODE);
        fieldList.add(KFSPropertyConstants.ORGANIZATION_CODE);
        fieldList.add(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        fieldList.add(KFSPropertyConstants.SUB_FUND_GROUP_CODE);

        return fieldList;
    }

    /**
     * builds orderByList for sort order.
     *
     * @return returnList
     */
    public List<String> buildOrderByList() {
        List<String> returnList = new ArrayList();
        returnList.add(KFSPropertyConstants.ORGANIZATION_CHART_OF_ACCOUNTS_CODE);
        returnList.add(KFSPropertyConstants.ORGANIZATION_CODE);
        returnList.add(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        returnList.add(KFSPropertyConstants.SUB_FUND_SORT_CODE);
        returnList.add(KFSPropertyConstants.FUND_GROUP_CODE);
        returnList.add(KFSPropertyConstants.SUB_FUND_GROUP_CODE);
        returnList.add(KFSPropertyConstants.ACCOUNT_NUMBER);
        returnList.add(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
        returnList.add(KFSPropertyConstants.INCOME_EXPENSE_CODE);

        return returnList;
    }

    public void setConfigurationService(ConfigurationService kualiConfigurationService) {
        this.kualiConfigurationService = kualiConfigurationService;
    }

    public void setBudgetConstructionReportsServiceHelper(BudgetConstructionReportsServiceHelper budgetConstructionReportsServiceHelper) {
        this.budgetConstructionReportsServiceHelper = budgetConstructionReportsServiceHelper;
    }
}
