/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2016 The Kuali Foundation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.gl.batch.service;

import org.kuali.kfs.gl.batch.BalanceForwardRuleHelper;

import java.util.List;
import java.util.Map;


/**
 * An interface declaring the methods that steps running year end services would need to use to close out activity of an ending
 * fiscal year (though, note: OrganizationReversion, the other year end job, is not defined here).
 */
public interface YearEndService {


    /**
     * The method responsible for creating origin entries that will forward qualifying encumbrances into the next fiscal year.
     *
     * @param originEntryGroup          the origin entry group where generated origin entries should be saved
     * @param jobParameters             the parameters necessary to run this job: the fiscal year to close and the university date the job was
     *                                  run
     * @param forwardEncumbrancesCounts the statistical counts generated by this job
     */
    public void forwardEncumbrances(String encumbranceForwardFileName, Map jobParameters, Map<String, Integer> counts);

    /**
     * Implementations of this method are responsible for generating origin entries to forward the balances of qualifying balances
     * from the previous fiscal year
     *
     * @param balanceForwardsUnclosedPriorYearAccountGroup the origin entry group to save balance forwarding origin entries with
     *                                                     open accounts
     * @param balanceForwardsClosedPriorYearAccountGroup   the origin entry group to save balance forwarding origin entries with
     *                                                     closed accounts
     * @param balanceForwardRuleHelper                     the BalanceForwardRuleHelper which holds the important state - the job parameters and
     *                                                     statistics - for the job to run
     */
    public void forwardBalances(String balanceForwardsUnclosedFileName, String balanceForwardsclosedFileName, BalanceForwardRuleHelper balanceForwardRuleHelper);

    /**
     * This method is considered responsible for generating all of the origin entries that will close out nominal activity for a
     * given fiscal year.
     *
     * @param nominalClosingOriginEntryGroup the origin entry group that nominal activity closing origin entries
     * @param nominalClosingJobParameters    the parameters needed by the job to run correctly: the current university date and the
     *                                       fiscal year to close
     * @param nominalClosingCounts           counts the Map of statistical counts generated by the process
     */
    public void closeNominalActivity(String nominalClosingFileName, Map nominalClosingJobParameters);

    /**
     * Logs all of the missing prior year accounts that balances and encumbrances processed by year end jobs would attempt to call
     * on
     *
     * @param balanceFiscalYear the fiscal year to find balances encumbrances for
     */
    public void logAllMissingPriorYearAccounts(Integer fiscalYear);

    /**
     * Logs all of the missing prior year accounts based on chart that balances and encumbrances processed by year end jobs would attempt to call
     * on
     *
     * @param fiscalYear the fiscal year to find prior year accounts for
     * @param charts     list of charts to find prior year accounts for
     */
    public void logAllMissingPriorYearAccounts(Integer fiscalYear, List<String> charts);

    /**
     * Logs all of the missing sub fund groups that balances and encumbrances processed by the year end job would attempt to call on
     *
     * @param balanceFiscalYear the fiscal year to find balances and encumbrances for
     */
    public void logAllMissingSubFundGroups(Integer fiscalYear);

    /**
     * Logs all of the missing sub fund groups based on chart that balances and encumbrances processed by the year end job would attempt to call on
     *
     * @param fiscalYear the fiscal year to find sub fund groups for
     * @param charts     list of charts to find sub fund groups for
     */
    public void logAllMissingSubFundGroups(Integer fiscalYear, List<String> charts);
}
